#pragma once

#ifndef tstorageclientstat_H
#define tstorageclientstat_H

#include "util/generic/stroka.h"
#include "util/system/mutex.h"
#include "util/generic/hash.h"
#include <mail/so/spamstop/tools/so-common/kfunc.h>
#include <vector>

//***********************************************************************************************************************
//                                              TStorageClientStat
//***********************************************************************************************************************

class TStorageClientStat {
public:
    enum TStorageActionType { SAT_UPDATE,
                              SAT_FINDONE,
                              SAT_ERASE,
                              SAT_SIZE,
                              SAT_MULTYACTION,
                              SAT_FIND,
                              SAT_CLEAR,
                              SAT_FINDANDMODIFY,
                              SAT_RESERV1,
                              SAT_END };

    struct TChangeLabelStruct {
        TStorageActionType m_acttype;
        TString m_weblabel;
        TString m_loglabel;

        TChangeLabelStruct() {
            Clear();
        }

        TChangeLabelStruct(TStorageActionType acttype, const TString& weblabel, const TString& loglabel) {
            m_acttype = acttype;
            m_weblabel = weblabel;
            m_loglabel = loglabel;
        }

        void Clear() {
            m_acttype = SAT_UPDATE;
            m_weblabel = "";
            m_loglabel = "";
        }
    };

    typedef std::vector<TChangeLabelStruct> TChangeLabelStructList;

private:
    struct TWebLabels {
        TString m_web_labels_array[SAT_END + 1];

        TWebLabels() {
            Init();
        }

        void Init() {
            m_web_labels_array[SAT_UPDATE] = "update_count (UP)";
            m_web_labels_array[SAT_FINDONE] = "findone_count (FO)";
            m_web_labels_array[SAT_ERASE] = "erase_count (E)";
            m_web_labels_array[SAT_SIZE] = "size_count (SZ)";
            m_web_labels_array[SAT_MULTYACTION] = "multiaction_count (M)";
            m_web_labels_array[SAT_FIND] = "find_count (F)";
            m_web_labels_array[SAT_CLEAR] = "clear_count (CLR)";
            m_web_labels_array[SAT_FINDANDMODIFY] = "findandmodify_count (FM)";
            m_web_labels_array[SAT_RESERV1] = "reserv1_count (R1)";
        }

        void ChangeLabel(TChangeLabelStructList& change_labels_list) {
            TChangeLabelStructList::iterator it;

            it = change_labels_list.begin();
            while (it != change_labels_list.end()) {
                m_web_labels_array[(*it).m_acttype] = (*it).m_weblabel;

                ++it;
            }
        }

        TString GetParamNameWeb(TStorageActionType acttype) {
            return m_web_labels_array[acttype];
        }
    };

    struct TLogLabels {
        TString m_log_labels_array[SAT_END + 1];

        TLogLabels() {
            Init();
        }

        void Init() {
            m_log_labels_array[SAT_UPDATE] = "UP";
            m_log_labels_array[SAT_FINDONE] = "FO";
            m_log_labels_array[SAT_ERASE] = "E";
            m_log_labels_array[SAT_SIZE] = "SZ";
            m_log_labels_array[SAT_MULTYACTION] = "M";
            m_log_labels_array[SAT_FIND] = "F";
            m_log_labels_array[SAT_CLEAR] = "CLR";
            m_log_labels_array[SAT_FINDANDMODIFY] = "FM";
            m_log_labels_array[SAT_RESERV1] = "R1";
        }

        void ChangeLabel(TChangeLabelStructList& change_labels_list) {
            TChangeLabelStructList::iterator it;

            it = change_labels_list.begin();
            while (it != change_labels_list.end()) {
                m_log_labels_array[(*it).m_acttype] = (*it).m_loglabel;

                ++it;
            }
        }

        TString GetParamNameLog(TStorageActionType acttype) {
            return m_log_labels_array[acttype];
        }
    };

    struct TOperCountStatItemPart {
        ui64 good;      //good request count
        ui64 bad;       //bad request count
        ui64 m_0_10;    //0 - 10 ms
        ui64 m_10_20;   //10 - 20 ms
        ui64 m_20_50;   //20 - 50 ms
        ui64 m_50_100;  //50 - 100 ms
        ui64 m_100_150; //100 - 150 ms
        ui64 m_150_190; //150 - 190 ms
        ui64 m_190_500; //190 - 500 ms
        ui64 m_more500; //more 500 ms

        bool Empty() {
            if ((good == 0) && (bad == 0))
                return true;
            else
                return false;
        }

        ui64 TimeSumm() const {
            return m_0_10 + m_10_20 + m_20_50 + m_50_100 + m_100_150 + m_150_190 + m_190_500 + m_more500;
        }

        float CalcPercentItem(float value, float summ) const {
            float res = 0;

            if (summ > 0)
                res = value / summ * 100;

            return res;
        }

        TString PrintWithPercent(bool print_percent, ui64 value) const {
            TString res = "";
            float percent = 0;

            if (print_percent) {
                percent = CalcPercentItem(value, TimeSumm());
                res = UI64ToStroka(value) + " (" + FloatToStr(percent) + "%)";

            } else {
                res = UI64ToStroka(value);
            }

            return res;
        }

        TString m_0_10_s(bool print_percent) const {
            return PrintWithPercent(print_percent, m_0_10);
        }

        TString m_10_20_s(bool print_percent) const {
            return PrintWithPercent(print_percent, m_10_20);
        }

        TString m_20_50_s(bool print_percent) const {
            return PrintWithPercent(print_percent, m_20_50);
        }

        TString m_50_100_s(bool print_percent) const {
            return PrintWithPercent(print_percent, m_50_100);
        }

        TString m_100_150_s(bool print_percent) const {
            return PrintWithPercent(print_percent, m_100_150);
        }

        TString m_150_190_s(bool print_percent) const {
            return PrintWithPercent(print_percent, m_150_190);
        }

        TString m_190_500_s(bool print_percent) const {
            return PrintWithPercent(print_percent, m_190_500);
        }

        TString m_more500_s(bool print_percent) const {
            return PrintWithPercent(print_percent, m_more500);
        }

        TOperCountStatItemPart() {
            Clear();
        }

        void Clear() {
            good = 0;
            bad = 0;
            m_0_10 = 0;
            m_10_20 = 0;
            m_20_50 = 0;
            m_50_100 = 0;
            m_100_150 = 0;
            m_150_190 = 0;
            m_190_500 = 0;
            m_more500 = 0;
        }

        ui64 SummGoodBadCount() {
            return IncMax64(good, bad);
        }

        ui64 SummQuantileCount() {
            ui64 res = 0;

            res = IncMax64(res, m_0_10);
            res = IncMax64(res, m_10_20);
            res = IncMax64(res, m_20_50);
            res = IncMax64(res, m_50_100);
            res = IncMax64(res, m_100_150);
            res = IncMax64(res, m_150_190);
            res = IncMax64(res, m_190_500);
            res = IncMax64(res, m_more500);

            return res;
        }

        void Increment(ui32 delay, int count) {
            if (delay < 10)
                m_0_10 = IncMax64(m_0_10, count);
            else if (delay < 20)
                m_10_20 = IncMax64(m_10_20, count);
            else if (delay < 50)
                m_20_50 = IncMax64(m_20_50, count);
            else if (delay < 100)
                m_50_100 = IncMax64(m_50_100, count);
            else if (delay < 150)
                m_100_150 = IncMax64(m_100_150, count);
            else if (delay < 190)
                m_150_190 = IncMax64(m_150_190, count);
            else if (delay < 500)
                m_190_500 = IncMax64(m_190_500, count);
            else
                m_more500 = IncMax64(m_more500, count);
        }

        TOperCountStatItemPart operator+=(const TOperCountStatItemPart& value) {
            good = IncMax64(good, value.good);
            bad = IncMax64(bad, value.bad);
            m_0_10 = IncMax64(m_0_10, value.m_0_10);
            m_10_20 = IncMax64(m_10_20, value.m_10_20);
            m_20_50 = IncMax64(m_20_50, value.m_20_50);
            m_50_100 = IncMax64(m_50_100, value.m_50_100);
            m_100_150 = IncMax64(m_100_150, value.m_100_150);
            m_150_190 = IncMax64(m_150_190, value.m_150_190);
            m_190_500 = IncMax64(m_190_500, value.m_190_500);
            m_more500 = IncMax64(m_more500, value.m_more500);

            return *this;
        }

        TString PrintToLog() {
            char tbuff[256];

            memset(tbuff, 0, sizeof(tbuff));
            snprintf(tbuff, sizeof(tbuff) - 1, "%llu-%llu,%llu-%llu-%llu-%llu-%llu-%llu-%llu-%llu", good, bad, m_0_10, m_10_20, m_20_50, m_50_100, m_100_150, m_150_190, m_190_500, m_more500);

            return TString(tbuff);
        }
    };

    struct TOperCountStatItem {
        TLogLabels m_loglabels;
        TOperCountStatItemPart update_count;
        TOperCountStatItemPart findone_count;
        TOperCountStatItemPart erase_count;
        TOperCountStatItemPart size_count;
        TOperCountStatItemPart find_count;
        TOperCountStatItemPart multiaction_count;
        TOperCountStatItemPart clear_count;
        TOperCountStatItemPart findandmodify_count;
        TOperCountStatItemPart reserv1_count;

        TOperCountStatItem() {
            Clear();
        }

        void Clear() {
            update_count.Clear();
            findone_count.Clear();
            erase_count.Clear();
            size_count.Clear();
            find_count.Clear();
            multiaction_count.Clear();
            clear_count.Clear();
            findandmodify_count.Clear();
            reserv1_count.Clear();
        }

        TOperCountStatItemPart GetAll() {
            TOperCountStatItemPart res = TOperCountStatItemPart();

            res += update_count;
            res += findone_count;
            res += erase_count;
            res += size_count;
            res += find_count;
            res += multiaction_count;
            res += clear_count;
            res += findandmodify_count;
            res += reserv1_count;

            return res;
        }

        TOperCountStatItem operator+=(const TOperCountStatItem& value) {
            update_count += value.update_count;
            findone_count += value.findone_count;
            erase_count += value.erase_count;
            size_count += value.size_count;
            find_count += value.find_count;
            multiaction_count += value.multiaction_count;
            clear_count += value.clear_count;
            findandmodify_count += value.findandmodify_count;
            reserv1_count += value.reserv1_count;

            return *this;
        }

        ui64 GoodCount() {
            ui64 res = 0;

            res += update_count.good;
            res += findone_count.good;
            res += erase_count.good;
            res += size_count.good;
            res += find_count.good;
            res += multiaction_count.good;
            res += clear_count.good;
            res += findandmodify_count.good;
            res += reserv1_count.good;

            return res;
        }

        ui64 BadCount() {
            ui64 res = 0;

            res += update_count.bad;
            res += findone_count.bad;
            res += erase_count.bad;
            res += size_count.bad;
            res += find_count.bad;
            res += multiaction_count.bad;
            res += clear_count.bad;
            res += findandmodify_count.bad;
            res += reserv1_count.bad;

            return res;
        }

        TString PrintToLog() {
            TString res = "";

            if (!update_count.Empty())
                res += "[" + m_loglabels.GetParamNameLog(SAT_UPDATE) + ": " + update_count.PrintToLog() + "]";
            if (!findone_count.Empty())
                res += "[" + m_loglabels.GetParamNameLog(SAT_FINDONE) + ": " + findone_count.PrintToLog() + "]";
            if (!erase_count.Empty())
                res += "[" + m_loglabels.GetParamNameLog(SAT_ERASE) + ": " + erase_count.PrintToLog() + "]";
            if (!size_count.Empty())
                res += "[" + m_loglabels.GetParamNameLog(SAT_SIZE) + ": " + size_count.PrintToLog() + "]";
            if (!find_count.Empty())
                res += "[" + m_loglabels.GetParamNameLog(SAT_FIND) + ": " + find_count.PrintToLog() + "]";
            if (!multiaction_count.Empty())
                res += "[" + m_loglabels.GetParamNameLog(SAT_MULTYACTION) + ": " + multiaction_count.PrintToLog() + "]";
            if (!clear_count.Empty())
                res += "[" + m_loglabels.GetParamNameLog(SAT_CLEAR) + ": " + clear_count.PrintToLog() + "]";
            if (!findandmodify_count.Empty())
                res += "[" + m_loglabels.GetParamNameLog(SAT_FINDANDMODIFY) + ": " + findandmodify_count.PrintToLog() + "]";
            if (!reserv1_count.Empty())
                res += "[" + m_loglabels.GetParamNameLog(SAT_RESERV1) + ": " + reserv1_count.PrintToLog() + "]";

            return res;
        }

        void ChangeLabelLog(TChangeLabelStructList& change_labels_list) {
            m_loglabels.ChangeLabel(change_labels_list);
        }
    };

    struct TOperCountStat {
        TOperCountStatItem today;
        TOperCountStatItem yesterday;

        TOperCountStat() {
            Clear();
        }

        void Clear() {
            today.Clear();
            yesterday.Clear();
        }

        void AddCount(TStorageActionType acttype, int count, bool good, ui32 tick) {
            switch (acttype) {
                case SAT_UPDATE:
                    if (good)
                        today.update_count.good = IncMax64(today.update_count.good, count);
                    else
                        today.update_count.bad = IncMax64(today.update_count.bad, count);
                    today.update_count.Increment(tick, count);
                    break;
                case SAT_FINDONE:
                    if (good)
                        today.findone_count.good = IncMax64(today.findone_count.good, count);
                    else
                        today.findone_count.bad = IncMax64(today.findone_count.bad, count);
                    today.findone_count.Increment(tick, count);
                    break;
                case SAT_ERASE:
                    if (good)
                        today.erase_count.good = IncMax64(today.erase_count.good, count);
                    else
                        today.erase_count.bad = IncMax64(today.erase_count.bad, count);
                    today.erase_count.Increment(tick, count);
                    break;
                case SAT_SIZE:
                    if (good)
                        today.size_count.good = IncMax64(today.size_count.good, count);
                    else
                        today.size_count.bad = IncMax64(today.size_count.bad, count);
                    today.size_count.Increment(tick, count);
                    break;
                case SAT_FIND:
                    if (good)
                        today.find_count.good = IncMax64(today.find_count.good, count);
                    else
                        today.find_count.bad = IncMax64(today.find_count.bad, count);
                    today.find_count.Increment(tick, count);
                    break;
                case SAT_MULTYACTION:
                    if (good)
                        today.multiaction_count.good = IncMax64(today.multiaction_count.good, count);
                    else
                        today.multiaction_count.bad = IncMax64(today.multiaction_count.bad, count);
                    today.multiaction_count.Increment(tick, count);
                    break;
                case SAT_CLEAR:
                    if (good)
                        today.clear_count.good = IncMax64(today.clear_count.good, count);
                    else
                        today.clear_count.bad = IncMax64(today.clear_count.bad, count);
                    today.clear_count.Increment(tick, count);
                    break;
                case SAT_FINDANDMODIFY:
                    if (good)
                        today.findandmodify_count.good = IncMax64(today.findandmodify_count.good, count);
                    else
                        today.findandmodify_count.bad = IncMax64(today.findandmodify_count.bad, count);
                    today.findandmodify_count.Increment(tick, count);
                    break;
                case SAT_RESERV1:
                    if (good)
                        today.reserv1_count.good = IncMax64(today.reserv1_count.good, count);
                    else
                        today.reserv1_count.bad = IncMax64(today.reserv1_count.bad, count);
                    today.reserv1_count.Increment(tick, count);
                    break;
            };
        }

        TOperCountStat operator+=(const TOperCountStat& value) {
            today += value.today;
            yesterday += value.yesterday;

            return *this;
        }

        void Midnight() {
            yesterday = today;
            today.Clear();
        }

        TString PrintToLog() {
            TString res = "";

            res = today.PrintToLog();

            return res;
        }

        void ChangeLabelLog(TChangeLabelStructList& change_labels_list) {
            today.ChangeLabelLog(change_labels_list);
        }
    };

public:
    TWebLabels m_weblabels;
    TOperCountStat data;

public:
    TStorageClientStat();
    ~TStorageClientStat();

    void AddCount(TStorageClientStat::TStorageActionType acttype, int count, bool good, ui32 tick);
    void Midnight();
    TOperCountStat GetStat() {
        return data;
    }
    void Clear() {
        data.Clear();
    }

    TStorageClientStat operator+=(const TStorageClientStat& value) {
        data += value.data;

        return *this;
    }

    void ChangeLabelWeb(TChangeLabelStructList& change_labels_list) {
        m_weblabels.ChangeLabel(change_labels_list);
    }
    TString PrintDataWEB(bool print_percent, bool disable_errors, bool disable_summary);
    TString PrintDataWEBShort(bool print_percent, bool disable_errors, bool disable_summary);

    TString PrintDataLog();
    void ChangeLabelLog(TChangeLabelStructList& change_labels_list);
};

//***********************************************************************************************************************
//                                              TStorageClientStatMain
//***********************************************************************************************************************

struct TStorageStat {
    TString m_name;
    TStorageClientStat m_stat;

    TStorageStat() {
        m_name = "";
        m_stat.Clear();
    }

    TStorageStat(const TString& name) {
        m_name = name;
        m_stat.Clear();
    }

    TStorageStat(const TStorageStat& data) {
        m_name = data.m_name;
        m_stat = data.m_stat;
    }

    TStorageStat(const TString& name, TStorageClientStat::TStorageActionType acttype, int count, bool good, ui32 tick) {
        m_name = name;
        m_stat.Clear();
        m_stat.AddCount(acttype, count, good, tick);
    }

    void ChangeLabelWeb(TStorageClientStat::TChangeLabelStructList& change_labels_list) {
        m_stat.ChangeLabelWeb(change_labels_list);
    }

    void ChangeLabelLog(TStorageClientStat::TChangeLabelStructList& change_labels_list) {
        m_stat.ChangeLabelLog(change_labels_list);
    }

    void Midnight() {
        m_stat.Midnight();
    }

    bool operator<(const TStorageStat& value) const {
        return (m_name < value.m_name);
    }

    bool operator==(const TStorageStat& value) const {
        return (m_name == value.m_name);
    }

    TString PrintDataWEB(bool print_percent, bool disable_errors, bool disable_summary) {
        TString res = "";

        res += "<i>" + m_name + " storage statistik.</i></b>&nbsp;";
        res += m_stat.PrintDataWEB(print_percent, disable_errors, disable_summary);

        return res;
    }

    TString PrintDataWEBShort(bool print_percent, bool disable_errors, bool disable_summary) {
        TString res = "";

        res += "<i>" + m_name + " storage statistik.</i></b>&nbsp;";
        res += m_stat.PrintDataWEBShort(print_percent, disable_errors, disable_summary);

        return res;
    }

    TString PrintDataLog() {
        return "<" + m_name + ": " + m_stat.PrintDataLog() + " >";
    }

    void Clear() {
        m_stat.Clear();
    }
};

class TStorageStatVector: public std::vector<TStorageStat> {
private:
public:
    TStorageStatVector operator+=(const TStorageStatVector& value) {
        TStorageStatVector::iterator it;
        TStorageStatVector::const_iterator it_val;

        it = begin();
        while (it != end()) {
            it_val = find(value.begin(), value.end(), TStorageStat((*it).m_name));
            if (it_val != value.end())
                (*it).m_stat += (*it_val).m_stat;

            ++it;
        }

        it_val = value.begin();
        while (it_val != value.end()) {
            it = find(begin(), end(), TStorageStat((*it_val).m_name));
            if (it == end())
                push_back(TStorageStat((*it_val)));

            ++it_val;
        }

        return *this;
    }

    void Clear() {
        TStorageStatVector::iterator it;

        for (it = begin(); it != end(); ++it)
            (*it).Clear();
    }

    TString PrintDataWEB(bool print_percent, bool disable_errors, bool disable_summary) {
        TStorageStatVector::iterator it;
        TString res = "";

        std::sort(begin(), end());
        it = begin();
        while (it != end()) {
            res += (*it).PrintDataWEB(print_percent, disable_errors, disable_summary) + "<br>";

            ++it;
        }

        return res;
    }

    TString PrintDataWEBInOrder(const std::vector<TString>& orderlist, bool print_percent, bool disable_errors, bool disable_summary) {
        TStorageStatVector::iterator it;
        TString res = "";
        std::vector<TString>::const_iterator lit;
        bool exists = false;
        TString st1 = "", st2 = "";

        std::sort(begin(), end());

        lit = orderlist.begin();
        while (lit != orderlist.end()) {
            if (!(*lit).empty()) {
                it = begin();
                while (it != end()) {
                    st1 = (*lit);
                    st2 = (*it).m_name;
                    if ((*lit) == (*it).m_name) {
                        res += (*it).PrintDataWEB(print_percent, disable_errors, disable_summary) + "<br>";
                        break;
                    }

                    ++it;
                }
            }

            ++lit;
        }

        it = begin();
        while (it != end()) {
            exists = false;
            lit = orderlist.begin();
            while (lit != orderlist.end()) {
                if ((*lit) == (*it).m_name) {
                    exists = true;
                    break;
                }

                ++lit;
            }
            if (!exists)
                res += (*it).PrintDataWEB(print_percent, disable_errors, disable_summary) + "<br>";

            ++it;
        }

        return res;
    }

    TString PrintDataWEBInOrderShort(const std::vector<TString>& orderlist, bool print_percent, bool disable_errors, bool disable_summary) {
        TStorageStatVector::iterator it;
        TString res = "";
        std::vector<TString>::const_iterator lit;
        bool exists = false;
        TString st1 = "", st2 = "";

        std::sort(begin(), end());

        lit = orderlist.begin();
        while (lit != orderlist.end()) {
            if (!(*lit).empty()) {
                it = begin();
                while (it != end()) {
                    st1 = (*lit);
                    st2 = (*it).m_name;
                    if ((*lit) == (*it).m_name) {
                        res += (*it).PrintDataWEBShort(print_percent, disable_errors, disable_summary) + "<br>";
                        break;
                    }

                    ++it;
                }
            }

            ++lit;
        }

        it = begin();
        while (it != end()) {
            exists = false;
            lit = orderlist.begin();
            while (lit != orderlist.end()) {
                if ((*lit) == (*it).m_name) {
                    exists = true;
                    break;
                }

                ++lit;
            }
            if (!exists)
                res += (*it).PrintDataWEBShort(print_percent, disable_errors, disable_summary) + "<br>";

            ++it;
        }

        return res;
    }

    TString PrintDataLog() {
        TStorageStatVector::iterator it;
        TString res = "";

        std::sort(begin(), end());
        it = begin();
        while (it != end()) {
            res += (*it).PrintDataLog();

            ++it;
        }

        return res;
    }

    void ChangeLabelWeb(TStorageClientStat::TChangeLabelStructList& change_labels_list) {
        TStorageStatVector::iterator it;

        it = begin();
        while (it != end()) {
            (*it).ChangeLabelWeb(change_labels_list);

            ++it;
        }
    }

    void ChangeLabelLog(TStorageClientStat::TChangeLabelStructList& change_labels_list) {
        TStorageStatVector::iterator it;

        it = begin();
        while (it != end()) {
            (*it).ChangeLabelLog(change_labels_list);

            ++it;
        }
    }
};
typedef TStorageStatVector::iterator TStorageStatVectorIt;

class TStorageClientStatMain {
private:
    TStorageStatVector dbcoll_stor;
    TMutex m_Mutex;

public:
    TStorageClientStatMain();
    ~TStorageClientStatMain();

    void AddCount(const TString& db, const TString& collection, TStorageClientStat::TStorageActionType acttype, int count, bool good, ui32 tick);
    void Midnight();
    TStorageStatVector GetWebStat();
    TStorageStatVector GetMonStat();

    void ChangeLabelWeb(TStorageClientStat::TChangeLabelStructList& change_labels_list);
    void ChangeLabelLog(TStorageClientStat::TChangeLabelStructList& change_labels_list);
};

//***********************************************************************************************************************

#endif
