#include <util/charset/utf8.h>
#include <util/string/builder.h>
#include <library/cpp/yson/node/node.h>
#include <library/cpp/iterator/enumerate.h>

#include "uids_features_reader.h"

void TUidStats::AddStat(ui8 fieldIndex, TPeriodStats &&stats) {
    Stats.emplace(fieldIndex, std::move(stats));
}

void TUidStats::ToFeatures(THashMap<TString, float> &features) const {
    for (const auto&[fieldIndex, stats]: Stats) {
        const auto &fieldName = ToUpperUTF8(FIELDS_NAMES[fieldIndex]);
        for (size_t i = 0; i < STATS_COUNT; i++) {
            features[TStringBuilder{} << "USER_FACTORS_" + fieldName << STATS_SUFFIXES[i]] = stats[i];
        }
    }
}

void TUidsStats::AddNode(const NYT::TNode &node) {
    TUidStats stats;
    bool allStatsAreZero = true;
    const ui64 uid = node.ChildAsUint64("uid");
    for (const auto&[fieldIndex, fieldName] : Enumerate(TUidStats::FIELDS_NAMES)) {
        const NYT::TNode::TListType &fields = node.ChildAsList(fieldName);

        if (fields.size() != TUidStats::STATS_COUNT) {
            ythrow TWithBackTrace<yexception>() << "expected " << TUidStats::STATS_COUNT << " for uid " << uid;
        }

        TUidStats::TPeriodStats periodStats;
        bool allAreZero = true;
        for (size_t i = 0; i < TUidStats::STATS_COUNT; i++) {
            const ui64 value = fields[i].AsUint64();
            if (value > std::numeric_limits<TUidStats::TPeriodStats::value_type>::max()) {
                ythrow TWithBackTrace<yexception>() << "value is too big, " << LabeledOutput(uid, value);
            }
            if (value > 0) {
                allAreZero = false;
            }
            periodStats[i] = value;
        }
        if (!allAreZero) {
            allStatsAreZero = false;
            stats.AddStat(fieldIndex, std::move(periodStats));
        }
    }
    if (!allStatsAreZero) {
        UidsStats.emplace(uid, stats);
    }
}

const TUidStats *TUidsStats::GetStatsOrNull(ui64 uid) const {
    return MapFindPtr(UidsStats, uid);
}

size_t TUidsStats::Size() const {
    return UidsStats.size();
}
