#include <library/cpp/testing/unittest/registar.h>
#include <util/generic/maybe.h>
#include <util/string/vector.h>
#include <library/cpp/yson/node/node.h>
#include <library/cpp/yson/node/node_io.h>
#include <util/stream/file.h>
#include <library/cpp/archive/yarchive.h>
#include <util/memory/blob.h>
#include "uids_features_reader.h"

static TAutoPtr<IInputStream> GetUidsFeaturesYson() {
    static const unsigned char data[] = {
#include "data.inc"
    };
    TArchiveReader reader(TBlob::NoCopy(data, sizeof(data)));
    return reader.ObjectByKey("/uids_features.yson");
}

Y_UNIT_TEST_SUITE(UidsFeaturesReaderTest) {

    Y_UNIT_TEST(Main) {
        const NYT::TNode root = NYT::NodeFromYsonString(GetUidsFeaturesYson()->ReadAll(),
                                                        NYson::EYsonType::ListFragment);
        const NYT::TNode::TListType &nodes = root.AsList();
        TUidsStats uidsStats(nodes.size());
        for (const NYT::TNode &node : nodes) {
            uidsStats.AddNode(node);
        }

        UNIT_ASSERT_VALUES_EQUAL(100, uidsStats.Size());

        {
            const TUidStats *uidStats = uidsStats.GetStatsOrNull(859);
            UNIT_ASSERT(uidStats);

            THashMap<TString, float> features;
            uidStats->ToFeatures(features);
            const THashMap<TString, float> expected = {
                    {"USER_FACTORS_SEARCH_LAST_2_WEEKS",                1.f},
                    {"USER_FACTORS_DELETED_YESTERDAY",                  0.f},
                    {"USER_FACTORS_HAM_READ_LAST_WEEK",                 6.f},
                    {"USER_FACTORS_SPAM_LAST_3_MONTHS",                 40.f},
                    {"USER_FACTORS_LABELED_YESTERDAY",                  1.f},
                    {"USER_FACTORS_HAM_YESTERDAY",                      4.f},
                    {"USER_FACTORS_LINK_CLICK_LAST_2_WEEKS",            2.f},
                    {"USER_FACTORS_EMAIL_SEND_COUNT_LAST_MONTH",        1.f},
                    {"USER_FACTORS_EMAIL_RECEIVE_UNIQUE_LAST_2_MONTHS", 164.f},
                    {"USER_FACTORS_HAM_READ_LAST_MONTH",                6.f},
                    {"USER_FACTORS_SEARCH_LAST_2_MONTHS",               1.f},
                    {"USER_FACTORS_EMAIL_SEND_UNIQUE_LAST_MONTH",       1.f},
                    {"USER_FACTORS_DELETED_LAST_2_MONTHS",              355.f},
                    {"USER_FACTORS_DELETED_FINAL_LAST_2_WEEKS",         17.f},
                    {"USER_FACTORS_DELETED_LAST_WEEK",                  355.f},
                    {"USER_FACTORS_MOVE_TO_TRASH_LAST_MONTH",           355.f},
                    {"USER_FACTORS_MOVE_LAST_3_MONTHS",                 355.f},
                    {"USER_FACTORS_HAM_LAST_MONTH",                     107.f},
                    {"USER_FACTORS_LINK_CLICK_YESTERDAY",               0.f},
                    {"USER_FACTORS_HAM_READ_LAST_3_MONTHS",             6.f},
                    {"USER_FACTORS_MARK_READ_LAST_3_MONTHS",            4.f},
                    {"USER_FACTORS_EMAIL_RECEIVE_UNIQUE_YESTERDAY",     5.f},
                    {"USER_FACTORS_MARK_READ_LAST_WEEK",                4.f},
                    {"USER_FACTORS_MOVE_YESTERDAY",                     0.f},
                    {"USER_FACTORS_HAM_READ_YESTERDAY",                 0.f},
                    {"USER_FACTORS_HAM_LAST_3_MONTHS",                  320.f},
                    {"USER_FACTORS_EMAIL_SEND_COUNT_LAST_2_WEEKS",      1.f},
                    {"USER_FACTORS_MOVE_LAST_MONTH",                    355.f},
                    {"USER_FACTORS_HAM_READ_LAST_2_MONTHS",             6.f},
                    {"USER_FACTORS_SPAM_LAST_MONTH",                    8.f},
                    {"USER_FACTORS_DELETED_LAST_MONTH",                 355.f},
                    {"USER_FACTORS_LINK_CLICK_LAST_3_MONTHS",           2.f},
                    {"USER_FACTORS_EMAIL_SEND_UNIQUE_LAST_3_MONTHS",    1.f},
                    {"USER_FACTORS_MOVE_TO_TRASH_LAST_2_WEEKS",         355.f},
                    {"USER_FACTORS_MARK_READ_YESTERDAY",                0.f},
                    {"USER_FACTORS_EMAIL_SEND_COUNT_LAST_3_MONTHS",     1.f},
                    {"USER_FACTORS_EMAIL_RECEIVE_UNIQUE_LAST_2_WEEKS",  42.f},
                    {"USER_FACTORS_DELETED_FINAL_LAST_MONTH",           32.f},
                    {"USER_FACTORS_HAM_READ_LAST_2_WEEKS",              6.f},
                    {"USER_FACTORS_SEARCH_LAST_3_MONTHS",               1.f},
                    {"USER_FACTORS_MOVE_TO_TRASH_LAST_WEEK",            355.f},
                    {"USER_FACTORS_LINK_CLICK_LAST_WEEK",               2.f},
                    {"USER_FACTORS_DELETED_LAST_3_MONTHS",              355.f},
                    {"USER_FACTORS_EMAIL_SEND_UNIQUE_LAST_WEEK",        1.f},
                    {"USER_FACTORS_LABELED_LAST_2_MONTHS",              56.f},
                    {"USER_FACTORS_LABELED_LAST_2_WEEKS",               15.f},
                    {"USER_FACTORS_LABELED_LAST_MONTH",                 30.f},
                    {"USER_FACTORS_EMAIL_SEND_UNIQUE_YESTERDAY",        0.f},
                    {"USER_FACTORS_EMAIL_SEND_UNIQUE_LAST_2_MONTHS",    1.f},
                    {"USER_FACTORS_MOVE_TO_TRASH_LAST_2_MONTHS",        355.f},
                    {"USER_FACTORS_DELETED_LAST_2_WEEKS",               355.f},
                    {"USER_FACTORS_MOVE_LAST_2_MONTHS",                 355.f},
                    {"USER_FACTORS_SEARCH_LAST_WEEK",                   1.f},
                    {"USER_FACTORS_SPAM_YESTERDAY",                     1.f},
                    {"USER_FACTORS_DELETED_FINAL_LAST_WEEK",            10.f},
                    {"USER_FACTORS_EMAIL_SEND_COUNT_LAST_WEEK",         1.f},
                    {"USER_FACTORS_DELETED_FINAL_LAST_2_MONTHS",        58.f},
                    {"USER_FACTORS_MOVE_TO_TRASH_YESTERDAY",            0.f},
                    {"USER_FACTORS_EMAIL_SEND_COUNT_YESTERDAY",         0.f},
                    {"USER_FACTORS_LINK_CLICK_LAST_MONTH",              2.f},
                    {"USER_FACTORS_SPAM_LAST_WEEK",                     3.f},
                    {"USER_FACTORS_SPAM_LAST_2_MONTHS",                 19.f},
                    {"USER_FACTORS_DELETED_FINAL_LAST_3_MONTHS",        98.f},
                    {"USER_FACTORS_SEARCH_YESTERDAY",                   0.f},
                    {"USER_FACTORS_MOVE_LAST_WEEK",                     355.f},
                    {"USER_FACTORS_LINK_CLICK_LAST_2_MONTHS",           2.f},
                    {"USER_FACTORS_MOVE_TO_TRASH_LAST_3_MONTHS",        355.f},
                    {"USER_FACTORS_HAM_LAST_2_WEEKS",                   65.f},
                    {"USER_FACTORS_MARK_READ_LAST_MONTH",               4.f},
                    {"USER_FACTORS_LABELED_LAST_3_MONTHS",              96.f},
                    {"USER_FACTORS_MARK_READ_LAST_2_MONTHS",            4.f},
                    {"USER_FACTORS_HAM_LAST_2_MONTHS",                  206.f},
                    {"USER_FACTORS_EMAIL_RECEIVE_UNIQUE_LAST_3_MONTHS", 261.f},
                    {"USER_FACTORS_MARK_READ_LAST_2_WEEKS",             4.f},
                    {"USER_FACTORS_EMAIL_RECEIVE_UNIQUE_LAST_MONTH",    79.f},
                    {"USER_FACTORS_SEARCH_LAST_MONTH",                  1.f},
                    {"USER_FACTORS_MOVE_LAST_2_WEEKS",                  355.f},
                    {"USER_FACTORS_EMAIL_SEND_UNIQUE_LAST_2_WEEKS",     1.f},
                    {"USER_FACTORS_EMAIL_SEND_COUNT_LAST_2_MONTHS",     1.f},
                    {"USER_FACTORS_EMAIL_RECEIVE_UNIQUE_LAST_WEEK",     27.f},
                    {"USER_FACTORS_SPAM_LAST_2_WEEKS",                  4.f},
                    {"USER_FACTORS_HAM_LAST_WEEK",                      51.f},
                    {"USER_FACTORS_LABELED_LAST_WEEK",                  8.f},
                    {"USER_FACTORS_DELETED_FINAL_YESTERDAY",            1.f},
            };
            UNIT_ASSERT_VALUES_EQUAL(expected, features);
        }
        {
            const TUidStats *uidStats = uidsStats.GetStatsOrNull(6843);
            UNIT_ASSERT(uidStats);
        }
    }
}
