#include <mail/so/api/so_api.pb.h>
#include <mail/so/libs/syslog/so_log.h>

#include <google/protobuf/util/json_util.h>

#include <library/cpp/getopt/last_getopt.h>

#include <util/generic/ptr.h>
#include <util/generic/yexception.h>
#include <util/memory/blob.h>
#include <util/stream/output.h>

enum class EFormat {
    None,
    Protobuf,
    ProtobufJson
};

enum class EType {
    Request,
    Response
};

EFormat ParseFormat(const TString& value, const TString& description) {
    if (value == "protobuf") {
        return EFormat::Protobuf;
    } else if (value == "protobuf-json") {
        return EFormat::ProtobufJson;
    } else {
        ythrow yexception() << "Invalid " << description << ", unknown format: " << value;
    }
}

EType ParseType(const TString& value) {
    if (value == "request") {
        return EType::Request;
    } else if (value == "response") {
        return EType::Response;
    } else {
        ythrow yexception() << "Invalid type: " << value;
    }
}

template <class TMessage>
int Validate(const TBlob& blob, EFormat inputFormat, EFormat outputFormat) {
    TMessage message;
    if (inputFormat == EFormat::Protobuf) {
        if (!message.ParseFromString({blob.AsCharPtr(), blob.Size()})) {
            Cerr << "Failed to parse protobuf message" << Endl;
            return 1;
        }
    } else if (inputFormat == EFormat::ProtobufJson) {
        auto status = NProtoBuf::util::JsonStringToMessage(NProtoBuf::string{blob.AsCharPtr(), blob.Size()}, &message);
        if (!status.ok()) {
            Cerr << "Failed to parse protobuf json message: " << status.ToString() << Endl;
            return 1;
        }
    }
    if (outputFormat == EFormat::Protobuf) {
        NProtoBuf::string str;
        Y_PROTOBUF_SUPPRESS_NODISCARD message.SerializeToString(&str);
        Cout << str;
    } else if (outputFormat == EFormat::ProtobufJson) {
        NProtoBuf::string str;
        NProtoBuf::util::JsonOptions options;
        options.add_whitespace = true;
        options.preserve_proto_field_names = true;
        NProtoBuf::util::MessageToJsonString(message, &str, options);
        Cout << str << Endl;
    }
    return 0;
}

int main(int argc, char* argv[]) {
    NLastGetopt::TOpts opts;
    NLastGetopt::TOpt& inputFormatOpt = opts.AddLongOption('i', "input-format", "Input format: protobuf or protobuf-json");
    inputFormatOpt.Required().RequiredArgument();
    NLastGetopt::TOpt& outputFormatOpt = opts.AddLongOption('o', "output-format", "Output format: protobuf or protobuf-json");
    outputFormatOpt.Optional();
    NLastGetopt::TOpt& typeOpt = opts.AddLongOption('t', "type", "Message type: request or response");
    typeOpt.Required().RequiredArgument();

    THolder<NLastGetopt::TOptsParseResult> r;
    EFormat inputFormat = EFormat::None;
    EFormat outputFormat = EFormat::None;
    EType type = EType::Request;
    try {
        r.Reset(new NLastGetopt::TOptsParseResult(&opts, argc, argv));
        inputFormat = ParseFormat(r->Get(&inputFormatOpt), "input format");;
        if (r->Has(&outputFormatOpt)) {
            outputFormat = ParseFormat(r->Get(&outputFormatOpt), "output format");
        }
        type = ParseType(r->Get(&typeOpt));
    } catch (...) {
        opts.PrintUsage(argv[0], Cerr);
        Cerr << CurrentExceptionMessageWithBt() << Endl;
        return 1;
    }
    TBlob input = TBlob::FromStream(Cin);
    if (type == EType::Request) {
        return Validate<mail::so::api::v1::SoRequest>(input, inputFormat, outputFormat);
    } else {
        return Validate<mail::so::api::v1::SoResponse>(input, inputFormat, outputFormat);
    }
}

