#!/usr/bin/python3
# encoding: utf-8
# kate: space-indent on; indent-width 4; replace-tabs on;
#
import os, os.path, sys, re, time, json, argparse, asyncio, aiohttp
from concurrent.futures import ThreadPoolExecutor, as_completed
from urllib.request import urlopen, Request
from traceback import format_exception
from html import unescape

global errorlog_fh, log_fh

DEFAULT_RETRY_CNT = 3
PASSPORT_RETRY_CNT = 10
BLACKBOX_RETRY_CNT = 10
QUEUE_SIZE = 100000
WORKERS_COUNT = 100
READERS_COUNT = 2
HOME_DIR = "/root"
LOG_DIR =  os.environ['PWD'] if 'PWD' in os.environ else "/var/log/so-logs"
ERRORLOG = "%s/admkarma.log" % LOG_DIR
LOG="-"
TVM = {
    "API": {
        "host":           "localhost",
        "port":           1111,
        "url":            "http://{0}:{1}/tvm",
        "URL":            "http://localhost:1111/tvm"
    },
    "BB": {
        "host":           "blackbox-mail.yandex.net",
        "url":            "https://{0}/blackbox?",
        "URL":            "https://blackbox-mail.yandex.net/blackbox?",
        "client_id":      2016577,
        "prod_client_id": 222,
        "timeout":        3.0
    },
    "BBcorp": {
        "host":           "blackbox.yandex-team.ru",
        "url":            "https://{0}/blackbox?",
        "URL":            "https://blackbox.yandex-team.ru/blackbox?",
        "client_id":      2016577,
        "prod_client_id": 223,
        "timeout":        3.0
    },
    "Passport": {
        "host":           'passport-internal.yandex.ru',
        "url":            'https://{0}',
        "URL":            'https://passport-internal.yandex.ru',
        "client_id":      2016577,
        "prod_client_id": 2000078,
        "timeout":        2.0
    }
}

def get_traceback():
    exc_type, exc_value, exc_traceback = sys.exc_info()
    tb = ''
    for step in format_exception(exc_type, exc_value, exc_traceback):
        try:
            tb += "\t" + step.strip() + "\n"
        except:
            pass
    return tb


def writelog(msg, isTB=False, addTS=True, log_fh=sys.stderr):
    if not msg:
        return
    try:
        tb = "\n"
        if isTB:
            tb += get_traceback()
        if addTS:
            msg = time.strftime("[%Y-%m-%d %H:%M:%S]: ") + msg
        if log_fh and not log_fh.closed:
            os.write(log_fh.fileno(), bytes(msg + tb, "utf8"))
        else:
             print(msg + tb, file=sys.stderr)
    except Exception as e:
        print("Writelog error: %s.%s" % (str(e), get_traceback()), file=sys.stderr)
        print(time.strftime("[%Y-%m-%d %H:%M:%S]: ") + msg + tb, file=sys.stderr)
        sys.stderr.flush()


async def errorlog(msg, isTB=True):
    global errorlog_fh
    writelog(msg, isTB, True, log_fh=errorlog_fh)


async def log(msg):
    global log_fh
    writelog(msg, False, False, log_fh)


async def requestService(url, headers={}, data=None, retDataType='str'):
    resp, code = "", 0
    client = aiohttp.ClientSession()
    if data:
        async with client.post(url, data=data, headers=headers) as r:
            if retDataType == "dict" or retDataType == "json":
                resp = await r.json()
            else:
                resp = await r.text()
            code = r.status
    else:
        async with client.get(url, headers=headers) as r:
            if retDataType == "dict" or retDataType == "json":
                resp = await r.json()
            else:
                resp = await r.text()
            code = r.status
    client.close()
    return resp, code


def loadTVMConfig():
    f, CURDIR, tvm_port = None, HOME_DIR, TVM["API"]["port"]
    try:
        with open("/etc/tvmtool/tvmtool.conf") as f:
            try:
                tvm_config = json.loads(f.read().strip())
                if 'port' in tvm_config:
                    tvm_port = int(tvm_config['port'])
            except Exception as e:
                errorlog("Failed to parse tvmtool's config! %s" % str(e))
        TVM['API']['URL'] = TVM['API']['url'].format(TVM['API']['host'], tvm_port)
        with open("/var/lib/tvmtool/local.auth") as f:
            TVM['API']['Auth'] = f.read().strip()
    except Exception as e:
        writelog("loadTVMConfig failed: %s" % str(e), True)


async def getTVM2ticket(service):
    resp, svc = {}, service.lower()
    try:
        resp, code = await requestService('%s/tickets?dsts=%s&src=%s' % (TVM['API']['URL'], svc, TVM[service]["client_id"]), headers={"Authorization": TVM['API']['Auth']}, retDataType="str")
        if code != 200:
            await errorlog("getTVM2ticket HTTPS error for service '%s' (status=%s): %s" % (service, code, str(resp)), False)
        try:
            if not resp or len(resp) < 1:
                await errorlog("getTVM2ticket error answer for service '%s' (code=%s): %s" % (service, code, str(resp)), False)
            else:
                resp = json.loads(resp)
        except Exception as e:
            await errorlog("getTVM2ticket decoding of JSON from TVM ticket query's answer failed for service '%s': %s. Answer: %s." % (service, str(e), str(resp)))
    except Exception as e:
        await errorlog("getTVM2ticket HTTPS request failed for service '%s': %s." % (service, str(e)))
    return resp[svc]['ticket'] if isinstance(resp, dict) and svc in resp and resp[svc] else None


async def requestServiceByTVM(url, service, headers={}, data=None, retDataType='str', retryCnt=DEFAULT_RETRY_CNT):
    resp, code = '', 200
    ticket = await getTVM2ticket(service)
    #if service.startswith("BB"):
    #    print("Ticket: %s" % ticket)
    if ticket:
        if not headers:
            headers = {}
        headers['X-Ya-Service-Ticket'] = ticket
        for i in range(retryCnt):
            try:
                resp, code = await requestService(url, headers=headers, data=data, retDataType=retDataType)
            except Exception as e:
                await errorlog('requestServiceByTVM failed: %s. TicketAnswer: %s' % (str(e), str(ticket)))
            if code == 200:
                break
            else:
                await errorlog('requestServiceByTVM failed (attempt #%s from %s, status=%s). Response: "%s"' % (i + 1, retryCnt, code, str(resp)), False)
                time.sleep(2 * i + 1)
                continue
    return resp, code


async def requestBlackBox(params, bb_type='BB', tvm=True, retryCnt=DEFAULT_RETRY_CNT):
    url = params if not isinstance(params, dict) else ('&'.join(map(lambda k, v: "{0}={1}".format(k, v), params.iteritems())) if isinstance(params, dict) else '')
    url, resp, code = TVM[bb_type]['URL'] + url, '', 0
    if tvm:
        resp = (await requestServiceByTVM(url, bb_type, retryCnt=retryCnt))[0]
    else:
        for i in range(retryCnt):
            try:
                resp, code = await requestService(url)
            except Exception as e:
                await errorlog('BB failed: %s' % str(e))
            if code == 200:
                break
            else:
                await errorlog('BB failed (attempt #%s from %s, status=%s). Response: "%s"' % (i + 1, retryCnt, code, resp), False)
                time.sleep(2 * i + 1)
                continue
    return resp


async def getUserInfo(filter_str, bb_type='BB', info_type="brief"):
    add_info_params = ""
    if info_type and info_type != "brief" and info_type != "uid":
        add_info_params = ",accounts.login.uid,subscription.suid.2"
        if info_type == "full":
            add_info_params += ",subscription.suid.669,userinfo.reg_date.uid,userinfo.country.uid&getemails=all&email_attributes=1"
    params = "method=userinfo&%s&userip=127.0.0.1&dbfields=subscription.suid.2%s" % (filter_str, add_info_params)   # "&format=json" - for JSON answer
    info = await requestBlackBox(params, bb_type, retryCnt=BLACKBOX_RETRY_CNT)
    info2 = ''
    m = re.search(r'<uid\b[^<>]*?>(\d+)<\/uid>', info)
    uid, login, suid, karma_status, reg_date, country, sid669 = m.group(1) if m and m.group(1) else '', '', '', '', '', '', ''
    if info_type and info_type != "uid":
        m = re.search(r'<dbfield id="accounts.login.uid">(.*?)<\/dbfield>', info)
        if m and m.group(1):
            login = m.group(1)
        else:
            m = re.search(r'<login>([^<]+)', info)
            if m and m.group(1):
                login = m.group(1)
        if info_type != "brief":
            m = re.search(r'^<dbfield id="subscription.suid.2">(\d+)</dbfield>', info, re.M)
            if m and m.group(1):
                suid = m.group(1)
            if info_type == "full":
                m = re.search(r'^<karma_status>(\d+)', info, re.M)
                if m and m.group(1):
                    karma_status = m.group(1)
                m = re.search(r'^<dbfield id="userinfo.reg_date.uid">([^<>]*?)</dbfield>', info, re.M)
                if m and m.group(1):
                    reg_date = m.group(1)
                m = re.search(r'^<dbfield id="userinfo.country.uid">([^<>]*?)</dbfield>', info, re.M)
                if m and m.group(1):
                    country = m.group(1)
                m = re.search(r'^<dbfield id="subscription.suid.669">([^<>]*?)</dbfield>', info, re.M)
                if m and m.group(1):
                    sid669 = m.group(1)
                m = re.search(r'<emails>(.+?)</emails>', info, re.S)
                if m and m.group(1):
                    info2 = ','.join(re.findall(r'<attribute.*?>([^<>]+?)</attribute>', m.group(1)))
    return uid, login, suid, karma_status, reg_date, country, sid669, info2


async def printUserInfo(info, info_type="uid"):
    s = "UID=%s" % info[0]
    if info_type != "uid":
        s += "\tLOGIN=%s" % info[1]
        if info_type != "brief":
            s += "\tSUID=%s" % info[2]
            if info_type == "full":
                s += "\tKARMA=%s\tREGDATE=%s\tCOUNTRY=%s\tSID669=%s" % (info[3], info[4], info[5], info[6])
                if len(info) > 7:
                    s += "\tEMAILS=%s" % info[7]
    await log(s)
    return True


def getAnswer(resp):
    s = ""
    if isinstance(resp, dict) and "status" in resp:
        if resp["status"] == "error":
            s = ', '.join(resp["errors"])
        else:
            s = resp["status"]
    else:
        s = resp
    return s


async def logoutUser(uid, params):
    resp, code, url = '', 200, "{0}/2/account/{1}/password_options/?consumer=so".format(TVM['Passport']['URL'], uid)
    try:
        errors = []
        data = {
            "comment":                      params.get("comment", ""),
            "admin_name":                   params.get("admin", ""),
            "global_logout":                params.get("global_logout", 1),
            "is_changing_required":         params.get("is_changing_required", 'yes'),
            "max_change_frequency_in_days": params.get("max_change_frequency_in_days", 4)
        }
        if uid.startswith("113000") and len(uid) == 16 and data["is_changing_required"] == "yes":
            info = await requestBlackBox("method=userinfo&uid=%s&sid=2&userip=127.0.0.1" % uid, retryCnt=BLACKBOX_RETRY_CNT)
            m, domain = re.search(r'<uid\b[^<>]*?\bdomain="([^<>\"]+?)"[^<>]*?>(\d+)<\/uid>', info), ""
            if m and m.group(1):
                domain = "domain=%s" % m.group(1)
            else:
                m = re.search(r'<uid\b[^<>]*?\bdomid="(\d+?)"[^<>]*?', info)
                if m and m.group(1):
                    domain = "domain_id=%s" % m.group(1)
            if domain:
                info2 = await requestBlackBox("method=hosted_domains&%s&aliases=true" % domain, retryCnt=BLACKBOX_RETRY_CNT)
                m = re.search(r'<item\b[^<>]*?\boptions="([^<>\"]*?)"', info2)
                if m and m.group(1):
                    try:
                        data2 = json.loads(unescape(m.group(1)))
                    except Exception as e:
                        await errorlog('logoutUser failed for PDD uid=%s: %s' % (uid, str(e)))
                    else:
                        if "can_users_change_password" in data2 and data2["can_users_change_password"] == 0:
                            params["is_changing_required"] = "no"
                            errors += ["user_cant_change_password"]
        resp, code = await requestServiceByTVM(url, "Passport", data=data, retDataType='json', retryCnt=PASSPORT_RETRY_CNT)
        if code == 200:
            await log("LOGOUT user with UID = %s: %s" % (uid, getAnswer(resp)))
        else:
            await log("LOGOUT user with UID = %s failed! (code=%s, url=%s, data=%s): %s" % (uid, code, url, str(data), resp))
    except Exception as e:
        await errorlog('logoutUser failed for uid=%s: %s' % (uid, str(e)))
        if errors:
            resp += ". Errors: %s" + ", ".join(errors)
    return resp


async def changeUserPasswordOptions(uid, params):
    resp, code, url = '', 200, "{0}/2/account/{1}/password_options/?consumer=so".format(TVM['Passport']['URL'], uid)
    try:
        data = {
            "comment":              params.get("comment", ""),
            "admin_name":           params.get("admin", ""),
            "is_changing_required": params.get("is_changing_required", 'yes')
        }
        if "max_change_frequency_in_days" in params:
            data["max_change_frequency_in_days"] = params["max_change_frequency_in_days"]
        resp, code = await requestServiceByTVM(url, "Passport", data=data, retDataType='json', retryCnt=PASSPORT_RETRY_CNT)
        if code == 200:
            await log("CHANGE PASSWORD OPTION for user with UID = %s: %s" % (uid, getAnswer(resp)))
        else:
            await log("CHANGE PASSWORD OPTION for user with UID = %s failed! (code=%s, url=%s, data=%s): %s" % (uid, code, url, str(data), resp))
    except Exception as e:
        await errorlog('changeUserPasswordOptions failed for uid=%s: %s' % (uid, str(e)))
    return resp


async def changeUserOptions(uid, params, options):
    resp, code, url = '', 200, "{0}/2/account/{1}/options/?consumer=so".format(TVM['Passport']['URL'], uid)
    try:
        data = {
            "comment":    params.get("comment", ""),
            "admin_name": params.get("admin", "")
        }
        for option in options:
            data[option] = "yes"
        resp, code = await requestServiceByTVM(url, "Passport", data=data, retDataType='json', retryCnt=PASSPORT_RETRY_CNT)
        if code == 200:
            await log("CHANGE OPTION for user with UID = %s: %s" % (uid, getAnswer(resp)))
        else:
            await log("CHANGE OPTION for user with UID = %s failed! (code=%s, url=%s, data=%s): %s" % (uid, code, url, str(data), resp))
    except Exception as e:
        await errorlog('changeUserOptions failed for uid=%s: %s' % (uid, str(e)))
    return resp


async def setKarma(uid, karma, params, strict=False):
    resp, code, url = '', 200, "{0}/1/account/{1}/karma/?consumer=so".format(TVM['Passport']['URL'], uid)
    data = {
        "comment":    params.get("comment", ""),
        "admin_name": params.get("admin", "")
    }
    if karma >= 1000:
        data["prefix"] = karma // 1000
        if strict:
            data["suffix"] = karma % 1000
    else:
        data["suffix"] = karma
        if strict:
            data["prefix"] = 0
    try:
        resp, code = await requestServiceByTVM(url, "Passport", data=data, retDataType='json', retryCnt=PASSPORT_RETRY_CNT)
        if code == 200:
            await log("SET_KARMA for user with UID = %s%s: %s" % (uid, " (strict)" if strict else "", getAnswer(resp)))
        else:
            await log("SET_KARMA for user with UID = %s failed! (code=%s, url=%s, data=%s): %s" % (uid, code, url, str(data), resp))
    except Exception as e:
        await errorlog('setKarma failed: %s' % str(e))
    return resp


async def banUser(uid, params):
    resp, code, url = '', 200, "{0}/1/account/{1}/karma/?consumer=so".format(TVM['Passport']['URL'], uid)
    data = {
        "comment":    params.get("comment", ""),
        "admin_name": params.get("admin", ""),
        "is_enabled": 0
    }
    try:
        resp, code = await requestServiceByTVM(url, "Passport", data=data, retDataType='json', retryCnt=PASSPORT_RETRY_CNT)
        if code == 200:
            await log("BAN user with UID = %s: %s" % (uid, getAnswer(resp)))
        else:
            await log("BAN user with UID = %s failed! (code=%s, url=%s, data=%s): %s" % (uid, code, url, str(data), resp))
    except Exception as e:
        await errorlog('banUser failed: %s' % str(e))
    return resp


async def processItem(params, q):
    while True:
        item = await q.get()
        if item is None:
            break
        sid, aInfo, karma, karma_strict, karma_set, info_printed, options = '&sid=2' if params["key_type"] == "suid" else "", "", 0, False, False, False, []
        filter_str = "{0}={1}{2}".format(str(params["key_type"]), str(item), sid)
        for action in params['actions']:
            if action.startswith("info"):
                info_type = 'common' if '_' not in action else action[action.find('_')+1:]
                if not aInfo or aInfo == "uid" and info_type != "uid" or aInfo == "brief" and (info_type == "common" or info_type == "full") or aInfo == "common" and info_type == "full":
                    aInfo = info_type
            elif action.startswith("setkarma"):
                m = re.match(r'setkarma_(\d+)(s)?', action)
                if m:
                    karma = int(m.group(1))
                    karma_strict = True if m.group(2) else False
            elif action.startswith("option_"):
                m = re.match(r'option_(\w+)?', action)
                if m:
                    options += [m.group(1)]
        info, uid = ('', '', '', '', '', '', ''), ''
        if aInfo == 'uid' and params["key_type"] == "uid":
            uid = str(item)
            info = (uid, '', '', '', '', '', '')
        else:
            if params['bb_type'] == 'big':
                info = await getUserInfo(filter_str, 'BB', aInfo)    # tuple (uid, login, suid, karma_status, reg_date, country)
            if not info[0] or params['bb_type'] == 'corp':
                info = await getUserInfo(filter_str, 'BBcorp', aInfo)
            uid = str(info[0])
        if uid:
            for action in params['actions']:
                if action.startswith("info") and not info_printed and info[0]:
                    await printUserInfo(info, aInfo)
                    info_printed = True
                elif action.startswith("setkarma") and not karma_set:
                    await setKarma(uid, karma, params, karma_strict)
                    karma_set = True
                elif action == "logout":
                    await logoutUser(uid, params)
                elif action == "change_options":
                    await changeUserPasswordOptions(uid, params)
                elif action == "ban":
                    await banUser(uid, params)
                elif action.startswith("option_") and len(options) > 0:
                    await changeUserOptions(uid, params, options)
        else:
            await errorlog("Unable find UID for user with %s=%s" % (str(params["key_type"]), str(item)), False)


async def readItem(f, q):
    for row in f:
        await q.put(row.strip())
    for i in range(WORKERS_COUNT):
        await q.put(None)


def processItems(params, f):
    loop = asyncio.get_event_loop()
    q = asyncio.Queue(QUEUE_SIZE)
    workers = [processItem(params, q) for _ in range(WORKERS_COUNT)]
    readers = [readItem(f, q) for _ in range(READERS_COUNT)]
    loop.run_until_complete(asyncio.gather(*workers, *readers))


if __name__ == "__main__":
    epilog = "" if 'USER' in os.environ and os.environ['USER'] == "root" else "WARNING: This script must be run under root!"
    parser = argparse.ArgumentParser(epilog=epilog)
    parser.add_argument('-a', '--action',          type=str,             help="Action(s) (one or more) which will be done on input users data. Possible values are: "
                        "1) one and only one of the 'info_uid' (default), 'info_brief', 'info', 'info_full';  "
                        "2) 'logout' (see https://wiki.yandex-team.ru/passport/api/bundle/changeaccount/#izmenitnastrojjkiparoljaprinuditelnajasmenaotzyvavtorizaciiit.p);  "
                        "3) 'change_options';  "
                        "4) one and the only one from the values of the form 'setkarma_*': 'setkarma_85', 'setkarma_100', 'setkarma_1000', 'setkarma_2000', 'setkarma_85s' (suffix 's' means 'strict'), 'setkarma_100s', 'setkarma_1000s', 'setkarma_2000s',;  "
                        "5) one and the only one from the values of the form 'option_*': 'option_revoke_tokens' (see https://wiki.yandex-team.ru/passport/python/api/bundle/account/#izmeneniesvojjstvakkaunta);  "
                        "6) 'ban'. Actions will be done in the specified sequence.")
    parser.add_argument('-b', '--bb_type',    default='big',   type=str, help="Preffered BlackBox type: big (by default) or corp")
    parser.add_argument('-c', '--comment',                     type=str, help="Comment (for Passport)")
    parser.add_argument('-e', '--error_log',  default='-',     type=str, help="Error log's file path or '-' (or 'stderr') for output to pipe")
    parser.add_argument('-i', '--input',      default='-',     type=str, help="Input source data: file or '-' (or 'stdin') for input from pipe (by default)")
    parser.add_argument('-o', '--output',     default='-',     type=str, help="Output file path or '-' (or 'stdout') for output to pipe (by default)")
    parser.add_argument('-t', '--input_type', default="uids",  type=str, help="Input data type: 'uids' (default) or 'logins' or 'suids'")
    parser.add_argument('--non_global_logout',     action='store_false', help="Whether not a global logout will be done (false by default, i.e. it'll be global)")
    parser.add_argument('--not_changing_required', action='store_false', help="Whether a password changing is not required (false by default, i.e. it'll be required)")
    parser.add_argument('--max_change_frequency',  default=-1, type=int, help="Max frequency of password's changing in days count")
    args = parser.parse_known_args()[0]
    if len(sys.argv) < 2:
        parser.print_help()
        sys.exit(0)
    loadTVMConfig()
    key_type, input_file = '', '-'
    params = {
        "actions": args.action.split(',') if args.action else ["info_uid"],
        "comment": args.comment if args.comment else '',
        "admin":   os.environ['USER'] if 'USER' in os.environ else 'root'
    }
    if args.error_log:
        ERRORLOG = args.error_log
    errorlog_fh = sys.stderr if ERRORLOG == '-' or ERRORLOG == 'stderr' else open(ERRORLOG, "a+t")
    if args.output:
        LOG = args.output
    log_fh = sys.stdout if LOG == '-' or LOG == 'stdout' else open(LOG, "a+t")
    if "logout" in params["actions"]:
        params['global_logout'] = 1 if args.non_global_logout else 0
    if "logout" in params["actions"] or "change_options" in params["actions"]:
        params['is_changing_required'] = 'yes' if args.not_changing_required else 'no'
        if args.max_change_frequency > -1:
            params["max_change_frequency_in_days"] = args.max_change_frequency
    elif any(map(lambda x: x.startswith("option_"), params["actions"])):
        params['is_password_change_required'] = 'yes' if args.not_changing_required else 'no'
        if args.max_change_frequency > -1:
            params["max_change_frequency_in_days"] = args.max_change_frequency
    if args.input_type == "logins":
        params['key_type'], input_file = "login", args.input
    elif args.input_type == "suids":
        params['key_type'], input_file = "suid", args.input
    else:
        params['key_type'], input_file = "uid", args.input
    if params.get('key_type', ''):
        if input_file != '-' and input_file != 'stdin' and not os.path.exists(input_file):
            print("Unable find file '%s'" % input_file)
            sys.exit(1)
        else:
            params['bb_type'] = args.bb_type
            try:
                f = sys.stdin if input_file == '-' or input_file == 'stdin' else open(input_file, 'rt')
                processItems(params, f)
                f.close()
            except Exception as e:
                writelog("Exception: %s" % str(e), True, log_fh=errorlog_fh)

