#!/usr/bin/python3
# encoding: utf-8
# kate: space-indent on; indent-width 4; replace-tabs on;
#
import os
import sys
import re
from socket import gethostname
from signal import signal, SIGTERM, SIGHUP, SIG_IGN
from tornado.httpclient import AsyncHTTPClient
from tornado.ioloop import IOLoop
from tornado.web import RequestHandler, Application, MissingArgumentError
from tornado.httpserver import HTTPServer
from tornado.process import task_id
from tornado.gen import multi
from tornado.escape import json_encode
from tornado.options import define, options, parse_command_line, print_help

from mail.so.spamstop.users.admkarma_app.logger import configureLoggers, trace
from mail.so.spamstop.users.admkarma_app.utils import isCorp, loadTVMConfig
from mail.so.spamstop.users.admkarma_app.blackbox import getUserInfo
from mail.so.spamstop.users.admkarma_app.passport import logoutUser, banUser, changeUserOptions, changeUserPasswordOptions, setKarma


define("debug",     default=True, type=bool, help="run in debug mode", group="general")
define("home_dir",  default="/opt/admkarma-app", type=str, help="home directory for application", group="general")
# define("config",    default="./config.py", type=str, help="config file path", group="general", callback=lambda path: parse_config_file(path, final=False))

define("port",      default=8102, type=int, help="run on the given port", group="server")
define("processes", default=10, type=int, help="run with number of workers specified", group="server")
# define("pidfile",   default="/var/run/admkarma.pid", type=str, help="pid file path", group="server")

define("log_path",  default="/var/log/so-logs/admkarma", type=str, help="log path prefix", group="logs")


# def writePid(pidfile, pid):
#     print(pid, file=open(pidfile, "wt"))


class UserInfo():
    async def info(self):
        key_type = self.get_query_argument("key_type", "uid")
        key = self.get_query_argument("key")
        sid = "&sid=2" if key_type == "suid" else ""
        filter_str = "{0}={1}{2}".format(key_type, key, sid)
        bb_type = 'BBcorp' if (key_type == "uid" and isCorp(key) or key_type != "uid") and self.get_query_argument('bb_type', 'big') == 'corp' else 'BB'
        info_type = self.get_query_argument("info_type", "simple")
        info = await getUserInfo(filter_str, bb_type, info_type)
        if not info["uid"]:
            bb_type == 'BBcorp' if bb_type == "BB" else "BB"
            info = await getUserInfo(filter_str, bb_type, info_type)
        return info

    async def uid(self):
        key_type, uid = self.get_query_argument("key_type", "uid"), None
        if key_type == "uid":
            uid = self.get_query_argument("key", "")
        else:
            uid = (await self.info())["uid"]
        return uid


class UsersInfo():
    async def userInfo(self, key, key_type, bb_type, info_type):
        bb_type = 'BBcorp' if (key_type == "uid" and isCorp(key) or key_type != "uid") and bb_type == 'corp' else 'BB'
        filter_str = "{0}={1}{2}".format(key_type, key, "&sid=2" if key_type == "suid" else "")
        info = await getUserInfo(filter_str, bb_type, info_type)
        if not info["uid"]:
            bb_type == 'BBcorp' if bb_type == "BB" else "BB"
            info = await getUserInfo(filter_str, bb_type, info_type)
        return info

    async def info(self):
        key_type = self.get_query_argument("key_type", "uid")
        bb_type = self.get_query_argument("bb_type", "big")
        info_type = self.get_query_argument("info_type", "simple")
        items = self.get_argument("keys", "").split(',')
        responses = await multi({item: self.userInfo(item, key_type, bb_type, info_type) for item in items.split(',')})
        return [response.body for response in responses.values()]

    async def uids(self):
        key_type, uids = self.get_query_argument("key_type", "uid"), []
        if key_type == "uid":
            uids = self.get_query_argument("keys", "").split(',')
        else:
            uids = [info["uid"] for info in await self.info()]
        return uids


class PingHandler(RequestHandler):
    async def get(self):
        # status = 200
        # if hasattr(self.application, "check_lock") and hasattr(self.application, "check_status"):
        #     async with self.application.check_lock.acquire():
        #         status = self.application.check_status
        self.write("")

    async def post(self):
        # status = 200
        # if hasattr(self.application, "check_lock") and hasattr(self.application, "check_status"):
        #     async with self.application.check_lock.acquire():
        #         status = self.application.check_status
        self.write("")


class HostNameHandler(RequestHandler):
    async def get(self):
        self.write(gethostname())

    async def post(self):
        self.write(gethostname())


class UserInfoHandler(RequestHandler, UserInfo):
    async def get(self):
        self.write(json_encode(await self.info()))


class UsersInfoHandler(RequestHandler, UsersInfo):
    async def get(self):
        self.write(json_encode(await self.info()))


class UserLogoutHandler(RequestHandler, UserInfo):
    async def get(self):
        params = {
            "comment":                      self.get_query_argument("comment", ""),
            "admin_name":                   self.get_query_argument("admin", ""),
            "global_logout":                self.get_query_argument("global_logout", 1),
            "is_changing_required":         self.get_query_argument("is_changing_required", 'yes'),
            "max_change_frequency_in_days": self.get_query_argument("max_change_frequency_in_days", 4)
        }
        self.write(await logoutUser(self.uid(), params))


class UsersLogoutHandler(RequestHandler, UsersInfo):
    async def get(self):
        params = {
            "comment":                      self.get_query_argument("comment", ""),
            "admin_name":                   self.get_query_argument("admin", ""),
            "global_logout":                self.get_query_argument("global_logout", 1),
            "is_changing_required":         self.get_query_argument("is_changing_required", 'yes'),
            "max_change_frequency_in_days": self.get_query_argument("max_change_frequency_in_days", 4)
        }
        responses = await multi({uid: logoutUser(uid, params) for uid in self.uids()})
        self.write(json_encode({k: response.body for (k, response) in responses.items()}))


class UserBanHandler(RequestHandler, UserInfo):
    async def get(self):
        params = {
            "comment":    self.get_query_argument("comment", ""),
            "admin_name": self.get_query_argument("admin", "")
        }
        self.write(await banUser(self.uid(), params))


class UsersBanHandler(RequestHandler, UsersInfo):
    async def get(self):
        params = {
            "comment":    self.get_query_argument("comment", ""),
            "admin_name": self.get_query_argument("admin", "")
        }
        responses = await multi({uid: banUser(uid, params) for uid in self.uids()})
        self.write(json_encode({k: response.body for (k, response) in responses.items()}))


class UserChangeOptionsHandler(RequestHandler, UserInfo):
    async def get(self):
        params = {
            "comment":    self.get_query_argument("comment", ""),
            "admin_name": self.get_query_argument("admin", "")
        }
        opts = self.get_query_argument("option", "").split(',')
        self.write(await changeUserOptions(self.uid(), params, opts))


class UsersChangeOptionsHandler(RequestHandler, UsersInfo):
    async def get(self):
        params = {
            "comment":    self.get_query_argument("comment", ""),
            "admin_name": self.get_query_argument("admin", "")
        }
        opts = self.get_query_argument("option", "").split(',')
        responses = await multi({uid: changeUserOptions(uid, params, opts) for uid in self.uids()})
        self.write(json_encode({k: response.body for (k, response) in responses.items()}))


class UserChangePasswordOptionsHandler(RequestHandler, UserInfo):
    async def get(self):
        params = {
            "comment":              self.get_query_argument("comment", ""),
            "admin_name":           self.get_query_argument("admin", ""),
            "is_changing_required": self.get_query_argument("is_changing_required", 'yes')
        }
        try:
            params["max_change_frequency_in_days"] = self.get_query_argument("max_change_frequency_in_days")
        except MissingArgumentError:
            pass
        self.write(await changeUserPasswordOptions(self.uid(), params))


class UsersChangePasswordOptionsHandler(RequestHandler, UsersInfo):
    async def get(self):
        params = {
            "comment":              self.get_query_argument("comment", ""),
            "admin_name":           self.get_query_argument("admin", ""),
            "is_changing_required": self.get_query_argument("is_changing_required", 'yes')
        }
        try:
            params["max_change_frequency_in_days"] = self.get_query_argument("max_change_frequency_in_days")
        except MissingArgumentError:
            pass
        responses = await multi({uid: changeUserPasswordOptions(uid, params) for uid in self.uids()})
        self.write(json_encode({k: response.body for (k, response) in responses.items()}))


class UserSetKarmaHandler(RequestHandler, UserInfo):
    async def get(self):
        params = {
            "comment":    self.get_query_argument("comment", ""),
            "admin_name": self.get_query_argument("admin", "")
        }
        karma_info = self.get_query_argument("karma", "0")
        m, karma, karma_strict = re.match(r'(\d+)(s)?', karma_info), None, None
        if m:
            karma = int(m.group(1))
            karma_strict = True if m.group(2) else False
        if karma is None:
            self.send_error(400, reason="Bad Request")
        else:
            self.write(await setKarma(self.uid(), karma, params, karma_strict))


class UsersSetKarmaHandler(RequestHandler, UsersInfo):
    async def get(self):
        params = {
            "comment":    self.get_query_argument("comment", ""),
            "admin_name": self.get_query_argument("admin", "")
        }
        karma_info = self.get_query_argument("karma", "0")
        m, karma, karma_strict = re.match(r'(\d+)(s)?', karma_info), None, None
        if m:
            karma = int(m.group(1))
            karma_strict = True if m.group(2) else False
        if karma is None:
            self.send_error(400, reason="Bad Request")
        else:
            responses = await multi({uid: setKarma(uid, karma, params, karma_strict) for uid in self.uids()})
            self.write(json_encode({k: response.body for (k, response) in responses.items()}))


def sigtermHandler(sig, frame):
    global MAINPROCESS_PID

    trace("SIGTERM received: %d" % os.getpid())
    if os.getpid() == MAINPROCESS_PID:
        trace("SIGTERM received")
        child_pids = os.popen("ps xao ppid,pid,cmd | grep -v ps | awk '$1 == %d { print $2 }'" % os.getpid()).read()
        for pid in child_pids.split():
            os.kill(int(pid), SIGTERM)
    if options.debug:
        os._exit(0)
    else:
        sys.exit()


def main():
    global MAINPROCESS_PID

    parse_command_line()

    if options.help:
        print_help()
        sys.exit()

    # if not options.debug:
    #     daemonize.becomeDaemon()

    MAINPROCESS_PID = os.getpid()
    signal(SIGTERM, sigtermHandler)
    signal(SIGHUP, SIG_IGN)
    # writePid(options.pidfile, MAINPROCESS_PID)

    AsyncHTTPClient.configure("tornado.curl_httpclient.CurlAsyncHTTPClient")
    configureLoggers()
    loadTVMConfig()

    application = Application(
        [
            (r"/ping",                          PingHandler),
            (r"/hostname",                      HostNameHandler),
            (r"/user_info",                     UserInfoHandler),
            (r"/users_info",                    UsersInfoHandler),
            (r"/user_logout",                   UserLogoutHandler),
            (r"/users_logout",                  UsersLogoutHandler),
            (r"/user_ban",                      UserBanHandler),
            (r"/users_ban",                     UsersBanHandler),
            (r"/user_change_options",           UserChangeOptionsHandler),
            (r"/users_change_options",          UsersChangeOptionsHandler),
            (r"/user_change_password_options",  UserChangePasswordOptionsHandler),
            (r"/users_change_password_options", UsersChangePasswordOptionsHandler),
            (r"/user_set_karma",                UserSetKarmaHandler),
            (r"/users_set_karma",               UsersSetKarmaHandler),
        ],
        autoreload=not options.debug,
        debug=options.debug,
    )

    server = HTTPServer(application)
    server.bind(options.port)
    server.start(options.processes)

    trace("http_qproxy v.2.0 with task %s on port %s started" % (task_id(), options.port))

    IOLoop.current().start()


if __name__ == "__main__":
    main()
