#!/usr/bin/python3
# encoding: utf-8
# kate: space-indent on; indent-width 4; replace-tabs on;
#

from json import loads
from typing import Optional
from tornado.gen import sleep
from tornado.escape import json_decode
from tornado.httpclient import AsyncHTTPClient, HTTPRequest, HTTPClientError
from mail.so.spamstop.users.admkarma_app.config import CFG, TVM
from mail.so.spamstop.users.admkarma_app.logger import error


def isCorp(uid) -> bool:
    return str(uid).startswith("113000") and len(str(uid)) == 16


async def requestService(url, headers={}, data=None, retDataType='str') -> tuple:
    client, request, response, body, code = AsyncHTTPClient(), None, None, "", 200
    if data:
        request = HTTPRequest(url, method="POST", headers=headers, body=data, validate_cert=False)
    else:
        request = HTTPRequest(url, method="GET", headers=headers, validate_cert=False)
    try:
        response = await client.fetch(request)
        body, code = response.body, response.code
    except HTTPClientError as e:
        body, code = e.response.body, e.code
    except Exception as e:
        error("requestService failed: %s" % e)
        body, code = "", 500
    if retDataType == "dict" or retDataType == "json":
        return json_decode(body).decode('utf-8') if body else {}, code
    else:
        return body.decode('utf-8') if body else "", code


def loadTVMConfig() -> None:
    f, tvm_port = None, TVM["API"]["port"]
    try:
        with open("/etc/tvmtool/tvmtool.conf") as f:
            try:
                tvm_config = loads(f.read().strip())
                if 'port' in tvm_config:
                    tvm_port = int(tvm_config['port'])
            except Exception as e:
                error("Failed to parse tvmtool's config! %s" % str(e))
        TVM['API']['URL'] = TVM['API']['url'].format(TVM['API']['host'], tvm_port)
        with open("/var/lib/tvmtool/local.auth") as f:
            TVM['API']['Auth'] = f.read().strip()
    except Exception as e:
        error("loadTVMConfig failed: %s" % str(e))


async def getTVM2ticket(service: str) -> Optional[str]:
    resp, svc = {}, service.lower()
    try:
        resp, code = await requestService('%s/tickets?dsts=%s&src=%s' % (TVM['API']['URL'], svc, TVM[service]["client_id"]), headers={"Authorization": TVM['API']['Auth']}, retDataType="str")
        if code != 200:
            await error("getTVM2ticket HTTPS error for service '%s' (status=%s): %s" % (service, code, str(resp)), False)
        try:
            if not resp or len(resp) < 1:
                await error("getTVM2ticket error answer for service '%s' (code=%s): %s" % (service, code, str(resp)), False)
            else:
                resp = loads(resp)
        except Exception as e:
            await error("getTVM2ticket decoding of JSON from TVM ticket query's answer failed for service '%s': %s. Answer: %s." % (service, str(e), str(resp)))
    except Exception as e:
        await error("getTVM2ticket HTTPS request failed for service '%s': %s." % (service, str(e)))
    return resp[svc]['ticket'] if isinstance(resp, dict) and svc in resp and resp[svc] else None


async def requestServiceByTVM(url, service, headers={}, data=None, retDataType='str', retryCnt=CFG["retry_cnt"]) -> tuple:
    resp, code = '', 200
    ticket = await getTVM2ticket(service)
    # if service.startswith("BB"):
    #     print("Ticket: %s" % ticket)
    if ticket:
        if not headers:
            headers = {}
        headers['X-Ya-Service-Ticket'] = ticket
        for i in range(retryCnt):
            try:
                resp, code = await requestService(url, headers=headers, data=data, retDataType=retDataType)
            except Exception as e:
                await error('requestServiceByTVM failed: %s. TicketAnswer: %s' % (str(e), str(ticket)))
            if code == 200:
                break
            else:
                await error('requestServiceByTVM failed (attempt #%s from %s, status=%s). Response: "%s"' % (i + 1, retryCnt, code, str(resp)), False)
                await sleep(2 * i + 1)
                continue
    return resp, code
