#!/usr/bin/python3
# encoding: utf-8
# kate: space-indent on; indent-width 4; replace-tabs on;
#
import os, os.path, sys, re, time, json, argparse, asyncio, aiohttp
from concurrent.futures import ThreadPoolExecutor, as_completed
from urllib.request import urlopen, Request
from traceback import format_exception

global errorlog_fh, log_fh

RETRY_CNT = 3
HOME_DIR = "/root"
LOG_DIR =  os.environ['PWD'] if 'PWD' in os.environ else "/var/log/so-logs"
ERRORLOG = "%s/admsearch.log" % LOG_DIR
LOG="-"
TVM = {
    "API": {
        "host":           "localhost",
        "port":           1111,
        "url":            "http://{0}:{1}/tvm",
        "URL":            "http://localhost:1111/tvm"
    },
    "BB": {
        "host":           "blackbox-mail.yandex.net",
        "url":            "https://{0}/blackbox?",
        "URL":            "https://blackbox-mail.yandex.net/blackbox?",
        "client_id":      2016577,
        "prod_client_id": 222,
        "timeout":        3.0
    },
    "BBcorp": {
        "host":           "blackbox.yandex-team.ru",
        "url":            "https://{0}/blackbox?",
        "URL":            "https://blackbox.yandex-team.ru/blackbox?",
        "client_id":      2016577,
        "prod_client_id": 223,
        "timeout":        3.0
    },
    "mail-search": {
        "host":           "meta-qloud.mail.yandex.net",
        "url":            "https://{0}/{1}?",
        #"URL":            "https://meta-qloud.mail.yandex.net/{0}?",
        "URL":            "http://meta.mail.yandex.net/{0}?",
        "client_id":      2000031,
        "prod_client_id": 2000499,
        "timeout":        3.0
    },
    "mail-search-corp": {
        "host":           "metacorp.mail.yandex.net",
        "url":            "https://{0}/{1}?",
        "URL":            "https://metacorp.mail.yandex.net/{0}?",
        "client_id":      2000031,
        "prod_client_id": 2000500,
        "timeout":        3.0
    }
}

def get_traceback():
    exc_type, exc_value, exc_traceback = sys.exc_info()
    tb = ''
    for step in format_exception(exc_type, exc_value, exc_traceback):
        try:
            tb += "\t" + step.strip() + "\n"
        except:
            pass
    return tb


def writelog(msg, isTB=False, addTS=True, log_fh=sys.stderr):
    if not msg:
        return
    try:
        tb = "\n"
        if isTB:
            tb += get_traceback()
        if addTS:
            msg = time.strftime("[%Y-%m-%d %H:%M:%S]: ") + msg
        if log_fh and not log_fh.closed:
            os.write(log_fh.fileno(), bytes(msg + tb, "utf8"))
        else:
             print(msg + tb, file=sys.stderr)
    except Exception as e:
        print("Writelog error: %s.%s" % (str(e), get_traceback()), file=sys.stderr)
        print(time.strftime("[%Y-%m-%d %H:%M:%S]: ") + msg + tb, file=sys.stderr)
        sys.stderr.flush()


async def errorlog(msg, isTB=True):
    global errorlog_fh
    writelog(msg, isTB, True, log_fh=errorlog_fh)


async def log(msg):
    global log_fh
    writelog(msg, False, False, log_fh)


async def requestService(url, headers={}, data=None, retDataType='str'):
    resp, code = "", 0
    client = aiohttp.ClientSession()
    if data:
        async with client.post(url, data=data, headers=headers) as r:
            if retDataType == "dict" or retDataType == "json":
                resp = await r.json()
            else:
                resp = await r.text()
            code = r.status
    else:
        async with client.get(url, headers=headers) as r:
            if retDataType == "dict" or retDataType == "json":
                resp = await r.json()
            else:
                resp = await r.text()
            code = r.status
    client.close()
    return resp, code


def loadTVMConfig():
    f, CURDIR, tvm_port = None, HOME_DIR, TVM["API"]["port"]
    try:
        with open("/etc/tvmtool/tvmtool.conf") as f:
            try:
                tvm_config = json.loads(f.read().strip())
                if 'port' in tvm_config:
                    tvm_port = int(tvm_config['port'])
            except Exception as e:
                errorlog("Failed to parse tvmtool's config!")
        TVM['API']['URL'] = TVM['API']['url'].format(TVM['API']['host'], tvm_port)
        with open("/var/lib/tvmtool/local.auth") as f:
            TVM['API']['Auth'] = f.read().strip()
    except Exception as e:
        writelog("loadTVMConfig failed: %s" % str(e), True)


async def getTVM2ticket(service):
    resp, svc = {}, service.lower()
    try:
        resp, code = await requestService('%s/tickets?dsts=%s&src=%s' % (TVM['API']['URL'], svc, TVM[service]["client_id"]), headers={"Authorization": TVM['API']['Auth']}, retDataType="str")
        if code != 200:
            await errorlog("getTVM2ticket HTTPS error for service '%s' (status=%s): %s" % (service, code, str(resp)), False)
        try:
            if not resp or len(resp) < 1:
                await errorlog("getTVM2ticket error answer for service '%s' (code=%s): %s" % (service, code, str(resp)), False)
            else:
                resp = json.loads(resp)
        except Exception as e:
            await errorlog("getTVM2ticket decoding of JSON from TVM ticket query's answer failed for service '%s': %s. Answer: %s." % (service, str(e), str(resp)))
    except Exception as e:
        await errorlog("getTVM2ticket HTTPS request failed for service '%s': %s." % (service, str(e)))
    return resp[svc]['ticket'] if isinstance(resp, dict) and svc in resp and resp[svc] else None


async def requestServiceByTVM(url, service, headers={}, data=None):
    resp, code = '', 200
    ticket = await getTVM2ticket(service)
    if ticket:
        if not headers:
            headers = {}
        headers['X-Ya-Service-Ticket'] = ticket
        for i in range(RETRY_CNT):
            try:
                resp, code = await requestService(url, headers=headers, data=data)
            except Exception as e:
                await errorlog('requestServiceByTVM failed: %s. TicketAnswer: %s' % (str(e), str(ticket)))
            if code == 200:
                break
            else:
                await errorlog('requestServiceByTVM failed (attempt #%s from %s, status=%s). Response: "%s"' % (i + 1, RETRY_CNT, code, resp), False)
                continue
    return resp, code


async def requestBlackBox(params, mail_type='big', tvm=True):
    url = params if not isinstance(params, dict) else ('&'.join(map(lambda k, v: "{0}={1}".format(k, v), params.iteritems())) if isinstance(params, dict) else '')
    bb_type = "BBcorp" if mail_type == "corp" else "BB"
    url, resp, code = TVM[bb_type]['URL'] + url, '', 0
    if tvm:
        resp = (await requestServiceByTVM(url, bb_type))[0]
    else:
        for i in range(RETRY_CNT):
            try:
                resp, code = await requestService(url)
            except Exception as e:
                await errorlog('BlackBox request failed: %s' % str(e))
            if code == 200:
                break
            else:
                await errorlog('BlackBox request failed (attempt #%s from %s, status=%s). Response: "%s"' % (i + 1, RETRY_CNT, code, resp), False)
                continue
    return resp


async def requestFilterSearch(params, fs_request='filter_search', mail_type='big', tvm=True):
    url = params if not isinstance(params, dict) else ('&'.join(map(lambda k, v: "{0}={1}".format(k, v), params.iteritems())) if isinstance(params, dict) else '')
    fs_svc_type = "mail-search-corp" if mail_type == "corp" else "mail-search"
    url, resp, code = TVM[fs_svc_type]['URL'].format(fs_request) + url, '', 0
    if tvm:
        resp = (await requestServiceByTVM(url, fs_svc_type))[0]
    else:
        for i in range(RETRY_CNT):
            try:
                resp, code = await requestService(url)
            except Exception as e:
                await errorlog('FilterSearch request failed: %s' % str(e))
            if code == 200:
                break
            else:
                await errorlog('FilterSearch request failed (attempt #%s from %s, status=%s). Response: "%s"' % (i + 1, RETRY_CNT, code, resp), False)
                continue
    return resp


async def getUserInfo(filter_str, mail_type='big'):
    params = "method=userinfo&%s&userip=127.0.0.1&dbfields=subscription.suid.2,accounts.login.uid,subscription.suid.2" % filter_str   # "&format=json" - for JSON answer
    info = await requestBlackBox(params, mail_type)
    m = re.search(r'<uid\b[^<>]*?>(\d+)<\/uid>', info)
    uid, login, suid = m.group(1) if m and m.group(1) else '', '', ''
    m = re.search(r'<dbfield id="accounts.login.uid">(.*?)<\/dbfield>', info)
    if m and m.group(1):
        login = m.group(1)
    else:
        m = re.search(r'<login>([^<]+)', info)
        if m and m.group(1):
            login = m.group(1)
        m = re.search(r'^<dbfield id="subscription.suid.2">(\d+)</dbfield>', info, re.M)
        if m and m.group(1):
            suid = m.group(1)
    return uid, login, suid


async def printUserInfo(uid='', login='', suid=''):
    s = ''
    if uid:
        s += "UID=%s" % uid
    if login:
        s += "\tLOGIN=%s" % login
    if suid:
        s += "\tSUID=%s" % suid
    await log(s)


def runAsyncFunc(f, *args, **kwargs):
    loop = asyncio.get_event_loop()
    result = loop.run_until_complete(f(*args, **kwargs))
    loop.close()
    return result


async def getObjectsInfo(params):
    filter_str = "{0}={1}".format(params["user_info_type"], params[params["user_info_type"]])
    params["uid"], params["login"], params["suid"] = await getUserInfo(filter_str, params["mail_type"])
    if params["uid"]:
        #await printUserInfo(params["uid"], params["login"], params["suid"])
        uri = "uid=%s&caller=msearch&mdb=pg" % params["uid"]
        if params['action'] == "filter_search":
            uri += "&order=default"
            if params["mail_type"] == "corp":
                uri += "&folder_set=default"
            for objects_type in ["mids", "lids", "fids"]:
                if objects_type in params:
                    for obj_id in params[objects_type]:
                        uri += "&{0}={1}".format(objects_type, obj_id)
            for p in ["unread", "only_useful", "full_folders_and_labels"]:
                if params[p]:
                    uri += "&{}=1".format(p)
        objectsInfo = await requestFilterSearch(uri, params['action'], params["mail_type"])
        await log(objectsInfo)
    else:
        await errorlog("Unable to determine user(s) for accomplishing search for case %s = %s!" % (str(params["key_type"]), str(item)), False)


if __name__ == "__main__":
    epilog = "" if 'USER' in os.environ and os.environ['USER'] == "root" else "WARNING: This script must be run under root!"
    parser = argparse.ArgumentParser(epilog=epilog)
    parser.add_argument('-a', '--action',          type=str,             help="Action which will be done on input users data. Possible values are one and only one of the next: "
                        "1) 'filter_search' (default) - get mail messages by given filter; "
                        "2) 'labels' - get mail labels for given user; "
                        "3) 'folders' - get mail folders for given user. Actions will be done in the specified sequence.")
    parser.add_argument('-e', '--error_log',  default='-',    type=str,  help="Error log's file path or '-' (or 'stderr') for output to pipe")
    parser.add_argument('-m', '--mail_type',  default='big',  type=str,  help="Preffered Mail type: big (by default) or corp")
    parser.add_argument('-o', '--output',     default='-',    type=str,  help="Output file path or '-' (or 'stdout') for output to pipe (by default)")
    parser.add_argument('-u', '--uid',        default='',     type=str,  help="Input user's ID if this search is for the only user")
    parser.add_argument('-l', '--login',      default='',     type=str,  help="Input user's login if this search is for the only user")
    parser.add_argument('-s', '--suid',       default='',     type=str,  help="Input user's suid if this search is for the only user")
    parser.add_argument('--mids',             default='',     type=str,  help="List of message IDs separated by comma")
    parser.add_argument('--lids',             default='',     type=str,  help="List of label IDs separated by comma")
    parser.add_argument('--fids',             default='',     type=str,  help="List of folder IDs separated by comma")
    parser.add_argument('--non_full_folders_and_labels', action='store_false', help="Wether to print not all info about folders and labels?")
    parser.add_argument('--only_useful',      action='store_true',       help="Wether to print messages from user's folders and Inbox only?")
    parser.add_argument('--unread',           action='store_true',       help="Wether to print unread messages only?")
    args = parser.parse_known_args()[0]
    if len(sys.argv) < 2:
        parser.print_help()
        sys.exit(0)
    loadTVMConfig()
    params = {
        "action":                  args.action if args.action else "filter_search",
        "admin":                   os.environ['USER'] if 'USER' in os.environ else 'root',
        "mail_type":               args.mail_type,
        "user_info_type":          "uid",
        "full_folders_and_labels": 1 if args.non_full_folders_and_labels else 0,
        "only_useful":             1 if args.only_useful else 0,
        "unread":                  1 if args.unread else 0
    }
    if args.error_log:
        ERRORLOG = args.error_log
    errorlog_fh = sys.stderr if ERRORLOG == '-' or ERRORLOG == 'stderr' else open(ERRORLOG, "a+t")
    if args.output:
        LOG = args.output
    log_fh = sys.stdout if LOG == '-' or LOG == 'stdout' else open(LOG, "a+t")
    if args.mids:
        params['key_type'], params["mids"] = "mid", args.mids.split(',')
    elif args.lids:
        params['key_type'], params["lids"] = "lid", args.lids.split(',')
    elif args.fids:
        params['key_type'], params["fids"] = "fid", args.fids.split(',')
    if args.uid:
        params["user_info_type"], params["uid"] = "uid", args.uid
    elif args.login:
        params["user_info_type"], params["login"] = "login", args.login
    elif args.suid:
        params["user_info_type"], params["suid"] = "suid", args.suid
    else:
        print("User undetermined!")
        sys.exit(1)
    try:
        runAsyncFunc(getObjectsInfo, params)
    except Exception as e:
        writelog("Exception: %s" % str(e), True, log_fh=errorlog_fh)

