#!/usr/bin/python3
# encoding: utf-8
# kate: space-indent on; indent-width 4; replace-tabs on;
#
import os, os.path, sys, re, time, json, argparse, asyncio, aiohttp, imaplib
from concurrent.futures import ThreadPoolExecutor, as_completed
from urllib.request import urlopen, Request
from urllib.parse import quote
from collections import OrderedDict
from traceback import format_exception

global errorlog_fh, log_fh

RETRY_CNT = 3
WORKERS_COUNT = 10
READERS_COUNT = 1
STIDS_BATCH_SIZE = 40
HOME_DIR = "/root"
LOG_DIR =  os.environ['PWD'] if 'PWD' in os.environ else "/var/log/so-logs"
ERRORLOG = "%s/admtransfer_spec_msgs.log" % LOG_DIR
LOG="-"
COWORKERS_SELECTION_URL = "http://complaint.so.yandex.net:8700/coworkers-selection?uid="
LUCENE_URL = "http://new-msearch-proxy.mail.yandex.net:8051/sequential/search?service=change_log&get=stid&prefix="
SPEC_LABEL_PREFIX = "_so_"
IMAP = {
    'host':   'imap.yandex-team.ru',
    'port':   993,
    'folder': 'so-spam-report'
}
ROBOT = {
    "login":    "robot-markspam",
    "password": ""
}
TVM = {
    "API": {
        "host":           "localhost",
        "port":           1111,
        "url":            "http://{0}:{1}/tvm",
        "URL":            "http://localhost:1111/tvm"
    },
    "BB": {
        "host":           "blackbox-mail.yandex.net",
        "url":            "https://{0}/blackbox?",
        "URL":            "https://blackbox-mail.yandex.net/blackbox?",
        "client_id":      2016577,
        "prod_client_id": 222,
        "timeout":        3.0
    },
    "BBcorp": {
        "host":           "blackbox.yandex-team.ru",
        "url":            "https://{0}/blackbox?",
        "URL":            "https://blackbox.yandex-team.ru/blackbox?",
        "client_id":      2016577,
        "prod_client_id": 223,
        "timeout":        3.0
    },
    "mail-search": {
        "host":           "meta-qloud.mail.yandex.net",
        "url":            "https://{0}/{1}?",
        #"URL":            "https://meta-qloud.mail.yandex.net/{0}?",
        "URL":            "http://meta.mail.yandex.net/{0}?",
        "client_id":      2000031,
        "prod_client_id": 2000499,
        "timeout":        3.0
    },
    "mail-search-corp": {
        "host":           "metacorp.mail.yandex.net",
        "url":            "https://{0}/{1}?",
        "URL":            "https://metacorp.mail.yandex.net/{0}?",
        "client_id":      2000031,
        "prod_client_id": 2000500,
        "timeout":        3.0
    },
    "MulcaGate": {
        "host":           "storage.mail.yandex.net",
        "url":            "https://{0}:4443/gate/get/",
        "URL":            "https://storage.mail.yandex.net:4443/gate/get/",
        "client_id":      2010368,
        "prod_client_id": 2000273,
        "timeout":        3.0,
        "retry_cnt":      3
    }
}

def get_traceback():
    exc_type, exc_value, exc_traceback = sys.exc_info()
    tb = ''
    for step in format_exception(exc_type, exc_value, exc_traceback):
        try:
            tb += "\t" + step.strip() + "\n"
        except:
            pass
    return tb


def writelog(msg, isTB=False, addTS=True, log_fh=sys.stderr):
    if not msg:
        return
    try:
        tb = "\n"
        if isTB:
            tb += get_traceback()
        if addTS:
            msg = time.strftime("[%Y-%m-%d %H:%M:%S]: ") + msg
        if log_fh and not log_fh.closed:
            os.write(log_fh.fileno(), bytes(msg + tb, "utf8"))
        else:
             print(msg + tb, file=sys.stderr)
    except Exception as e:
        print("Writelog error: %s.%s" % (str(e), get_traceback()), file=sys.stderr)
        print(time.strftime("[%Y-%m-%d %H:%M:%S]: ") + msg + tb, file=sys.stderr)
        sys.stderr.flush()


async def errorlog(msg, isTB=True):
    global errorlog_fh
    writelog(msg, isTB, True, log_fh=errorlog_fh)


async def log(msg):
    global log_fh
    writelog(msg, False, False, log_fh)


async def logMsg(stid, msg):
    CURDIR = os.environ["PWD"] if "PWD" in os.environ else "/root"
    file_name = stid.replace(':', '_')
    f = open("{}/{}".format(CURDIR, file_name), "a+t")
    writelog(msg, isTB=False, addTS=False, log_fh=f)
    f.close()


async def printUserInfo(uid='', login='', suid=''):
    s = ''
    if uid:
        s += "UID=%s" % uid
    if login:
        s += "\tLOGIN=%s" % login
    if suid:
        s += "\tSUID=%s" % suid
    await log(s)


def loadRobotCredentials():
    HOME_DIR = os.environ["HOME"] if "HOME" in os.environ else "/root"
    filename = '{}/.robot.{}'.format(HOME_DIR, ROBOT["login"])
    if not os.path.exists(filename) and "PWD" in os.environ and os.environ["PWD"]:
        HOME_DIR = os.environ["PWD"]
        filename = '{0}/.robot.{1}'.format(os.environ["PWD"], ROBOT["login"])
    if os.path.exists(filename):
        try:
            f = open(filename)
            ROBOT['password'] = f.read().strip()
            f.close()
        except Exception as e:
            writelog("loadRobotCredentials: Unable to read secret for robot '%s': %s" % (ROBOT["login"], str(e)), isTB=True)
    else:
        writelog("loadRobotCredentials: Unable to locate secret for robot '%s'" % ROBOT["login"], isTB=False)

async def requestService(url, headers={}, data=None, retDataType='str'):
    resp, code = "", 0
    client = aiohttp.ClientSession()
    if data:
        async with client.post(url, data=data, headers=headers) as r:
            if retDataType == "dict" or retDataType == "json":
                resp = await r.json()
            else:
                resp = await r.text()
            code = r.status
    else:
        async with client.get(url, headers=headers) as r:
            if retDataType == "dict" or retDataType == "json":
                resp = await r.json()
            else:
                resp = await r.text()
            code = r.status
    client.close()
    return resp, code


def loadTVMConfig():
    f, CURDIR, tvm_port = None, HOME_DIR, TVM["API"]["port"]
    try:
        with open("/etc/tvmtool/tvmtool.conf") as f:
            try:
                tvm_config = json.loads(f.read().strip())
                if 'port' in tvm_config:
                    tvm_port = int(tvm_config['port'])
            except Exception as e:
                errorlog("Failed to parse tvmtool's config!")
        TVM['API']['URL'] = TVM['API']['url'].format(TVM['API']['host'], tvm_port)
        with open("/var/lib/tvmtool/local.auth") as f:
            TVM['API']['Auth'] = f.read().strip()
    except Exception as e:
        writelog("loadTVMConfig failed: %s" % str(e), True)


async def getTVM2ticket(service):
    resp, svc = {}, service.lower()
    try:
        resp, code = await requestService('%s/tickets?dsts=%s&src=%s' % (TVM['API']['URL'], svc, TVM[service]["client_id"]), headers={"Authorization": TVM['API']['Auth']}, retDataType="str")
        if code != 200:
            await errorlog("getTVM2ticket HTTPS error for service '%s' (status=%s): %s" % (service, code, str(resp)), False)
        try:
            if not resp or len(resp) < 1:
                await errorlog("getTVM2ticket error answer for service '%s' (code=%s): %s" % (service, code, str(resp)), False)
            else:
                resp = json.loads(resp)
        except Exception as e:
            await errorlog("getTVM2ticket decoding of JSON from TVM ticket query's answer failed for service '%s': %s. Answer: %s." % (service, str(e), str(resp)))
    except Exception as e:
        await errorlog("getTVM2ticket HTTPS request failed for service '%s': %s." % (service, str(e)))
    return resp[svc]['ticket'] if isinstance(resp, dict) and svc in resp and resp[svc] else None


async def requestServiceByTVM(url, service, headers={}, data=None):
    resp, code = '', 200
    ticket = await getTVM2ticket(service)
    if ticket:
        if not headers:
            headers = {}
        headers['X-Ya-Service-Ticket'] = ticket
        for i in range(RETRY_CNT):
            try:
                resp, code = await requestService(url, headers=headers, data=data)
            except Exception as e:
                await errorlog('requestServiceByTVM failed: %s. TicketAnswer: %s' % (str(e), str(ticket)))
            if code == 200:
                break
            else:
                await errorlog('requestServiceByTVM failed (attempt #%s from %s, status=%s). Response: "%s"' % (i + 1, RETRY_CNT, code, resp), False)
                continue
    return resp, code


async def requestBlackBox(params, mail_type='big', tvm=True):
    url = params if not isinstance(params, dict) else ('&'.join(map(lambda it: "{0}={1}".format(it[0], it[1]), params.items())) if isinstance(params, dict) else '')
    bb_type = "BBcorp" if mail_type == "corp" else "BB"
    url, resp, code = TVM[bb_type]['URL'] + url, '', 0
    if tvm:
        resp = (await requestServiceByTVM(url, bb_type))[0]
    else:
        for i in range(RETRY_CNT):
            try:
                resp, code = await requestService(url)
            except Exception as e:
                await errorlog('BlackBox request failed: %s' % str(e))
            if code == 200:
                break
            else:
                await errorlog('BlackBox request failed (attempt #%s from %s, status=%s). Response: "%s"' % (i + 1, RETRY_CNT, code, resp), False)
                continue
    return resp


async def getUserInfo(filter_str, mail_type='big'):
    params = "method=userinfo&%s&userip=127.0.0.1&dbfields=subscription.suid.2,accounts.login.uid,subscription.suid.2" % filter_str   # "&format=json" - for JSON answer
    info = await requestBlackBox(params, mail_type)
    m = re.search(r'<uid\b[^<>]*?>(\d+)<\/uid>', info)
    uid, login, suid = str(m.group(1) if m and m.group(1) else ''), '', ''
    m = re.search(r'<dbfield id="accounts.login.uid">(.*?)<\/dbfield>', info)
    if m and m.group(1):
        login = m.group(1)
    else:
        m = re.search(r'<login>([^<]+)', info)
        if m and m.group(1):
            login = m.group(1)
        m = re.search(r'^<dbfield id="subscription.suid.2">(\d+)</dbfield>', info, re.M)
        if m and m.group(1):
            suid = m.group(1)
    return uid, login, suid


async def requestFilterSearch(params, fs_request='filter_search', mail_type='big', tvm=True):
    url = params if not isinstance(params, dict) else ('&'.join(map(lambda it: "{0}={1}".format(it[0], it[1]), params.items())) if isinstance(params, dict) else '')
    fs_svc_type = "mail-search-corp" if mail_type == "corp" else "mail-search"
    url, resp, code = TVM[fs_svc_type]['URL'].format(fs_request) + url, '', 0
    if tvm:
        resp, code = await requestServiceByTVM(url, fs_svc_type)
    else:
        for i in range(RETRY_CNT):
            try:
                resp, code = await requestService(url)
            except Exception as e:
                await errorlog('FilterSearch request failed: %s' % str(e))
            if code == 200:
                break
            else:
                await errorlog('FilterSearch request failed (attempt #%s from %s, status=%s). Response: "%s"' % (i + 1, RETRY_CNT, code, resp), False)
                continue
    return resp


async def getObjectsInfo(userInfo, params, action="filter_search"):
    if params["key_type"] != "uid" and ("uid" not in userInfo or not userInfo["uid"]):
        filter_str, info = "{0}={1}".format(params["key_type"], params[params["key_type"]]), {}
        userInfo["uid"], userInfo["login"], userInfo["suid"] = await getUserInfo(filter_str, "big")
        if not userInfo["uid"]:
            userInfo["uid"], userInfo["login"], userInfo["suid"] = await getUserInfo(filter_str, "corp")
    if userInfo["uid"]:
        uri, mail_type = "uid={}&caller=msearch&mdb=pg".format(userInfo["uid"]), "corp" if params['mail_type'] == 'auto' and userInfo["uid"].startswith("112000") else params['mail_type']
        if action == "filter_search":
            uri += "&order=default"
            if mail_type == "corp":
                uri += "&folder_set=default"
            for objects_type in ["mids", "lids", "fids"]:
                if objects_type in params:
                    for obj_id in params[objects_type]:
                        uri += "&{0}={1}".format(objects_type, obj_id)
            for p in ["unread", "only_useful", "full_folders_and_labels"]:
                if params[p]:
                    uri += "&{}=1".format(p)
        objectsInfo = await requestFilterSearch(uri, action, mail_type)
        try:
            info = json.loads(objectsInfo)
        except Exception as e:
            await errorlog('Failed to parse JSON in answer of FilterSearch: %s' % str(e))
    else:
        await errorlog("Unable to determine user(s) for accomplishing search for case %s = %s!" % (str(params["key_type"]), str(item)), False)
    return info


async def requestLucene(uid, params):
    url = ""
    if isinstance(params, dict):
        for p in params:
            if isinstance(params[p], list):
                for v in params[p]:
                    url += "{}{}={}".format('&' if url else '', p, quote(v))
            else:
                url += "{}{}={}".format('&' if url else '', p, quote(params[p]))
    else:
        url = str(params)
    url, resp, code, data = LUCENE_URL + uid + "&" + url, '', 0, {}
    for i in range(RETRY_CNT):
        try:
            resp, code = await requestService(url)
        except Exception as e:
            await errorlog('Request #%s to Lucene "%s" failed: %s' % (i + 1, url, str(e)))
            continue
        if code == 200:
            #await log("Lucene request #{} '{}' result (code={}): {}".format(i + 1, url, code, resp))
            try:
                data = json.loads(resp)
            except Exception as e:
                await errorlog('Failed to parse JSON in answer of Lucene: %s' % str(e))
            break
        else:
            await errorlog('Request #%s to Lucene failed (status=%s). Response: "%s"' % (i + 1, code, resp), False)
    return data


async def requestCoworkersSelectionHandle(uid, action, params, dryRun=False):
    url = ""
    if isinstance(params, dict):
        for p in params:
            if isinstance(params[p], list):
                for v in params[p]:
                    url += "{}{}={}".format('&' if url else '', p, quote(v))
            else:
                url += "{}{}={}".format('&' if url else '', p, quote(params[p]))
    else:
        url = str(params)
    url, resp, code, data = COWORKERS_SELECTION_URL + uid + '&action=' + action + "&" + url, '', 0, {}
    try:
        if dryRun:
            await log("Request to CoworkersSelectionHandle: %s" % url)
        else:
            resp, code = await requestService(url)
            await log("CoworkersSelectionHandle request '%s' (code=%s). Response: '%s'." % (url, code, str(resp)))
    except Exception as e:
        await errorlog('Request to CoworkersSelectionHandle "%s" failed: %s' % (url, str(e)))
    if code != 200 and not dryRun:
        await errorlog('Request to CoworkersSelectionHandle failed (status=%s). Response: "%s"' % (code, resp), False)
    return (code == 200 or dryRun)


async def requestMulcaGate(stid):
    url = "%s?raw=&service=so" % stid
    resp, code = await requestServiceByTVM(TVM['MulcaGate']['URL'] + url, 'MulcaGate')
    return resp, code


def addHeaderSTID(msg, stid):
    return re.sub(r'^(.+?)(\r?\n\r?\n)(.+)$', r'\1\nX-Yandex-STID: {}\2\3'.format(stid), msg, 1, re.S)


async def addMsgToIMAPFolder(msg, acttime=None, folder_name=IMAP["folder"], user=ROBOT['login'], password=ROBOT['password']):
    status = ""
    if not acttime:
        acttime = int(time.time())
    if not user or not password:
        s = "Invalid IMAP user's credentials: user=%s, passwd=%s." % (user, password)
        await log(s)
        status += ('\n' if status else '') + s
    try:
        imap = imaplib.IMAP4_SSL(IMAP['host'], IMAP['port'])
        loginInfo = imap.login(user, password)
        if loginInfo[0] == "NO":
            s = "Authenticating for user '%s' IMAP failed: %s." % (user, str(loginInfo[1]))
            await log(s)
            status += ('\n' if status else '') + s
        else:
            folderInfo = imap.select(folder_name)
            if folderInfo[0] == "NO" and folderInfo[1][0].find("No such folder") > -1:
                createInfo = imap.create(folder_name)
                if createInfo[0] == "NO":
                    s = "Creating IMAP folder '%s' result: %s." % (folder_name, str(createInfo[1]))
                    status += ('\n' if status else '') + s
                    await log(s)
            appendInfo = imap.append(folder_name, '', imaplib.Time2Internaldate(acttime), msg.encode('utf-8'))
            if appendInfo[0] == "NO":
                s = "Appending of message to IMAP folder '%s' failed: %s." % (folder_name, str(appendInfo[1]))
                await log(s)
                status += ('\n' if status else '') + s
            imap.logout()
    except Exception as e:
        s = "Adding message to corp IMAP folder '%s' failed: %s." % (folder_name, str(e))
        await errorlog(s)
        status += ('\n' if status else '') + s
    return status


async def addMessagesToIMAPFolder(stids, folder):
    messages = []
    for stid in stids:
        if not stid:
            continue
        content, code = await requestMulcaGate(stid)
        if code == 200:
            if re.match(r'<\?xml ', content):
                content = re.sub(r'(?s)^<\?xml.*?</message>\s+(.*)$', r'\1', content)
            content = addHeaderSTID(content, stid)
            timeout = 10
            status = await addMsgToIMAPFolder(content, folder_name=folder, user=ROBOT["login"], password=ROBOT["password"])
            if status:
                await log("addMessagesToIMAPFolder: Adding messages failed: {}".format(status))
                if status.find("UNAVAILABLE") > -1:
                    await logMsg(stid, content)
        else:
            await log("addMessagesToIMAPFolder: Unable get message for stid={} from MulcaGate".format(stid))


async def processItem(params, q):
    while True:
        item = await q.get()
        if item is None:
            break
        userInfo= {"uid": "", "login": "", "suid": ""}
        if params["key_type"] == "uid":
            userInfo["uid"], userInfo["login"], userInfo["suid"] = str(item).strip(), '', ''
        else:
            sid = '&sid=2' if params["key_type"] == "suid" else ""
            filter_str = "{0}={1}{2}".format(str(params["key_type"]), str(item).strip(), sid)
            if params['mail_type'] == 'auto' or params['mail_type'] == 'big':
                userInfo["uid"], userInfo["login"], userInfo["suid"] = await getUserInfo(filter_str, 'BB')    # tuple (uid, login, suid)
            if params['mail_type'] == 'auto' and not userInfo["uid"] or params['mail_type'] == 'corp':
                userInfo["uid"], userInfo["login"], userInfo["suid"] = await getUserInfo(filter_str, 'BBcorp')
        if userInfo["uid"]:
            specLabels = {}
            user = "LOGIN={}".format(userInfo["login"]) if userInfo["login"] else "UID={}".format(userInfo["uid"])
            await log("Processing UID={}\n".format(userInfo["uid"]))
            labelsInfo = await getObjectsInfo(userInfo, params, "labels")
            if "labels" in labelsInfo and len(labelsInfo["labels"]) > 0:
                for lid, labelInfo in labelsInfo["labels"].items():
                    if labelInfo["name"].startswith(SPEC_LABEL_PREFIX) and (not params["label"] or params["label"] and params["label"] == labelInfo["name"]):
                        specLabels[lid] = labelInfo["name"]
                await log("%s specLabels: %s" % (user, str(specLabels)))
                for lid in specLabels:
                    stids, n = [], 0
                    if params["info_source"] == "lucene":
                        url = OrderedDict({"text": "hid:0 AND lids:%s" % lid})
                        url["postfilter"] = []
                        if "mintime" in params:
                            url["postfilter"].append("received_date >= %s" % params["mintime"])
                        if "maxtime" in params:
                            url["postfilter"].append("received_date <= %s" % params["maxtime"])
                        msgsInfo = await requestLucene(userInfo["uid"], url)
                        if "hitsArray" in msgsInfo and len(msgsInfo["hitsArray"]) > 0:
                            n = len(msgsInfo["hitsArray"])
                            for msgInfo in msgsInfo["hitsArray"]:
                                stids.append(msgInfo["stid"])
                    elif params["info_source"] == "filter_search":
                        pass
                    if n > 0:
                        await log("{} label={} msgsCnt={}".format(user, specLabels[lid], n))
                        url, offset = "label={}".format(specLabels[lid]), 0
                        while offset < n:
                            await addMessagesToIMAPFolder(stids[offset:offset+STIDS_BATCH_SIZE], specLabels[lid][4:])
                            #for stid in stids[offset:offset+STIDS_BATCH_SIZE]:
                            #    url += "&stid={}".format(stid)
                            #await requestCoworkersSelectionHandle(userInfo["uid"], params["action"], url, params['dry_run'])
                            offset += STIDS_BATCH_SIZE
                            await log("{} label={}: processed {} messages from {}".format(user, specLabels[lid], min(offset, n), n))
                            #await asyncio.sleep(1)
            else:
                await log("User {} has no spec labels!".format(user))
            await log("Processing UID={} DONE\n".format(userInfo["uid"]))
        else:
            await errorlog("Unable find UID for user with %s=%s" % (str(params["key_type"]), str(item)), False)
        await log("Processing item: {} DONE.".format(str(item)))


async def readItem(f, q):
    for row in f:
        await q.put(row.strip())
    for i in range(WORKERS_COUNT):
        await q.put(None)


def processItems(params, f):
    loop = asyncio.get_event_loop()
    q = asyncio.Queue(100)
    workers = [processItem(params, q) for _ in range(WORKERS_COUNT)]
    readers = [readItem(f, q) for _ in range(READERS_COUNT)]
    loop.run_until_complete(asyncio.gather(*workers, *readers))


if __name__ == "__main__":
    epilog = "" if 'USER' in os.environ and os.environ['USER'] == "root" else "WARNING: This script must be run under root!"
    parser = argparse.ArgumentParser(epilog=epilog)
    parser.add_argument('-a', '--action',      default='add',    type=str, help="Action which will be done on input users selected messages. Possible values are one and only one of the next: "
                        "1) 'add' (default) - add selected messages to robot's mailbox; 2) 'del' - remove selected messages from robot's mailbox.")
    parser.add_argument('-e', '--error_log',   default='-',      type=str, help="Error log's file path or '-' (or 'stderr') for output to pipe")
    parser.add_argument('-i', '--input',       default='-',      type=str, help="Input source data: file or '-' (or 'stdin') for input from pipe (by default)")
    parser.add_argument('-l', '--label',       default='',       type=str, help="Process this spec label only: only that labeled messages will be precessed")
    parser.add_argument('-m', '--mail_type',   default='auto',   type=str, help="Preffered Mail type: auto (by default) (depends from uid's format), big or corp")
    parser.add_argument('-o', '--output',      default='-',      type=str, help="Output file path or '-' (or 'stdout') for output to pipe (by default)")
    parser.add_argument('-s', '--info_source', default="lucene", type=str, help="Source for mail info: 1) 'lucene' (default) or 'filter_search'")
    parser.add_argument('-t', '--input_type',  default="uids",   type=str, help="Input data type: 'uids' (default) or 'logins' or 'suids'")
    parser.add_argument('--mids',              default='',       type=str, help="List of message IDs separated by comma")
    parser.add_argument('--lids',              default='',       type=str, help="List of label IDs separated by comma")
    parser.add_argument('--fids',              default='',       type=str, help="List of folder IDs separated by comma")
    parser.add_argument('--mintime',           default='',       type=str, help="Low boundary for received_time of user messages")
    parser.add_argument('--maxtime',           default='',       type=str, help="High boundary for received_time of user messages")
    parser.add_argument('--dry_run',           action='store_true',        help="Simulation of request to CoworkersSelectionHandle")
    args = parser.parse_known_args()[0]
    if len(sys.argv) < 2:
        parser.print_help()
        sys.exit(0)
    loadTVMConfig()
    loadRobotCredentials()
    input_file = '-'
    params = {
        "action":                  'lids_' + (args.action if args.action else "add"),
        "info_source":             args.info_source,
        "admin":                   os.environ['USER'] if 'USER' in os.environ else 'root',
        "mail_type":               args.mail_type,
        "key_type":                "uid",
        "full_folders_and_labels": 1,
        "only_useful":             1,
        "dry_run":                 1 if args.dry_run else 0,
        "label":                   args.label
    }
    if args.error_log:
        ERRORLOG = args.error_log
    errorlog_fh = sys.stderr if ERRORLOG == '-' or ERRORLOG == 'stderr' else open(ERRORLOG, "a+t")
    if args.output:
        LOG = args.output
    log_fh = sys.stdout if LOG == '-' or LOG == 'stdout' else open(LOG, "a+t")
    if args.mids:
        params["mids"] = args.mids.split(',')
    if args.lids:
        params["lids"] = args.lids.split(',')
    if args.fids:
        params["fids"] = args.fids.split(',')
    if args.mintime:
        params["mintime"] = args.mintime
    if args.maxtime:
        params["maxtime"] = args.maxtime
    if args.input_type == "logins":
        params['key_type'], input_file = "login", args.input
    elif args.input_type == "suids":
        params['key_type'], input_file = "suid", args.input
    elif args.input_type == "uids":
        params['key_type'], input_file = "uid", args.input
    else:
        print("Input users undetermined!")
        sys.exit(1)
    if input_file != '-' and input_file != 'stdin' and not os.path.exists(input_file):
        print("Unable find file '%s'" % input_file)
        sys.exit(1)
    else:
        try:
            f = sys.stdin if input_file == '-' or input_file == 'stdin' else open(input_file, 'rt')
            processItems(params, f)
            f.close()
        except Exception as e:
            writelog("Exception: %s" % str(e), True, log_fh=errorlog_fh)

