#!/usr/bin/perl
## -*- Encoding:utf-8; Mode:perl; -*-
# kate: space-indent on; indent-width 4; replace-tabs on;
#
##---------------------------------------------------------------------------##
##  File:
##	$Id: MHUtils.pm,v 1.1 2014/02/20 11:08:38 klimiky Exp $
##  Author:
##      Earl Hood       mhonarc@mhonarc.org
##  Description:
##      Utility routines for SO::Mail::Message
##---------------------------------------------------------------------------##
##    MHonArc -- Internet mail-to-HTML converter
##    Copyright (C) 1995-1999	Earl Hood, mhonarc@mhonarc.org
##
##    This program is free software; you can redistribute it and/or modify
##    it under the terms of the GNU General Public License as published by
##    the Free Software Foundation; either version 2 of the License, or
##    (at your option) any later version.
##
##    This program is distributed in the hope that it will be useful,
##    but WITHOUT ANY WARRANTY; without even the implied warranty of
##    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##    GNU General Public License for more details.
##
##    You should have received a copy of the GNU General Public License
##    along with this program; if not, write to the Free Software
##    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
##    02111-1307, USA
##---------------------------------------------------------------------------##

package SO::Mail::Message;

use SO::Mail::Message::RFC822;
use Time::Local;
use Carp;
$Carp::Verbose = 1;

our $HTMLSpecials = '"&<>';
our %HTMLSpecials = (
  '"'	=> '&quot;',
  '&'	=> '&amp;',
  '<'	=> '&lt;',
  '>'	=> '&gt;',
  # '@'	=> '&#x40;',  # XXX: Screws up ISO-2022-JP conversion
);

##---------------------------------------------------------------------------
##	Initialize ReadMail.pm variables
##
sub mhinit_readmail_vars {
    $ReadMail::DEBUG = $DEBUG;

    ##	Default decoders
    %ReadMail::MIMEDecoders = (
	'7bit'             => 'as-is',
	'8bit'             => 'as-is',
	'binary'           => 'as-is',
	'base64'           => 'base64::b64decode',
	'quoted-printable' => 'QuotedPrintable::qprdecode',
	'x-uuencode'       => 'base64::uudecode',
	'x-uue'            => 'base64::uudecode',
	'uuencode'         => 'base64::uudecode',
    );
    %ReadMail::MIMEDecodersSrc = (
	'base64'           => 'Base64.pm',
	'quoted-printable' => 'QPrint.pm',
	'x-uuencode'       => 'Base64.pm',
	'x-uue'            => 'Base64.pm',
	'uuencode'         => 'Base64.pm',
    );
    $IsDefault{'MIMEDECODERS'} = 1;

    ##	Default filters
    %ReadMail::MIMEFilters = (
	# Content-type			Filter
	#-----------------------------------------------------------------
	'application/ms-tnef',		'm2h_null::filter',
	'application/octet-stream',	'm2h_external::filter',
	'application/x-patch',		'm2h_text_plain::filter',
	'message/delivery-status',  	'm2h_text_plain::filter',
	'message/external-body',	'm2h_msg_extbody::filter',
	'message/partial',   		'm2h_text_plain::filter',
	'text/enriched',    		'm2h_text_enriched::filter',
	'text/html',			'm2h_text_html::filter',
	'text/plain',			'm2h_text_plain::filter',
	'text/richtext',    		'm2h_text_enriched::filter',
	'text/tab-separated-values',	'm2h_text_tsv::filter',
	'text/x-html',			'm2h_text_html::filter',

	'application/*',		'm2h_external::filter',
	'audio/*',			'm2h_external::filter',
	'chemical/*',  			'm2h_external::filter',
	'image/*',  			'm2h_external::filter',
	'model/*',  			'm2h_external::filter',
	'text/*',   			'm2h_text_plain::filter',
	'video/*',  			'm2h_external::filter',

	'x-sun-attachment',		'm2h_text_plain::filter',
    );

    %ReadMail::MIMEFiltersSrc = (
	# Content-type			Filter
	#-----------------------------------------------------------------
	'application/ms-tnef',		'MHNull.pm',
	'application/octet-stream',	'MHExternal.pm',
	'application/x-patch',		'MHTxtPlain.pm',
	'message/delivery-status',  	'MHTxtPlain.pm',
	'message/external-body',	'MHMsgExtBody.pm',
	'message/partial',   		'MHTxtPlain.pm',
	'text/enriched',    		'MHTxtEnrich.pm',
	'text/html',			'MHTxtHtml.pm',
	'text/plain',			'MHTxtPlain.pm',
	'text/richtext',    		'MHTxtEnrich.pm',
	'text/tab-separated-values',	'MHTxtTsv.pm',
	'text/x-html',			'MHTxtHtml.pm',

	'application/*',		'MHExternal.pm',
	'audio/*',			'MHExternal.pm',
	'chemical/*',  			'MHExternal.pm',
	'image/*',  			'MHExternal.pm',
	'model/*',  			'MHExternal.pm',
	'text/*',   			'MHTxtPlain.pm',
	'video/*',  			'MHExternal.pm',

	'x-sun-attachment',		'MHTxtPlain.pm',
    );
    $IsDefault{'MIMEFILTERS'} = 1;

    ##  Default filter arguments
    %ReadMail::MIMEFiltersArgs = (
	# Content-type			Arguments
	#-----------------------------------------------------------------
	'm2h_external::filter',		'inline',
    );
    $IsDefault{'MIMEARGS'} = 1;

    ##  Charset filters
    %ReadMail::MIMECharSetConverters = (
	# Character set		Converter Function
	#-----------------------------------------------------------------
	'plain',     		'SO::Mail::Message::htmlize',
	'us-ascii',   		'SO::Mail::Message::htmlize',
	'iso-8859-1',   	'SO::Mail::Message::CharEnt::str2sgml',
	'iso-8859-2',   	'SO::Mail::Message::CharEnt::str2sgml',
	'iso-8859-3',   	'SO::Mail::Message::CharEnt::str2sgml',
	'iso-8859-4',   	'SO::Mail::Message::CharEnt::str2sgml',
	'iso-8859-5',   	'SO::Mail::Message::CharEnt::str2sgml',
	'iso-8859-6',   	'SO::Mail::Message::CharEnt::str2sgml',
	'iso-8859-7',   	'SO::Mail::Message::CharEnt::str2sgml',
	'iso-8859-8',   	'SO::Mail::Message::CharEnt::str2sgml',
	'iso-8859-9',   	'SO::Mail::Message::CharEnt::str2sgml',
	'iso-8859-10',   	'SO::Mail::Message::CharEnt::str2sgml',
	'iso-8859-11',   	'SO::Mail::Message::CharEnt::str2sgml',
	'iso-8859-13',   	'SO::Mail::Message::CharEnt::str2sgml',
	'iso-8859-14',   	'SO::Mail::Message::CharEnt::str2sgml',
	'iso-8859-15',   	'SO::Mail::Message::CharEnt::str2sgml',
	'iso-8859-16',   	'SO::Mail::Message::CharEnt::str2sgml',
	'iso-2022-jp',   	'SO::Mail::Message::CharEnt::str2sgml',
	'iso-2022-kr',    	'SO::Mail::Message::CharEnt::str2sgml',
	'euc-jp',    		'SO::Mail::Message::CharEnt::str2sgml',
	'utf-8',    		'SO::Mail::Message::CharEnt::str2sgml',
	'cp866',    		'SO::Mail::Message::CharEnt::str2sgml',
	'cp932',    		'SO::Mail::Message::CharEnt::str2sgml',
	'cp936',    		'SO::Mail::Message::CharEnt::str2sgml',
	'cp949',    		'SO::Mail::Message::CharEnt::str2sgml',
	'cp950',    		'SO::Mail::Message::CharEnt::str2sgml',
	'cp1250',   		'SO::Mail::Message::CharEnt::str2sgml',
	'cp1251',   		'SO::Mail::Message::CharEnt::str2sgml',
	'cp1252',   		'SO::Mail::Message::CharEnt::str2sgml',
	'cp1253',   		'SO::Mail::Message::CharEnt::str2sgml',
	'cp1254',   		'SO::Mail::Message::CharEnt::str2sgml',
	'cp1255',   		'SO::Mail::Message::CharEnt::str2sgml',
	'cp1256',   		'SO::Mail::Message::CharEnt::str2sgml',
	'cp1257',   		'SO::Mail::Message::CharEnt::str2sgml',
	'cp1258',   		'SO::Mail::Message::CharEnt::str2sgml',
	'koi-0',            	'SO::Mail::Message::CharEnt::str2sgml',
	'koi-7',            	'SO::Mail::Message::CharEnt::str2sgml',
	'koi8-a',            	'SO::Mail::Message::CharEnt::str2sgml',
	'koi8-b',            	'SO::Mail::Message::CharEnt::str2sgml',
	'koi8-e',            	'SO::Mail::Message::CharEnt::str2sgml',
	'koi8-f',            	'SO::Mail::Message::CharEnt::str2sgml',
	'koi8-r',            	'SO::Mail::Message::CharEnt::str2sgml',
	'koi8-u',            	'SO::Mail::Message::CharEnt::str2sgml',
	'gost19768-87',         'SO::Mail::Message::CharEnt::str2sgml',
	'viscii',            	'SO::Mail::Message::CharEnt::str2sgml',
	'big5-eten',		'SO::Mail::Message::CharEnt::str2sgml',
	'big5-hkscs',		'SO::Mail::Message::CharEnt::str2sgml',
	'gb2312',    		'SO::Mail::Message::CharEnt::str2sgml',
	'macarabic',		'SO::Mail::Message::CharEnt::str2sgml',
	'maccentraleurroman',	'SO::Mail::Message::CharEnt::str2sgml',
	'maccroatian',		'SO::Mail::Message::CharEnt::str2sgml',
	'maccyrillic',		'SO::Mail::Message::CharEnt::str2sgml',
	'macgreek',		'SO::Mail::Message::CharEnt::str2sgml',
	'machebrew',		'SO::Mail::Message::CharEnt::str2sgml',
	'macicelandic',		'SO::Mail::Message::CharEnt::str2sgml',
	'macromanian',		'SO::Mail::Message::CharEnt::str2sgml',
	'macroman',		'SO::Mail::Message::CharEnt::str2sgml',
	'macthai',		'SO::Mail::Message::CharEnt::str2sgml',
	'macturkish',		'SO::Mail::Message::CharEnt::str2sgml',
	'hp-roman8',		'SO::Mail::Message::CharEnt::str2sgml',
	'default',     		'-ignore-',
    );
    %ReadMail::MIMECharSetConvertersSrc = (
	# Character set		Converter Function
	#-----------------------------------------------------------------
	'plain',     		undef,
	'us-ascii',   		undef,
	'iso-8859-1',   	'Message/CharEnt.pm',
	'iso-8859-2',   	'Message/CharEnt.pm',
	'iso-8859-3',   	'Message/CharEnt.pm',
	'iso-8859-4',   	'Message/CharEnt.pm',
	'iso-8859-5',   	'Message/CharEnt.pm',
	'iso-8859-6',   	'Message/CharEnt.pm',
	'iso-8859-7',   	'Message/CharEnt.pm',
	'iso-8859-8',   	'Message/CharEnt.pm',
	'iso-8859-9',   	'Message/CharEnt.pm',
	'iso-8859-10',   	'Message/CharEnt.pm',
	'iso-8859-11',   	'Message/CharEnt.pm',
	'iso-8859-13',   	'Message/CharEnt.pm',
	'iso-8859-14',   	'Message/CharEnt.pm',
	'iso-8859-15',   	'Message/CharEnt.pm',
	'iso-8859-16',   	'Message/CharEnt.pm',
	'iso-2022-jp',   	'Message/CharEnt.pm',
	'iso-2022-kr',    	'Message/CharEnt.pm',
	'euc-jp',    		'Message/CharEnt.pm',
	'utf-8',    		'Message/CharEnt.pm',
	'cp866',    		'Message/CharEnt.pm',
	'cp932',    		'Message/CharEnt.pm',
	'cp936',    		'Message/CharEnt.pm',
	'cp949',    		'Message/CharEnt.pm',
	'cp950',    		'Message/CharEnt.pm',
	'cp1250',   		'Message/CharEnt.pm',
	'cp1251',   		'Message/CharEnt.pm',
	'cp1252',   		'Message/CharEnt.pm',
	'cp1253',   		'Message/CharEnt.pm',
	'cp1254',   		'Message/CharEnt.pm',
	'cp1255',   		'Message/CharEnt.pm',
	'cp1256',   		'Message/CharEnt.pm',
	'cp1257',   		'Message/CharEnt.pm',
	'cp1258',   		'Message/CharEnt.pm',
	'koi-0',            	'Message/CharEnt.pm',
	'koi-7',            	'Message/CharEnt.pm',
	'koi8-a',            	'Message/CharEnt.pm',
	'koi8-b',            	'Message/CharEnt.pm',
	'koi8-e',            	'Message/CharEnt.pm',
	'koi8-f',            	'Message/CharEnt.pm',
	'koi8-r',            	'Message/CharEnt.pm',
	'koi8-u',            	'Message/CharEnt.pm',
	'gost19768-87',         'Message/CharEnt.pm',
	'viscii',            	'Message/CharEnt.pm',
	'big5-eten',		'Message/CharEnt.pm',
	'big5-hkscs',		'Message/CharEnt.pm',
	'gb2312',    		'Message/CharEnt.pm',
	'macarabic',		'Message/CharEnt.pm',
	'maccentraleurroman',	'Message/CharEnt.pm',
	'maccroatian',		'Message/CharEnt.pm',
	'maccyrillic',		'Message/CharEnt.pm',
	'macgreek',		'Message/CharEnt.pm',
	'machebrew',		'Message/CharEnt.pm',
	'macicelandic',		'Message/CharEnt.pm',
	'macromanian',		'Message/CharEnt.pm',
	'macroman',		'Message/CharEnt.pm',
	'macthai',		'Message/CharEnt.pm',
	'macturkish',		'Message/CharEnt.pm',
	'hp-roman8',		'Message/CharEnt.pm',
	'default',     		undef,
    );
    $IsDefault{'CHARSETCONVERTERS'} = 1;

    ##	Default charset aliases
    ReadMail::MAILset_charset_aliases({ 
	'us-ascii'     => [ 'ascii',
			    'ansi_x3.4-1968',
			    'iso646', 'iso646-us', 'iso646.irv:1991',
			    'cp367', 'ibm367',
			    'csascii',
			    'iso-ir-6',
			    'us' ],
	'iso-8859-1'   => [ 'latin1', 'l1',
			    'iso_8859_1', 'iso_8859-1:1987',
			    'iso8859-1', 'iso8859_1', '8859-1', '8859_1',
			    'cp819', 'ibm819',
			    'x-mac-latin1',
			    'iso-ir-100' ],
	'iso-8859-2'   => [ 'latin2', 'l2',
			    'iso_8859_2', 'iso_8859-2:1987',
			    'iso8859-2', 'iso8859_2', '8859-2', '8859_2',
			    'iso-ir-101' ],
	'iso-8859-3'   => [ 'latin3', 'l3',
			    'iso_8859_3', 'iso_8859-3:1988',
			    'iso8859-3', 'iso8859_3', '8859-3', '8859_3',
			    'iso-ir-109' ],
	'iso-8859-4'   => [ 'latin4', 'l4',
			    'iso_8859_4', 'iso_8859-4:1988',
			    'iso8859-4', 'iso8859_4', '8859-4', '8859_4',
			    'iso-ir-110' ],
	'iso-8859-5'   => [ 'iso_8859-5:1988',
			    'cyrillic',
			    'iso-ir-144' ],
	'iso-8859-6'   => [ 'iso_8859-6:1987',
			    'arabic',
			    'asmo-708',
			    'ecma-114',
			    'iso-ir-127' ],
	'iso-8859-7'   => [ 'iso_8859-7:1987',
			    'greek', 'greek8',
			    'ecma-118',
			    'elot_928',
			    'iso-ir-126' ],
	'iso-8859-8'   => [ 'iso-8859-8-i', 'iso_8859-8:1988',
			    'hebrew',
			    'iso-ir-138' ],
	'iso-8859-9'   => [ 'latin5', 'l5',
			    'iso_8859_9', 'iso-8859_9:1989',
			    'iso8859-9', 'iso8859_9', '8859-9', '8859_9',
			    'iso-ir-148' ],
	'iso-8859-10'  => [ 'latin6', 'l6',
			    'iso_8859_10', 'iso_8859-10:1993',
			    'iso8859-10', 'iso8859_10',
			    '8859-10', '8859_10',
			    'iso-ir-157' ],
	'iso-8859-13'  => [ 'latin7' ,'l7' ],
	'iso-8859-14'  => [ 'latin8' ,'l8' ],
	'iso-8859-15'  => [ 'latin9', 'latin0', 'l9', 'l0',
			    'iso_8859_15',
			    'iso8859-15', 'iso8859_15',
			    '8859-15', '8859_15' ],
	'iso-2022-jp'  => [ 'iso-2022-jp-1' ],
	'utf-8'        => [ 'utf8' ],
	'cp932'        => [ 'shiftjis', 'shift_jis', 'shift-jis',
			    'x-sjis',
			    'ms_kanji',
			    'csshiftjis' ],
	'cp936'        => [ 'gbk',
			    'ms936',
			    'windows-936' ],
	'cp949'        => [ 'euc-kr',
			    'ks_c_5601-1987', 'ks_c_5601-1989',
			    'ksc_5601',
			    'iso-ir-149',
			    'windows-949', 'ms949',
			    'korean' ],
	'cp950'        => [ 'windows-950' ],
	'cp1250'       => [ 'windows-1250' ],
	'cp1251'       => [ 'windows-1251' ],
	'cp1252'       => [ 'windows-1252' ],
	'cp1253'       => [ 'windows-1253' ],
	'cp1254'       => [ 'windows-1254' ],
	'cp1255'       => [ 'windows-1255' ],
	'cp1256'       => [ 'windows-1256' ],
	'cp1257'       => [ 'windows-1257' ],
	'cp1258'       => [ 'windows-1258' ],
	'koi-0'          => [ 'gost-13052' ],
	'koi8-e'         => [ 'iso-ir-111',
			      'ecma-113:1986' ],
	'koi8-r'         => [ 'cp878' ],
	'gost-19768-87'  => [ 'ecma-cyrillic',
			      'ecma-113', 'ecma-113:1988' ],
	'big5-eten'      => [ 'big5', 'csbig5',
			      'tcs-big5', 'tcsbig5' ],
	'big5-hkscs'     => [ 'big5hkscs', 'big5hk',
			      'hkscs-big5', 'hk-big5' ],
	'gb2312'	     => [ 'gb_2312-80', 'csgb2312', 'hz-gb-2312',
			          'iso-ir-58',
			          'euc-cn',
			          'chinese',
			          'csiso58gb231280' ],
	'macarabic'	     => [ 'apple-arabic',
				  'x-mac-arabic' ],
	'maccentraleurroman' => [ 'apple-centeuro',
				  'x-mac-centraleurroman' ],
	'maccroatian'        => [ 'apple-croatian',
				  'x-mac-croatian' ],
	'maccyrillic'        => [ 'apple-cyrillic',
				  'x-mac-cyrillic' ],
	'macgreek'	     => [ 'apple-greek',
				  'x-mac-greek' ],
	'machebrew'	     => [ 'apple-hebrew',
				  'x-mac-hebrew' ],
	'macicelandic'       => [ 'apple-iceland',
				  'x-mac-icelandic' ],
	'macromanian'        => [ 'apple-romanian',
				  'x-mac-romanian' ],
	'macroman'	     => [ 'apple-roman',
				  'mac', 'macintosh',
				  'x-mac-roman' ],
	'macthai'	     => [ 'apple-thai',
				  'x-mac-thai' ],
	'macturkish'         => [ 'apple-turkish',
				  'x-mac-turkish' ],
    });
    $IsDefault{'CHARSETALIASES'} = 1;

    ##  Content-Types to exclude:
    ##    Nothing is excluded by default.
    %ReadMail::MIMEExcs = ( );
    $IsDefault{'MIMEEXCS'} = 1;

    ##  Content-Types to only include:
    ##    Blank by default: include everything
    %ReadMail::MIMEIncs = ( );
    $IsDefault{'MIMEIncs'} = 1;

    ##  Content-type multipart/alternative preferences
    ##    Note: The variable is not a ReadMail package variable, but it
    ##	    is used to set ReadMail package properties.
    @MIMEAltPrefs = ( );
    $IsDefault{'MIMEALTPREFS'} = 1;

    ##	Text encoding
    $ReadMail::TextEncode = undef;
    $IsDefault{'TEXTENCODE'} = 1;

    $ReadMail::FormatHeaderFunc = \&SO::Mail::Message::htmlize_header;
    $MHeadCnvFunc = \&ReadMail::MAILdecode_1522_str;
}

##---------------------------------------------------------------------------
##	Remove duplicates in an array.
##	Returns list with duplicates removed.
##
sub remove_dups {
    my $a = shift;
    return ()  unless scalar(@$a);
    my %dup = ();
    grep(!$dup{$_}++, @$a);
}

##---------------------------------------------------------------------------
##	"Entify" special characters

sub htmlize {			# Older name
    return ''  unless scalar(@_) && defined($_[0]);
    my $txt   = shift;
    my $txt_r = ref($txt) ? $txt : \$txt;
    $$txt_r =~ s/([$HTMLSpecials])/$HTMLSpecials{$1}/go;
    $$txt_r;
}

sub entify {			# Alternate name
    return ''  unless scalar(@_) && defined($_[0]);
    my $txt   = shift;
    my $txt_r = ref($txt) ? $txt : \$txt;
    $$txt_r =~ s/([$HTMLSpecials])/$HTMLSpecials{$1}/go;
    $$txt_r;
}

##	commentize entifies certain characters to avoid problems when a
##	string will be included in a comment declaration

sub commentize {
    my($txt) = $_[0];
    $txt =~ s/([\-&])/'&#'.unpack('C',$1).';'/ge;
    $txt;
}

sub uncommentize {
    my($txt) = $_[0];
    $txt =~ s/&#(\d+);/pack("C",$1)/ge;
    $txt;
}

##---------------------------------------------------------------------------
##	Copy a file.
##
sub cp {
    my($src, $dst) = @_;
    open(SRC, $src) || die("ERROR: Unable to open $src\n");
    open(DST, "> $dst") || die("ERROR: Unable to create $dst\n");
    print DST <SRC>;
    close(SRC);
    close(DST);
}

##---------------------------------------------------------------------------
##	Translate html string back to regular string
##
sub dehtmlize {
    my $str   = shift;
    my $str_r = ref($str) ? $str : \$str;
    $$str_r =~ s/\&lt;/</g;
    $$str_r =~ s/\&gt;/>/g;
    $$str_r =~ s/\&#[xX]0*40;/@/g;
    $$str_r =~ s/\&#64;/@/g;
    $$str_r =~ s/\&quot;/"/g;
    $$str_r =~ s/\&amp;/\&/g;
    $$str_r;
}

##---------------------------------------------------------------------------
##	Escape special characters in string for URL use.
##
sub urlize {
    my($url) = shift || '';
    my $url_r = ref($url) ? $url : \$url;
    $$url_r =~ s/([^\w\.\-:])/sprintf("%%%02X",unpack("C",$1))/ge;
    $$url_r;
}

sub urlize_path {
    my($url) = shift || "";
    my $url_r = ref($url) ? $url : \$url;
    $$url_r =~ s/([^\w\.\-:\/])/sprintf("%%%02X",unpack("C",$1))/ge;
    $$url_r;
}

##---------------------------------------------------------------------------##
##	Perform a "modified" rot13 on a string.  This version includes
##	the '@' character so addresses can be munged a little better.
##
sub mrot13 {
    my $str	= shift;
    $str =~ tr/@A-Z[a-z/N-Z[@A-Mn-za-m/;
    $str;
}

## RFC 2369 header fields to check for URLs
%HFieldsList = (
    'list-archive'  	=> 1,
    'list-help'  	=> 1,
    'list-owner'  	=> 1,
    'list-post'  	=> 1,
    'list-subscribe'  	=> 1,
    'list-unsubscribe' 	=> 1,
);

## Do not apply ADDRESSMODIFYCODE headerfields
%HFieldsAsIsList = (
    %HFieldsList,
    'content-disposition' => 1,
    'content-id'          => 1,
    'content-type'        => 1,
    'message-id'          => 1,
    'references'          => 1,
    'in-reply-to'         => 1,
);

## Header fields that contain addresses
%HFieldsAddr = (
    'apparently-from'	=> 1,
    'apparently-to'	=> 1,
    'bcc'		=> 1,
    'cc'		=> 1,
    'dcc'		=> 1,
    'from'		=> 1,
    'mail-followup-to'	=> 1,
    'mail-reply-to'	=> 1,
    'notify-bcc'	=> 1,
    'notify-cc' 	=> 1,
    'notify-to' 	=> 1,
    'original-bcc'	=> 1,
    'original-cc'	=> 1,
    'original-from'	=> 1,
    'original-sender'	=> 1,
    'original-to'	=> 1,
    'reply-to'		=> 1,
    'resent-bcc'	=> 1,
    'resent-cc'		=> 1,
    'resent-from'	=> 1,
    'resent-sender'	=> 1,
    'resent-to'		=> 1,
    'return-path'	=> 1,
    'sender'		=> 1,
    'to'		=> 1,
    'x-envelope'	=> 1,
);

##---------------------------------------------------------------------------
##    Convert message header string to HTML encoded in
##    $ReadMail::TextEncode encoding.
##
sub htmlize_enc_head {
    my($cnvfunc, $charset) = ReadMail::MAILload_charset_converter($ReadMail::TextEncode);
    return htmlize($_[0]) if ($cnvfunc eq '-decode-' || $cnvfunc eq '-ignore-');
    return &$cnvfunc($_[0], $charset);
}

##---------------------------------------------------------------------------
##    Clip text to specified length.
##
sub clip_text {
    my $str      = \shift;  # Prevent unnecessary copy.
    my $len      = shift;   # Clip length
    my $is_html  = shift;   # If entity references should be considered
    my $has_tags = shift;   # If html tags should be stripped

    if (!$is_html) {
      return substr($$str, 0, $len);
    }

    my $text = '';
    my $subtext = '';
    my $html_len = length($$str);
    my($pos, $sublen, $real_len, $semi);
    my $er_len = 0;
    
    for ( $pos=0, $sublen=$len; $pos < $html_len; ) {
	$subtext = substr($$str, $pos, $sublen);
	$pos += $sublen;

	# strip tags
	if ($has_tags) {
	    # Strip full tags
	    $subtext =~ s/<[^>]*>//g;
	    # Check if clipped part of a tag
	    if ($subtext =~ s/<[^>]*\Z//) {
		my $gt = index($$str, '>', $pos);
		$pos = ($gt < 0) ? $html_len : ($gt+1);
	    }
	}

	# check for clipped entity reference
	if (($pos < $html_len) && ($subtext =~ /\&[^;]*\Z/)) {
	    my $semi = index($$str, ';', $pos);
	    if ($semi < 0) {
		# malformed entity reference
		$subtext .= substr($$str, $pos);
		$pos = $html_len;
	    } else {
		$subtext .= substr($$str, $pos, $semi-$pos+1);
		$pos = $semi+1;
	    }
	}

	# compute entity reference lengths to determine "real" character
	# count and not raw character count.
	while ($subtext =~ /(\&[^;]+);/g) {
	    $er_len += length($1);
	}

	$text .= $subtext;

	# done if we have enough
	$real_len = length($text)-$er_len;
	if ($real_len >= $len) {
	    last;
	}
	$sublen = $len - (length($text)-$er_len);
    }
    $text;
}

##---------------------------------------------------------------------------
##	Get an e-mail address from (HTML) $str.
##
sub extract_email_address {
    return ''  unless defined $_[0];
    scalar(SO::Mail::Message::RFC822::first_addr_spec(shift));
}

##---------------------------------------------------------------------------
##	Get an e-mail name from $str.
##
sub extract_email_name {
    my @tokens   = SO::Mail::Message::RFC822::tokenise(shift);
    my @bare     = ( );
    my $possible = undef;
    my $skip	 = 0;

    my $tok;
    foreach $tok (@tokens) {
	next  if $skip;
	if ($tok =~ /^"/) {   # Quoted string
	    $tok =~ s/^"//;  $tok =~ s/"$//;
            $tok =~ s/\\(.)/$1/g;
	    return $tok;
	}
	if ($tok =~ /^\(/) {  # Comment
	    $tok =~ s/^\(//; $tok =~ s/\)$//;
            $tok =~ s/\\(.)/$1/g;
	    return $tok;
	}
	if ($tok =~ /^<$/) {  # Address spec, skip
	    $skip = 1;
	    next;
	}
	if ($tok =~ /^>$/) {
	    $skip = 0;
	    next;
	}
	push(@bare, $tok);    # Bare name
    }

    my $str;
    if (@bare) {
	$str = join(' ', @bare);
	$str =~ s/@.*//;
	$str =~ s/^\s+//; $str =~ s/\s+$//;
	return $str;
    }
    $str = SO::Mail::Message::RFC822::first_addr_spec(@tokens);
    $str =~ s/@.*//;
    $str;
}

##---------------------------------------------------------------------------
##	Routine to sort messages
##
sub sort_messages {
    my($nosort, $subsort, $authsort, $revsort) = @_;
    $nosort   = $NOSORT    if !defined($nosort);
    $subsort  = $SUBSORT   if !defined($subsort);
    $authsort = $AUTHSORT  if !defined($authsort);
    $revsort  = $REVSORT   if !defined($revsort);

    if ($nosort) {
	## Process order
	if ($revsort) {
	    return sort { $IndexNum{$b} <=> $IndexNum{$a} } keys %Subject;
	} else {
	    return sort { $IndexNum{$a} <=> $IndexNum{$b} } keys %Subject;
	}

    } elsif ($subsort) {
	## Subject order
	my(%sub, $idx, $sub);
	use locale;
	eval {
	    my $hs = scalar(%Subject);  $hs =~ s|^[^/]+/||;
	    keys(%sub) = $hs;
	};
	while (($idx, $sub) = each(%Subject)) {
	    $sub = lc $sub;
	    1 while $sub =~ s/$SubReplyRxp//io;
	    $sub =~ s/$SubArtRxp//io;
	    $sub{$idx} = $sub;
	}
	if ($revsort) {
	    return sort { ($sub{$a} cmp $sub{$b}) ||
			  ($Time{$b} <=> $Time{$a})
			} keys %Subject;
	} else {
	    return sort { ($sub{$a} cmp $sub{$b}) ||
			  ($Time{$a} <=> $Time{$b})
			} keys %Subject;
	}
	
    } elsif ($authsort) {
	## Author order
	my(%from, $idx, $from);
	use locale;
	eval {
	    my $hs = scalar(%From);  $hs =~ s|^[^/]+/||;
	    keys(%from) = $hs;
	};
	if ($DoFromName && %FromName) {
	    while (($idx, $from) = each(%FromName)) {
		$from{$idx} = lc $from;
	    }
	} else {
	    while (($idx, $from) = each(%From)) {
		$from{$idx} = lc extract_email_name($from);
	    }
	}
	if ($revsort) {
	    return sort { ($from{$a} cmp $from{$b}) ||
			  ($Time{$b} <=> $Time{$a})
			} keys %Subject;
	} else {
	    return sort { ($from{$a} cmp $from{$b}) ||
			  ($Time{$a} <=> $Time{$a})
			} keys %Subject;
	}

    } else {
	## Date order
	if ($revsort) {
	    return sort { ($Time{$b} <=> $Time{$a})
			  || ($IndexNum{$b} <=> $IndexNum{$a})
			} keys %Subject;
	} else {
	    return sort { ($Time{$a} <=> $Time{$b})
			  || ($IndexNum{$a} <=> $IndexNum{$b})
			} keys %Subject;
	}

    }
}

##---------------------------------------------------------------------------
##	Message-sort routines for sort().
##
sub increase_index {
    (&get_time_from_index($a) <=> &get_time_from_index($b)) ||
	($IndexNum{$a} <=> $IndexNum{$b});
}

##---------------------------------------------------------------------------
##	Routine for formating a message number for use in filenames or links.
##
sub fmt_msgnum {
    sprintf("%05d", $_[0]);
}

##---------------------------------------------------------------------------
##	Routine to get filename of a message number.
##
sub msgnum_filename {
    my($fmtstr) = "$MsgPrefix%05d.$HtmlExt";
    $fmtstr .= '.gz'  if $GzipLinks;
    sprintf($fmtstr, $_[0]);
}

##---------------------------------------------------------------------------
##	Routine to get filename of an index
##
sub get_filename_from_index {
    &msgnum_filename($IndexNum{$_[0]});
}

##---------------------------------------------------------------------------
##	Routine to get time component from index
##
sub get_time_from_index {
    $Time{$_[0]} || (split(/$X/o, $_[0], 2))[0];
}

##---------------------------------------------------------------------------
##	Routine to get full pathname to annotation directory
##
sub get_note_dir {
    if (!OSis_absolute_path($NoteDir)) {
	return join($DIRSEP, $OUTDIR, $NoteDir);
    }
    $NoteDir;
}

##---------------------------------------------------------------------------
##	Routine to get lc author name from index
##
sub get_base_author {
    if ($DoFromName && %FromName) {
      return lc $FromName{$_[0]};
    }
    lc extract_email_name($From{$_[0]});
}

##---------------------------------------------------------------------------
##	Determine time from date.  Use %Zone for timezone offsets
##
sub get_time_from_date {
    my($mday, $mon, $yr, $hr, $min, $sec, $zone) = @_;
    my($time) = 0;

    $yr -= 1900  if $yr >= 1900;  # if given full 4 digit year
    $yr += 100   if $yr <= 37;    # in case of 2 digit years
    if (($yr < 70) || ($yr > 137)) {
	warn "Warning: Bad year (", $yr+1900, ") using current\n";
	$yr = (localtime(time))[5];
    }

    ## If $zone, grab gmt time, else grab local
    if ($zone) {
	$zone =~ tr/a-z/A-Z/;
	$time = &timegm($sec,$min,$hr,$mday,$mon,$yr);

	# try to modify time/date based on timezone
	OFFSET: {
	    # numeric timezone
	    if ($zone =~ /^[\+-]\d+$/) {
		$time -= &zone_offset_to_secs($zone);
		last OFFSET;
	    }
	    # Zone
	    if (defined($Zone{$zone})) {
		# timezone abbrev
		$time += &zone_offset_to_secs($Zone{$zone});
		last OFFSET;

	    }
	    # Zone[+-]DDDD
	    if ($zone =~ /^([A-Z]\w+)([\+-]\d+)$/) {
		$time -= &zone_offset_to_secs($2);
		if (defined($Zone{$1})) {
		    $time += &zone_offset_to_secs($Zone{$1});
		    last OFFSET;
		}
	    }
	    # undefined timezone
	    warn qq|Warning: Unrecognized time zone, "$zone"\n|;
	}

    } else {
	$time = &timelocal($sec,$min,$hr,$mday,$mon,$yr);
    }
    $time;
}

##---------------------------------------------------------------------------
##	Routine to check if time has expired.
##
sub expired_time {
    ($ExpireTime && (time - $_[0] > $ExpireTime)) || ($_[0] < $ExpireDateTime);
}

##---------------------------------------------------------------------------
##      Get HTML tags for formatting message headers
##
sub get_header_tags {
    my($f) = shift;
    my($ftago, $ftagc, $tago, $tagc);
 
    ## Get user specified tags (this is one funcky looking code)
    $tag = (defined($HeadHeads{$f}) ? $HeadHeads{$f} : $HeadHeads{'-default-'});
    $ftag = (defined($HeadFields{$f}) ? $HeadFields{$f} : $HeadFields{'-default-'});
    if ($tag) { $tago = "<$tag>";  $tagc = "</$tag>"; }
    else { $tago = $tagc = ''; }
    if ($ftag) { $ftago = "<$ftag>";  $ftagc = "</$ftag>"; }
    else { $ftago = $ftagc = ''; }
 
    ($tago, $tagc, $ftago, $ftagc);
}

##---------------------------------------------------------------------------
##	Format message headers in HTML.
##	$html = htmlize_header($fields_hash_ref);
##
sub htmlize_header {
    my ($fields, $FieldNames) = @_;
    my ($key, $tago, $tagc, $ftago, $ftagc, $item, @array);
    my ($tmp);

    my $mesg = '';
    my %hf = %$fields;
    foreach $item (@FieldOrder) {
	if ($item eq '-extra-') {
	    foreach $key (sort keys %hf) {
		next  if $FieldODefs{$key};
		next  if $key =~ /^x-mha-/;
		#delete $hf{$key}, next if &exclude_field($key);
		@array = @{$hf{$key}};
		foreach $tmp (@array) {
		    $tmp = $HFieldsList{$key} ? mlist_field_add_links($tmp) : &$MHeadCnvFunc($tmp);
		    $tmp = field_add_links($key, $tmp, $fields) unless $HFieldsAsIsList{$key};
		    ($tago, $tagc, $ftago, $ftagc) = get_header_tags($key);
		    $mesg .= join('', $LABELBEG, $tago, htmlize(exists($$FieldNames{$key}) ? $$FieldNames{$key} : ucfirst($key)), $tagc, $LABELEND,
				  $FLDBEG, $ftago, $tmp, $ftagc, $FLDEND, "\n");
		}
		delete $hf{$key};
	    }
	} else {
            if (!&exclude_field($item) && $hf{$item}) {
		@array = @{$hf{$item}};
		foreach $tmp (@array) {
		    $tmp = $HFieldsList{$item} ? mlist_field_add_links($tmp) : &$MHeadCnvFunc($tmp);
		    $tmp = field_add_links($item, $tmp, $fields) unless $HFieldsAsIsList{$item};
		    ($tago, $tagc, $ftago, $ftagc) = &get_header_tags($item);
		    $mesg .= join('', $LABELBEG, $tago, htmlize(exists($$FieldNames{$item}) ? $$FieldNames{$item} : ucfirst($item)), $tagc, $LABELEND,
				  $FLDBEG, $ftago, $tmp, $ftagc, $FLDEND, "\n");
		}
	    }
	    delete $hf{$item};
	}
    }
    if ($mesg) { $mesg = $FIELDSBEG . $mesg . $FIELDSEND; }
    $mesg;
}

##---------------------------------------------------------------------------

sub mlist_field_add_links {
    my $txt	= shift;
    my $ret	= '';
    local($_);
    foreach (split(/(<[^<>]+>)/, $txt)) {
	if (/^<\w+:/) {
	    chop; substr($_, 0, 1) = '';
	    $ret .= qq|&lt;<a href="$_">$_</a>&gt;|;
	} else {
	    $ret .= &$MHeadCnvFunc($_);
	}
    }
    $ret;
}

##---------------------------------------------------------------------------
##	Routine to add mailto/news links to a message header string.
##
sub field_add_links {
    my $label = lc shift;
    my $fld_text = shift;
    my $fields	 = shift;

    LBLSW: {
	if (!$NONEWS && ($label eq 'newsgroup' || $label eq 'newsgroups')) {
	    $fld_text = newsurl($fld_text, $fields->{'x-mha-message-id'});
	    last LBLSW;
	}
	if (!$NOMAILTO) {
	    $fld_text =~ s{($HAddrExp)}
			  {&mailUrl($1, $fields->{'x-mha-message-id'},
					$fields->{'x-mha-subject'},
					$fields->{'x-mha-from'});
			  }gexo;
	} else {
	    $fld_text =~ s{($HAddrExp)}
			  {&htmlize(&rewrite_address($1))
			  }gexo;
	}
	last LBLSW;
    }
    $fld_text;
}


##---------------------------------------------------------------------------
##	Routine to add news links of newsgroups names
##
sub newsurl {
    my $str     = shift;
    my $msgid_u = urlize(shift);
    my $h = "";

    local $_;
    if ($str =~ s/^([^:]*:\s*)//) {
	$h = $1;
    }
    $str =~ s/[\s<>]//g;
    my @groups = split(/,/, $str);
    my $group;
    foreach $group (@groups) {
	my $url     = $NewsUrl;
	my $group_u = urlize($group);
	$url =~ s/\$NEWSGROUP(?::U)?\$/$group_u/g;
	$url =~ s/\$MSGID(?::U)?\$/$msgid_u/g;
	$group = qq{<a href="$url">$group</a>};
    }
    $h . join(', ', @groups);	# Rejoin string
}

##---------------------------------------------------------------------------
##	$html = mailUrl($email_addr, $msgid, $subject, $from);
##
sub mailUrl {
    my $eaddr = shift || '';
    my $msgid = shift || '';
    my $sub = shift || '';
    my $from = shift || '';
    dehtmlize(\$eaddr);

    local $_;
    my($url) = ($MAILTOURL);
    my($to) = (&urlize($eaddr));
    my($toname, $todomain) = map { urlize($_) } split(/@/,$eaddr,2);
    my($froml, $msgidl) = (&urlize($from), &urlize($msgid));
    my($fromaddrl) = (&extract_email_address($from));
    my($faddrnamel, $faddrdomainl) = map { urlize($_) } split(/@/,$fromaddrl,2);
    $fromaddrl = &urlize($fromaddrl);
    my($subjectl);

    # Add "Re:" to subject if not present
    if ($sub !~ /^$SubReplyRxp/io) {
	$subjectl = 'Re:%20' . &urlize($sub);
    } else {
	$subjectl = &urlize($sub);
    }
    $url =~ s/\$FROM\$/$froml/g;
    $url =~ s/\$FROMADDR\$/$fromaddrl/g;
    $url =~ s/\$FROMADDRNAME\$/$faddrnamel/g;
    $url =~ s/\$FROMADDRDOMAIN\$/$faddrdomainl/g;
    $url =~ s/\$MSGID\$/$msgidl/g;
    $url =~ s/\$SUBJECT(?:NA)?\$/$subjectl/g;
    $url =~ s/\$TO\$/$to/g;
    $url =~ s/\$TOADDRNAME\$/$toname/g;
    $url =~ s/\$TOADDRDOMAIN\$/$todomain/g;
    $url =~ s/\$ADDR\$/$to/g;
    qq|<a href="$url">| . &htmlize(&rewrite_address($eaddr)) . q|</a>|;
}

##---------------------------------------------------------------------------##
##	Routine to parse variable definitions in a string.  The
##	function returns a list of variable/value pairs.  The format of
##	the string is similiar to attribute specification lists in
##	SGML, but NAMEs are any non-whitespace character.
##
sub parse_vardef_str {
    my $org = shift;
    my $lower = shift;
    my %hash = ();
    my ($str, $q, $var, $value);
    
    ($str = $org) =~ s/^\s+//;
    while ($str =~ s/^([^=\s]+)\s*=\s*//) {
	$var = $1;
	if ($str =~ s/^(['"])//) {
	    $q = $1;
	    if (!($q eq "'" ? $str =~ s/^([^']*)'// : $str =~ s/^([^"]*)"//)) {
                carp("Org: '".($org || '')."', Lower: '".($lower || '')."', Str: '".($str || '')."'\n");
		warn 'Warning: Unclosed quote '.$q." in: $org\n";
		return ();
	    }
	    $value = $1;

	} else {
	    if ($str =~ s/^(\S+)//) {
		$value = $1;
	    } else {
		warn "Warning: No value after $var in: $org\n";
		return ();
	    }
	}
	$str =~ s/^\s+//;
	$hash{$lower? lc($var): $var} = $value;
    }
    if ($str =~ /\S/) {
	warn "Warning: Trailing characters in: $org\n";
    }
    %hash;
}

##---------------------------------------------------------------------------##

sub msgid_to_filename {
    my $msgid = shift;
    if ($VMS) {
	$msgid =~ s/([^\w\-])/sprintf("=%02X",unpack("C",$1))/geo;
    } else {
	$msgid =~ s/([^\w.\-\@])/sprintf("=%02X",unpack("C",$1))/geo;
    }
    $msgid;
}

##---------------------------------------------------------------------------##
##	Check if new follow up list for a message is different from
##	old follow up list.
##
sub is_follow_ups_diff {
    my $f	= $Follow{$_[0]};
    my $o	= $FollowOld{$_[0]};
    if (defined($f) && defined($o)) {
	return 1  unless @$f == @$o;
	local $^W = 0;
	my $i;
	for ($i=0; $i < @$f; ++$i) {
	    return 1  if $f->[$i] ne $o->[$i];
	}
	return 0;
    }
    return (defined($f) || defined($o));
}

##---------------------------------------------------------------------------##
##	Retrieve icon URL for specified content-type.
##
sub get_icon_url {
    my $ctype = shift;
    my $icon = $Icons{$ctype};
    ICON: {
	last ICON  if defined $icon;
	if ($ctype =~ s|/.*||) {
	  $ctype .= '/*';
	  $icon = $Icons{$ctype};
	  last ICON  if defined $icon;
	}
	$icon = $Icons{'*/*'} || $Icons{'unknown'};
    }
    if (!defined($icon)) {
	return (undef, undef, undef);
    }
    if ($icon =~ s/\[(\d+)x(\d+)\]//) {
	return ($IconURLPrefix.$icon, $1, $2);
    }
    ($IconURLPrefix.$icon, undef, undef);
}

##---------------------------------------------------------------------------##

sub iconize_attachment {
    my ($namelabel, $urlfile, $desc, $target, $ctype, $args, $desc2) = @_;
    my ($icon_mu, $iconurl, $iw, $ih, $ret);
    $desc2 ||= 'Объект';
    if ($args =~ /\buseicon\b/i) {      # check if using icon
        if ($args =~ /\biconurl="([^"]+)"/i) {
            $iconurl = $1;
            ($iw, $ih) = ($1, $2) if $iconurl =~ s/\[(\d+)x(\d+)\]//;
        } else {
            ($iconurl, $iw, $ih) = get_icon_url($ctype);
        }
        if ($iconurl) {
            $icon_mu  = '<img src="' . $iconurl . '" align="left" border=0 alt="'.$desc2.':"';
            $icon_mu .= ' width="' . $iw . '"'  if $iw;
            $icon_mu .= ' height="' . $ih . '"'  if $ih;
            $icon_mu .= '>';
        }
    }
    my $frame = $args =~ /\bframe\b/;
    if (!$frame) {
        if ($icon_mu) {
            $ret =<<EOT;
<p><strong><a href="$urlfile" $target>$icon_mu</a>
<a href="$urlfile" $target><tt>$namelabel</tt></a></strong><br>
$desc</p>
EOT
        } else {
            $ret =<<EOT;
<p><strong>$desc2:
<a href="$urlfile" $target><tt>$namelabel</tt></a></strong><br>
$desc</p>
EOT
        }
    } else {
        if ($icon_mu) {
            $ret =<<EOT;
<table border="1" cellspacing="0" cellpadding="4">
<tr valign="top"><td><strong><a href="$urlfile" $target>$icon_mu</a>
<a href="$urlfile" $target><tt>$namelabel</tt></a></strong><br>
$desc</td></tr></table>
EOT
        } else {
            $ret =<<EOT;
<table border="1" cellspacing="0" cellpadding="4">
<tr><td><strong>$desc2:
<a href="$urlfile" $target><tt>$namelabel</tt></a></strong><br>
$desc</td></tr></table>
EOT
        }
    }
}

##---------------------------------------------------------------------------##

sub get_ext_by_ctype {
    my %hash = (
        '*/*' => '',
        'application/*' => '',
        'application/rtf' => 'rtf',
        'application/doc' => 'doc',
        'application/word' => 'doc',
        'application/msword' => 'doc',
        'application/winword' => 'doc',
        'application/vnd.ms-word' => 'doc',
        'application/x-msword' => 'doc',
        'application/vnd.openxmlformats-officedocument.wordprocessingml.document' => 'docx',
        'application/vnd.openxmlformats-officedocument.wordprocessingml.template' => 'dotx',
        'application/vnd.ms-word.document.macroEnabled.12' => 'docm',
        'application/vnd.ms-word.template.macroEnabled.12' => 'dotm',
        'application/excel' => 'xls',
        'application/msexcel' => 'xls',
        'application/x-excel' => 'xls',
        'application/x-ms-excel' => 'xls',
        'application/x-msexcel' => 'xls',
        'application/vnd.ms-excel' => 'xls',
        'application/vnd.msexcel' => 'xls',
        'application/vnd.openxmlformats-offedocument.spreadsheetml.sheet' => 'xlsx',
        'application/vnd.openxmlformats-officedocument.spreadsheetml.template' => 'xltx',
        'application/vnd.ms-excel.sheet.macroEnabled.12' => 'xlsm',
        'application/vnd.ms-excel.template.macroEnabled.12' => 'xltm',
        'application/vnd.ms-excel.addin.macroEnabled.12' => 'xlam',
        'application/vnd.ms-excel.sheet.binary.macroEnabled.12' => 'xlsb',
        'application/powerpoint' => 'ppt',
        'application/x-powerpoint' => 'ppt',
        'application/mspowerpoint' => 'ppt',
        'application/vnd.ms-powerpoint' => 'ppt',
        'application/ms-powerpoint' => 'ppt',
        'application/vnd-powerpoint' => 'ppt',
        'application/vnd.openxmlformats-officedocument.presentationml.presentation' => 'pptx',
        'application/vnd.openxmlformats-officedocument.presentationml.template' => 'potx',
        'application/vnd.openxmlformats-officedocument.presentationml.slideshow' => 'ppsx',
        'application/vnd.ms-powerpoint.addin.macroEnabled.12' => 'ppam',
        'application/vnd.ms-powerpoint.presentation.macroEnabled.12' => 'pptm',
        'application/vnd.ms-powerpoint.template.macroEnabled.12' => 'potm',
        'application/vnd.ms-powerpoint.slideshow.macroEnabled.12' => 'ppsm',
        'application/vnd.oasis.opendocument.text' => 'odt',
        'application/vnd.oasis.opendocument.text-template' => 'ott',
        'application/vnd.oasis.opendocument.spreadsheet' => 'ods',
        'application/vnd.oasis.opendocument.spreadsheet-template' => 'ots',
        'application/vnd.oasis.opendocument.presentation' => 'odp',
        'application/vnd.oasis.opendocument.presentation-template' => 'otp',
        'application/vnd.oasis.opendocument.graphics' => 'odg',
        'application/vnd.oasis.opendocument.graphics-template' => 'otg',
        'application/vnd.oasis.opendocument.chart' => 'odc',
        'application/vnd.oasis.opendocument.chart-template' => 'otc',
        'application/vnd.oasis.opendocument.image' => 'odi',
        'application/vnd.oasis.opendocument.image-template' => 'oti',
        'application/vnd.oasis.opendocument.formula' => 'odf',
        'application/vnd.oasis.opendocument.formula-template' => 'otf',
        'application/vnd.oasis.opendocument.text-master' => 'odm',
        'application/vnd.oasis.opendocument.text-web' => 'oth',
        'application/x-shockwave-flash' => 'swf',
        'application/octet-stream' => '',
        'application/pdf' => 'pdf',
        'application/postscript' => 'ps',
        'application/x-sh' => 'sh',
        'application/x-csh' => 'csh',
        'application/x-ksh' => 'ksh',
        'application/x-script' => '',
        'application/x-dvi' => 'dvi',
        'application/x-gtar' => 'gtar',
        'application/x-gzip' => 'gzip',
        'application/x-rar' => 'rar',
        'application/x-rar-compressed' => 'rar',
        'application/x-latex' => 'tex',
        'application/x-patch' => '',
        'application/x-tar' => 'tar',
        'application/x-tex' => 'tex',
        'application/x-zip-compressed' => 'zip',
        'application/zip' => 'zip',
        'application/x-javascript' => 'js',
        'audio/ac3' => 'ac3',
        'audio/aac' => 'aac',
        'audio/mpeg' => 'mpeg',
        'audio/mp4' => 'mp4',
        'audio/x-ms-wma' => 'wma',
        'audio/x-wav' => 'wav',
        'audio/vnd.wave' => 'wav',
        'audio/ogg' => 'ogg',
        'audio/webm' => 'webm',
        'audio/x-aiff' => 'aif',
        'audio/x-matroska' => 'mka',
        'audio/*' => '',
        'chemical/*' => '',
        'image/gif' => 'gif',
        'image/jpeg' => 'jpeg',
        'image/pjpeg' => 'jpeg',
        'image/tiff' => 'tiff',
        'image/png' => 'png',
        'image/psd' => 'psd',
        'image/bmp' => 'bmp',
        'image/vnd.microsoft.icon' => 'ico',
        'image/svg+xml' => 'svg',
        'image/vnd.wap.wbmp' => 'wbmp',
        'image/*' => '',
        'message/external-body' => 'eml',
        'message/rfc822' => 'eml',
        'message/partial' => 'eml',
        'multipart/*' => 'eml',
        'text/html' => 'html',
        'text/plain' => 'txt',
        'text/css' => 'css',
        'text/csv' => 'csv',
        'text/xml' => 'xml',
        'text/*' => 'txt',
        'video/mpeg' => 'mpeg',
        'video/x-msvideo' => 'avi',
        'video/quicktime' => 'mov',
        'video/mp4' => 'mp4',
        'video/x-ms-wmv' => 'wmv',
        'video/x-flv' => 'flv',
        'video/ogg' => 'ogg',
        'video/webm' => 'webm',
        'video/x-matroska' => 'mkv',
        'video/ogv' => 'ogv',
        'video/*' => '',
        'application/vnd.mozilla.xul+xml' => 'xul'
    );
    $hash{shift || '*/*'}
}

##---------------------------------------------------------------------------##

sub get_ctype_by_ext {
    my %hash = (
        '' => 'application/octet-stream',
        'bin' => 'application/octet-stream',
        'rtf' => 'application/rtf',
        'doc' => 'application/msword',
        'dot' => 'application/msword',
        'docx' => 'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
        'dotx' => 'application/vnd.openxmlformats-officedocument.wordprocessingml.template',
        'docm' => 'application/vnd.ms-word.document.macroEnabled.12',
        'dotm' => 'application/vnd.ms-word.template.macroEnabled.12',
        'xls' => 'application/x-excel',
        'xlt' => 'application/x-excel',
        'xla' => 'application/x-excel',
        'xlsx' => 'application/vnd.openxmlformats-offedocument.spreadsheetml.sheet',
        'xltx' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.template',
        'xlsm' => 'application/vnd.ms-excel.sheet.macroEnabled.12',
        'xltm' => 'application/vnd.ms-excel.template.macroEnabled.12',
        'xlam' => 'application/vnd.ms-excel.addin.macroEnabled.12',
        'xlsb' => 'application/vnd.ms-excel.sheet.binary.macroEnabled.12',
        'ppt' => 'application/vnd.ms-powerpoint',
        'pot' => 'application/vnd.ms-powerpoint',
        'pps' => 'application/vnd.ms-powerpoint',
        'ppa' => 'application/vnd.ms-powerpoint',
        'pptx' => 'application/vnd.openxmlformats-officedocument.presentationml.presentation',
        'potx' => 'application/vnd.openxmlformats-officedocument.presentationml.template',
        'ppsx' => 'application/vnd.openxmlformats-officedocument.presentationml.slideshow',
        'ppam' => 'application/vnd.ms-powerpoint.addin.macroEnabled.12',
        'pptm' => 'application/vnd.ms-powerpoint.presentation.macroEnabled.12',
        'potm' => 'application/vnd.ms-powerpoint.template.macroEnabled.12',
        'ppsm' => 'application/vnd.ms-powerpoint.slideshow.macroEnabled.12',
        'odt' => 'application/vnd.oasis.opendocument.text',
        'ott' => 'application/vnd.oasis.opendocument.text-template',
        'ods' => 'application/vnd.oasis.opendocument.spreadsheet',
        'ots' => 'application/vnd.oasis.opendocument.spreadsheet-template',
        'odp' => 'application/vnd.oasis.opendocument.presentation',
        'otp' => 'application/vnd.oasis.opendocument.presentation-template',
        'odg' => 'application/vnd.oasis.opendocument.graphics',
        'otg' => 'application/vnd.oasis.opendocument.graphics-template',
        'odc' => 'application/vnd.oasis.opendocument.chart',
        'otc' => 'application/vnd.oasis.opendocument.chart-template',
        'odi' => 'application/vnd.oasis.opendocument.image',
        'oti' => 'application/vnd.oasis.opendocument.image-template',
        'odf' => 'application/vnd.oasis.opendocument.formula',
        'otf' => 'application/vnd.oasis.opendocument.formula-template',
        'odm' => 'application/vnd.oasis.opendocument.text-master',
        'oth' => 'application/vnd.oasis.opendocument.text-web',
        'swf' => 'application/x-shockwave-flash',
        'pdf' => 'application/pdf',
        'ps' => 'application/postscript',
        'ai' => 'application/postscript',
        'eps' => 'application/postscript',
        'sh' => 'application/x-sh',
        'csh' => 'application/x-csh',
        'ksh' => 'application/x-ksh',
        'dvi' => 'application/x-dvi',
        'gtar' => 'application/x-gtar',
        'gzip' => 'application/x-gzip',
        'rar' => 'application/x-rar',
        'tex' => 'application/x-tex',
        'tar' => 'application/x-tar',
        'zip' => 'application/zip',
        'js' => 'application/x-javascript',
        'mpeg' => 'audio/mpeg',
        'mp3' => 'audio/mpeg',
        'mpga' => 'audio/mpeg',
        'ac3' => 'audio/ac3',
        'aac' => 'audio/aac',
        'aif' => 'audio/x-aiff',
        'aifc' => 'audio/aiff',
        'aiff' => 'audio/aiff',
        'wma' => 'audio/x-ms-wma',
        'wav' => 'audio/x-wav', # audio/vnd.wave
        'ogg' => 'audio/ogg',
        'mka' => 'audio/x-matroska',
        'gif' => 'image/gif',
        'jpeg' => 'image/jpeg',
        'jpg' => 'image/jpeg',
        'jpe' => 'image/jpeg',
        'tiff' => 'image/tiff',
        'tif' => 'image/tiff',
        'png' => 'image/png',
        'psd' => 'image/psd',
        'bmp' => 'image/bmp',
        'ico' => 'image/vnd.microsoft.icon',
        'svg' => 'image/svg+xml',
        'eml' => 'message/external-body',
        'mht' => 'message/rfc822',
        'mhtml' => 'message/rfc822',
        'txt' => 'text/plain',
        'htm' => 'text/html',
        'html' => 'text/html',
        'css' => 'text/css',
        'xml' => 'text/xml',
        'csv' => 'text/csv',
        'mpeg' => 'video/mpeg',
        'mpg' => 'video/mpeg',
        'mpe' => 'video/mpeg',
        'avi' => 'video/x-msvideo',
        'mov' => 'video/quicktime',
        'qt' => 'video/quicktime',
        'mp4' => 'video/mp4',
        'wmv' => 'video/x-ms-wmv',
        'flv' => 'video/x-flv',
        'mkv' => 'video/x-matroska',
        'ogv' => 'video/ogv',
        'wrl' => 'x-world/x-vrml',
        'vrml' => 'x-world/x-vrml',
        'xul' => 'application/vnd.mozilla.xul+xml'
    );
    $hash{shift || ''}
}
##---------------------------------------------------------------------------##

sub log_mesg {
    my $fh	= shift;
    my $doDate	= shift;

    if ($doDate) {
	my($sec,$min,$hour,$mday,$mon,$year) = localtime(time);
	print $fh sprintf("[%4d-%02d-%02d %02d:%02d:%02d] ",
			  $year+1900, $mon+1, $mday, $hour, $min, $sec);
    }
    print $fh @_;
}

##---------------------------------------------------------------------------##

sub dump_hash {
    my $fh = shift;
    my $h = shift;
    local $_;
    foreach (sort keys %$h) {
	print $fh "$_ => ", $h->{$_}, "\n";
    }
}

##---------------------------------------------------------------------------
##	Routine to convert a msgid to an anchor
##
sub link_refmsgid {
    my $refmsgid = dehtmlize(shift);
    my $onlynew  = shift;

    if (defined($MsgId{$refmsgid}) && defined($IndexNum{$MsgId{$refmsgid}}) && (!$onlynew || $NewMsgId{$refmsgid})) {
	my($lreftmpl) = $MSGIDLINK;
	$lreftmpl =~ s/$VarExp/&replace_li_var($1,$MsgId{$refmsgid})/geo;
	return $lreftmpl;
    }
    htmlize($refmsgid);
}

##---------------------------------------------------------------------------
##	Retrieve next available message number.  Should only be used
##	when an archive is locked.
##
sub getNewMsgNum {
    $NumOfMsgs++; $LastMsgNum++;
    $LastMsgNum;
}

##---------------------------------------------------------------------------##
1;
