#!/usr/bin/perl
# -*- Encoding: utf-8; Mode: cperl -*-
# kate: space-indent on; indent-width 4; replace-tabs on;
#
package SO::MongoCollection;
use base qw(Exporter);

use strict;
use warnings;
use SO::MongoDatabase;
use SO::LogFile;
use Data::Dumper;

our @EXPORT = qw(isValid upsert update findOne find aggregate);
our @EXPORT_OK = qw(insert2Array $mongoDatabase insert get remove switchDB);
our %EXPORT_TAGS = (
    'savedata' => [qw(insert2Array upsert update findOne find aggregate)],
    'getlog' => [qw(find get findOne)],
    'operations' => [qw(insert upsert update remove find findOne aggregate)],
    'monalarms' => [qw(get insert upsert update find findOne aggregate)]
);
our $VERSION = 1.1;
our $mongoDatabase;

# Local variables
my $mongoDefaultDatabase = 'monitoring';
my $mongoOperationsAttempts = 10;

# Methods of class

sub new {
    my $class = shift();
    my $self  = bless {}, $class; startProfiling();
    $self->{'collection_name'} = shift();
    $self->{'db_name'} = shift() || $mongoDefaultDatabase;
    $mongoDatabase = SO::MongoDatabase->new($self->{'db_name'}) unless $mongoDatabase and $mongoDatabase->{'db'};
    $self->{'db'} = $mongoDatabase;
    $self->{'collection'} = $self->{'db'}->get_collection($self->{'collection_name'}) if $self->{'collection_name'};
    $self->{'mongoOperationsAttempts'} = $self->{'db'}->{'mongoOperationsAttempts'} || $mongoOperationsAttempts;
    endProfiling('SO::MongoCollection->new');
    return $self;
}

sub isValid { my $self = shift(); !!($self->{'collection_name'} and $self->{'collection'}) }

sub switchDB {
    my ($iniFileSection, $iniFilePath, $logFilePath) = @_;
    $SO::LogFile::defaultLogFile = $logFilePath if $logFilePath;
    $mongoDatabase = undef;
    $SO::MongoDatabase::defaultIniFile = $iniFilePath if $iniFilePath;
    $SO::MongoDatabase::defaultIniFileSection = $iniFileSection if $iniFileSection;
}

sub insert {
    my ($self, $doc) = @_;
    return undef unless isValid($self);
    my ($res, $i); $i = 0; startProfiling();
    LOOP: {
        $res = eval { $self->{'collection'}->insert($doc) }; $i++;
        last LOOP unless $@ and $@ !~ /duplicate key error/;
        if ($i <= $$self{'mongoOperationsAttempts'}) {
            if ($@ =~ /\bnot master\b/i) { $self->{'db'}->reconnect(); redo LOOP }
            if ($@ =~ /\btry\b/i or $@ =~ /\bresponse\b/i) { redo LOOP }
        } else {
            $self->{'lastError'} = "SO::MongoCollection('".$$self{'collection_name'}."')::insert: $@Attempt $i. Data for insertion: ".Dumper($doc);
            writeLog($$self{'lastError'}, $$self{'db'}{'logFile'})
        }
    };
    endProfiling('SO::MongoCollection->insert');
    return $res;
}

sub insert2Array {
    my ($self, $query, $array_name, $el) = @_;
    return undef unless isValid($self);
    my $sf = ref($el) eq 'HASH' ? '_id' : '';
    my %data = (%$query, $array_name.($sf ? '.'.$sf : '') => {'$ne' => $sf ? $el->{$sf} : $el});
    return upsert($self, \%data, {'$push' => {$array_name => $el}});
}

sub upsert {
    my ($self, $query, $update, $is_multi) = @_;
    return undef unless isValid($self) and $query and $update;
    my ($res, $i, $h); $i = 0; startProfiling();
    $h = {'upsert' => 1}; $h->{'multi'} = 1 if $is_multi;
    LOOP: {
        $res = eval { $self->{'collection'}->update($query, $update, $h) }; $i++;
        #writeLog('Attempt '.$i.'. SO::MongoCollection->upsert: '.$@.'Query: '.Dumper($query), $$self{'db'}{'logFile'}) if $@;
        last LOOP unless $@ and $@ !~ /duplicate key error/;
        if ($i <= $$self{'mongoOperationsAttempts'}) {
            if ($@ =~ /\bnot master\b/i) { $self->{'db'}->reconnect(); redo LOOP }
            if ($@ =~ /\btry\b/i or $@ =~ /\bresponse\b/i) { redo LOOP }
        } else {
            $self->{'lastError'} = "SO::MongoCollection('".$$self{'collection_name'}."')::upsert: $@Attempt $i. Query: ".Dumper($query).'Update operation: '.Dumper($update).'Result: '.Dumper($res);
            writeLog($$self{'lastError'}, $$self{'db'}{'logFile'})
        }
    };
    endProfiling('SO::MongoCollection->upsert');
    $res
}

sub update {
    my ($self, $query, $update) = @_;
    return undef unless isValid($self) and $query and $update;
    my ($res, $i); $i = 0; startProfiling();
    LOOP: {
        $res = eval { $self->{'collection'}->update($query, $update, {'multi' => 1}) }; $i++;
        last LOOP unless $@ and $@ !~ /duplicate key error/;
        if ($i <= $$self{'mongoOperationsAttempts'}) {
            if ($@ =~ /\bnot master\b/i) { $self->{'db'}->reconnect(); redo LOOP }
            if ($@ =~ /\btry\b/i or $@ =~ /\bresponse\b/i) { redo LOOP }
        } else {
            $self->{'lastError'} = "SO::MongoCollection('".$$self{'collection_name'}."')::update: $@Attempt $i. Query: ".Dumper($query).'Update operation: '.Dumper($update);
            writeLog($$self{'lastError'}, $$self{'db'}{'logFile'})
        }
    }
    endProfiling('SO::MongoCollection->update');
    $res
}

sub findOne {
    my ($self, $query, $field) = @_;
    return undef unless isValid($self); $query ||= {};
    my ($res, $i); $i = 0; startProfiling();
    LOOP: {
        $res = eval { $self->{'collection'}->find_one($query, $field ? {$field => 1} : {}) }; $i++;
        last LOOP unless $@;
        if ($i <= $$self{'mongoOperationsAttempts'}) {
            if ($@ =~ /\bnot master\b/i) { $self->{'db'}->reconnect(); redo LOOP }
            if ($@ =~ /\btry\b/i or $@ =~ /\bresponse\b/i) { redo LOOP }
        } else {
            $self->{'lastError'} = "SO::MongoCollection('".$$self{'collection_name'}."')::findOne error (attempt $i): $@Query: ".Dumper($query)."Field: '".($field || '')."'";
            writeLog($$self{'lastError'}, $$self{'db'}{'logFile'})
        }
    }
    endProfiling('SO::MongoCollection->findOne', 0.8);
    ($field and $res and !$@) ? $res->{$field} : $res
}

sub find {
    my ($self, $filter, $subkey, $field, $sort, $limit, $skip) = @_;
    return {} unless isValid($self);
    my %data = (); startProfiling();
    my $i = 0;
    LOOP: {
        eval {
            my $cursor = $self->{'collection'}->find(($filter and ref($filter) eq 'HASH') ? $filter : {}); $i++;
            $cursor = $cursor->fields({$field => 1}) if $field;
            $cursor = $cursor->sort($sort) if $sort and ref($sort) eq 'HASH';
            $cursor = $cursor->limit($limit) if $limit and $limit > 0;
            $cursor = $cursor->skip($skip) if $skip and $skip > 0;
            while (my $row = $cursor->next) {
                $data{$subkey ? $$row{'_id'}{$subkey} : $$row{'_id'}} = $field ? $$row{$field} : $row
            }
        };
        last LOOP unless $@;
        if ($i <= $$self{'mongoOperationsAttempts'}) {
            if ($@ =~ /\bnot master\b/i) { $self->{'db'}->reconnect(); redo LOOP }
            if ($@ =~ /\btry\b/i or $@ =~ /\bresponse\b/i) { redo LOOP }
        } else {
            $self->{'lastError'} = "SO::MongoCollection('".$$self{'collection_name'}."')::find error (attempt $i): $@Subkey: '".($subkey || '')."', Filter: ".Dumper($filter);
            writeLog($$self{'lastError'}, $$self{'db'}{'logFile'})
        }
    }
    endProfiling('SO::MongoCollection->find', 0.3);
    \%data
}

sub get {
    my ($self, $filter, $field, $is_distinct) = @_;
    my (@res, $i); @res = (); $i = 0; startProfiling();
    if ($is_distinct and $self->{'db'}->isValidConnection()) {
        $field ||= '_id'; $filter ||= {};
        LOOP1: {
            my $res = eval { $self->{'db'}->{'db'}->run_command(['distinct' => $$self{'collection_name'}, 'key' => $field, 'query' => $filter]) };
            $i++; unless ($@) { @res = @{$res->{'values'}}; last LOOP1 }
            if ($i <= $$self{'mongoOperationsAttempts'}) {
                if ($@ =~ /\bnot master\b/i) { $self->{'db'}->reconnect(); redo LOOP1 }
                if ($@ =~ /\btry\b/i or $@ =~ /\bresponse\b/i) { redo LOOP1 }
            } else {
                $self->{'lastError'} = "SO::MongoCollection('".$$self{'collection_name'}."')::get error (attempt $i): $@Field: '".($field || '')."', IsDistinct: 1, Query: ".Dumper($filter);
                writeLog($$self{'lastError'}, $$self{'db'}{'logFile'})
            }
        }
    } elsif (isValid($self)) {
        LOOP2: {
            eval {
                my $cursor = $self->{'collection'}->find($filter and ref($filter) eq 'HASH' ? $filter : {}, $field ? {$field => 1} : undef); $i++;
                while (my $row = $cursor->next) {
                    push @res, $field ? $row->{$field} : $row
                }
            };
            last LOOP2 unless $@;
            if ($i <= $$self{'mongoOperationsAttempts'}) {
                if ($@ =~ /\bnot master\b/i) { $self->{'db'}->reconnect(); redo LOOP2 }
                if ($@ =~ /\btry\b/i or $@ =~ /\bresponse\b/i) { redo LOOP2 }
            } else {
                $self->{'lastError'} = "SO::MongoCollection('".$$self{'collection_name'}."')::get error (attempt $i): $@Field: '".($field || '')."', IsDistinct: 0, Query: ".Dumper($filter);
                writeLog($$self{'lastError'}, $$self{'db'}{'logFile'})
            }
        }
    }
    endProfiling('SO::MongoCollection->get', 0.3);
    wantarray ? @res : \@res
}

sub count {
    my ($self, $filter, $limit, $skip) = @_;
    my ($res, $i, %doc); $i = 0; startProfiling();
    if ($self->{'db'}->isValidConnection()) {
        $filter ||= {}; %doc = ('count' => $$self{'collection_name'}, 'query' => $filter);
        $doc{'limit'} = $limit if defined($limit);
        $doc{'skip'} = $skip if defined($skip);
        LOOP1: {
            $res = eval { $self->{'db'}->{'db'}->run_command([%doc]) };
            $i++; unless ($@) { $res = ref($res) eq 'HASH' ? ($res->{'n'} || 0) : undef; last LOOP1 }
            if ($i <= $$self{'mongoOperationsAttempts'}) {
                if ($@ =~ /\bnot master\b/i) { $self->{'db'}->reconnect(); redo LOOP1 }
                if ($@ =~ /\btry\b/i or $@ =~ /\bresponse\b/i) { redo LOOP1 }
            } else {
                $self->{'lastError'} = "SO::MongoCollection('".$$self{'collection_name'}."')::count error (attempt $i): $@Query: ".Dumper($filter);
                writeLog($$self{'lastError'}, $$self{'db'}{'logFile'})
            }
        }
    }
    endProfiling('SO::MongoCollection->count', 0.3);
    $res
}

sub aggregate {
    my ($self, $array) = @_;
    return undef unless isValid($self) and $array and ref($array) eq 'ARRAY' and scalar(@$array) > 0;
    my ($res, $i); $i = 0; startProfiling();
    LOOP: {
        $res = eval { $self->{'collection'}->aggregate($array) }; $i++;
        last LOOP unless $@;
        if ($i <= $$self{'mongoOperationsAttempts'}) {
            if ($@ =~ /\bnot master\b/i) { $self->{'db'}->reconnect(); redo LOOP }
            if ($@ =~ /\btry\b/i or $@ =~ /\bresponse\b/i) { redo LOOP }
        } else {
            $self->{'lastError'} = "SO::MongoCollection('".$$self{'collection_name'}."')::aggregate error (attempt $i): $@Aggregation pipeline array: ".Dumper($array);
            writeLog($$self{'lastError'}, $$self{'db'}{'logFile'})
        }
    }
    endProfiling('SO::MongoCollection->aggregate', 0.3);
    ref($res) eq 'ARRAY' ? $$res[0] : {}
}

sub aggregateFull {
    my ($self, $array) = @_;
    return undef unless isValid($self) and $array and ref($array) eq 'ARRAY' and scalar(@$array) > 0;
    my ($res, $i); $i = 0; startProfiling();
    foreach (@$array) { next unless exists $$_{'$limit'}; $$_{'$limit'} += 0 }
    LOOP: {
        $res = eval { $self->{'collection'}->aggregate($array) }; $i++;
        last LOOP unless $@;
        if ($i <= $$self{'mongoOperationsAttempts'}) {
            if ($@ =~ /\bnot master\b/i) { $self->{'db'}->reconnect(); redo LOOP }
            if ($@ =~ /\btry\b/i or $@ =~ /\bresponse\b/i) { redo LOOP }
        } else {
            $self->{'lastError'} = "SO::MongoCollection('".$$self{'collection_name'}."')::aggregate error (attempt $i): $@Aggregation pipeline array: ".Dumper($array);
            writeLog($$self{'lastError'}, $$self{'db'}{'logFile'})
        }
    }
    endProfiling('SO::MongoCollection->aggregateFull', 0.3);
    ($res and ref($res) eq 'ARRAY') ? (wantarray() ? @$res : $res) : []
}

sub remove {
    my ($self, $query) = @_;
    return undef unless isValid($self) and $query;
    my ($res, $i); $i = 0; startProfiling();
    LOOP: {
        $res = eval { $self->{'collection'}->remove($query) }; $i++;
        last LOOP unless $@;
        if ($i <= $$self{'mongoOperationsAttempts'}) {
            if ($@ =~ /\bnot master\b/i) { $self->{'db'}->reconnect(); redo LOOP }
            if ($@ =~ /\btry\b/i or $@ =~ /\bresponse\b/i) { redo LOOP }
        } else {
            $self->{'lastError'} = "SO::MongoCollection('".$$self{'collection_name'}."')::remove: $@Attempt $i. Query: ".Dumper($query);
            writeLog($$self{'lastError'}, $$self{'db'}{'logFile'})
        }
    }
    endProfiling('SO::MongoCollection->remove');
    $res
}

sub isPartitioned {
    my $self = shift;
    return unless isValid($self);
    my $res = eval { $self->{'db'}->run_command(['buildInfo' => 1]) };
    return if $@ or ref($res) ne 'HASH' or !exists $res->{'tokumxVersion'} or $res->{'tokumxVersion'} lt '1.5.0';
    $res = eval { $self->{'db'}->run_command(['getPartitionInfo' => $$self{'collection_name'}]) };
    ref($res) eq 'HASH' and $$res{'ok'}
}

sub getPartitionInfo {
    my $self = shift;
    return unless isValid($self);
    my $res = eval { $self->{'db'}->run_command(['buildInfo' => 1]) };
    return if $@ or ref($res) ne 'HASH' or !exists $res->{'tokumxVersion'} or $res->{'tokumxVersion'} lt '1.5.0';
    eval { $self->{'db'}->run_command(['getPartitionInfo' => $$self{'collection_name'}]) }
}

sub addPartition {
    my ($self, $ops) = @_;
    return unless isValid($self); $ops ||= {};
    startProfiling();
    my $res = eval { $self->{'db'}->run_command(['buildInfo' => 1]) };
    return if $@ or ref($res) ne 'HASH' or !exists $res->{'tokumxVersion'} or $res->{'tokumxVersion'} lt '1.5.0';
    $res = eval { $self->{'db'}->run_command(['addPartition' => $$self{'collection_name'}, 'newMax' => $ops]) };
    if ($@) {
        $self->{'lastError'} = "SO::MongoCollection('".$$self{'collection_name'}."')::addPartition: $@Options: ".Dumper($ops);
        writeLog($$self{'lastError'}, $$self{'db'}{'logFile'})
    }
    endProfiling('SO::MongoCollection->addPartition');
    $res
}

sub dropPartition {
    my ($self, $ops) = @_;
    return unless isValid($self) and $ops and ref($ops) eq 'HASH' and (exists $$ops{'id'} or exists $$ops{'max'});
    startProfiling();
    my $res = eval { $self->{'db'}->run_command(['buildInfo' => 1]) };
    return if $@ or ref($res) ne 'HASH' or !exists $res->{'tokumxVersion'} or $res->{'tokumxVersion'} lt '1.5.0';
    $res = eval { $self->{'db'}->run_command(['dropPartition' => $$self{'collection_name'}, %$ops]) };
    if ($@) {
        $self->{'lastError'} = "SO::MongoCollection('".$$self{'collection_name'}."')::dropPartition: $@Options: ".Dumper($ops);
        writeLog($$self{'lastError'}, $$self{'db'}{'logFile'})
    }
    endProfiling('SO::MongoCollection->dropPartition');
    $res
}

sub ensureIndex {
    my ($self, $ops) = @_;
    return unless isValid($self) and $ops and ref($ops) eq 'HASH';
    startProfiling();
    my $res = eval { $self->{'collection'}->ensure_index($ops) };
    if ($@) {
        $self->{'lastError'} = "SO::MongoCollection('".$$self{'collection_name'}."')::ensureIndex: $@Options: ".Dumper($ops);
        writeLog($$self{'lastError'}, $$self{'db'}{'logFile'})
    }
    endProfiling('SO::MongoCollection->ensureIndex');
    $res
}

1;