#!/usr/bin/perl
# -*- Encoding: utf-8; Mode: cperl -*-
# kate: space-indent on; indent-width 4; replace-tabs on;
#
package SO::MongoDatabase;
use base qw(Exporter);

use strict;
use warnings;
use Sys::Syslog qw(:standard :macros);
use Time::HiRes;

BEGIN {
    push @INC, '/usr/lib/perl5', '/usr/lib/perl5/x86_64-linux-thread-multi';
    openlog('MongoDB:', 'ndelay,pid', LOG_DAEMON);
}

use MongoDB;
use Data::Dumper;
use SO::IniFile qw(loadIni);
use SO::LogFile;
use Net::HTTP;

our @EXPORT_OK = qw(isValidConnection $defaultIniFile $defaultIniFileSection startProfiling endProfiling sysLog);
our @EXPORT = @EXPORT_OK;
our $VERSION = 1.00;
our $defaultIniFile = 'WORKING_DIR/so-monlog.ini';
our $defaultIniFileSection = 'general';

# Local variables
my $t0 = 0;
my ($mongoDefaultHost, $mongoDefaultPort, $mongoDefaultDatabase, $errorsLogFile) = ('localhost', 27017, 'monitoring', 'WORKING_DIR/logs/monlog-errors.log');

# Local functions

sub startProfiling { $t0 = Time::HiRes::time() }
sub endProfiling {
    my ($prompt, $max_dt, $dt) = (shift() || 'process time', shift() || 0.1, Time::HiRes::time() - $t0);
    syslog(LOG_INFO, '%s: %g', $prompt, $dt) if $dt >= $max_dt
}
sub sysLog { syslog(LOG_INFO, shift()) }
sub getHosts4Group {
    my ($group, @res) = shift();
    my $http = Net::HTTP->new('Host' => 'c.yandex-team.ru', 'HTTPVersion' => '1.0', 'KeepAlive' => 0, 'Timeout' => 1);
    writeLog('getHosts4Group: cannot create HTTP object'), return wantarray() ? @res : \@res unless $http;
    $http->write_request('GET' => '/api-cached/groups2hosts/'.$group);
    my ($code, $mess, %h) = eval { $http->read_response_headers; };
    my ($answer, $buf, $size) = ('', '', 0);
    if ($@) { writeLog("getHosts4Group: $@") }
    elsif ($code < 300) {
        while (1) {
            $size = $http->read_entity_body($buf, 1024);
            last unless $size;
            $answer .= $buf;
        }
        foreach (split /\n/, $answer) {
            next unless /(\S+)/;
            push @res, $1;
        }
    } else { writeLog("getHosts4Group: $code $mess") }
    wantarray() ? @res : \@res
}
# Methods of class

sub new {
    my $class = shift();
    my $self  = bless {}, $class; startProfiling();
    my ($iniFile, %q) = SO::IniFile->new($defaultIniFile); %q = ();
    $iniFile->loadIni(\%q, $defaultIniFileSection);
    $self->{'ini'} = \%q;
    $self->{'lastError'} = $iniFile->{'lastError'}, $iniFile->{'lastError'} = '' if $iniFile->{'lastError'};
    $self->{'db_name'} = shift() || $q{'db_name'} || $mongoDefaultDatabase;
    $self->{'port'} = shift() || $q{'db_port'} || $mongoDefaultPort;
    $q{'db_hosts'} = join(' ', @{getHosts4Group($q{'db_cluster'}) || []}) || $q{'db_hosts'} if $q{'db_cluster'};
    $self->{'hosts'} = shift() || $q{'db_hosts'} || '';
    $self->{'user'} = shift() || $q{'db_user'} || '';
    $self->{'passwd'} = shift() || $q{'db_passwd'} || '';
    $self->{'defaultHost'} = shift() || $q{'db_host'} || $mongoDefaultHost;
    $self->{'logFile'} = shift() || $q{'errorsLogFile'} || $errorsLogFile;
    $self->{'w'} = $q{'db_w'} || 0;
    $self->{'timeout'} = $q{'db_timeout'} || 30000;
    $self->{'query_timeout'} = $q{'db_query_timeout'} || -1;
    reconnect($self);
    endProfiling('SO::MongoDatabase->new');
    return $self;
}

sub isValidConnection { eval { ref(shift->{'db'}) eq 'MongoDB::Database' } }

sub isMaster {
    my $self = shift();
    return -1 unless isValidConnection($self); startProfiling();
    my $res = eval { $self->{'db'}->run_command(['isMaster' => 1]) };
    if ($@) {
        my ($i, $s, $package, $filename, $line) = (0, '');
        while(($package, $filename, $line) = caller($i++)) {
            $s .= ($s ? "\t\t" : '').$i.': package '.$package.', file "'.$filename.'", line '.$line."\n";
        }
        $s = ($self->{'lastError'} ? "\n" : '')."SO::MongoDatabase::isMaster: $@\tStack: $s\tDB: ".Dumper($self->{'db'});
        $self->{'lastError'} .= $s;
        writeLog($s, $$self{'logFile'});
        return -1 if $@ =~ /can't get db response/
    }
    endProfiling('SO::MongoDatabase->isMaster');
    $res ? eval($$res{'ismaster'}) : 0
}

sub selectMaster {
    my ($self, $hosts) = @_;
    my ($is_master, $count) = (0, 0); startProfiling();
    foreach my $host (split /,?\s+/, $hosts) {
        $self->{'client'} = eval { MongoDB::MongoClient->new('host' => $host, 'port' => $self->{'port'}, 'w' => $self->{'w'}, 'timeout' => $self->{'timeout'}, 'query_timeout' => $self->{'query_timeout'}) };
        $self->{'lastError'} .= ($self->{'lastError'} ? "\n" : '').$@ if $@;
        next unless $self->{'client'};
        if ($$self{'user'}) {
            my $res = eval { $self->{'client'}->authenticate($$self{'db_name'}, $$self{'user'}, $$self{'passwd'}) };
            writeLog("selectMaster - authenticate error: '$@'. Auth result: ".Dumper($res).' for db: '.Dumper($self)) unless ref($res) eq 'HASH' and $$res{'ok'} and !$@;
        }
        $self->{'db'} = eval { $self->{'client'}->get_database($self->{'db_name'}) };
        writeLog('selectMaster - DB selection error: '.Dumper($self->{'db'}), $$self{'logFile'}) unless ref($$self{'db'}) eq 'MongoDB::Database';
        writeLog('selectMaster error: '.$@, $$self{'logFile'}) if $@;
        $is_master = isMaster($self);
        if ($is_master == -1 && $count < 10) { sleep 1; $count++; redo }
        elsif ($is_master == 1) {
            $self->{'host'} = $host; return $host
        }
        $count = 0;
    }
    endProfiling('SO::MongoDatabase->selectMaster'); 0
}

sub reconnect {
    my $self = shift(); $t0 = Time::HiRes::time();
    my $host = `hostname`; chomp $host;
    unless ($self->{'hosts'} and selectMaster($self, $self->{'hosts'})) {
        $self->{'host'} ||= $$self{'defaultHost'};
        $self->{'lastWarning'} = "Master DB not found among $$self{hosts}. DB host $$self{host} will accepted.";
        writeLog($self->{'lastWarning'}, $$self{'logFile'}) if $self->{'hosts'};
        $self->{'host'} = 'localhost' if $self->{'host'} eq $host;
        $self->{'client'} = eval { MongoDB::MongoClient->new('host' => $self->{'host'}, 'port' => $self->{'port'}, 'w' => $self->{'w'}, 'timeout' => $self->{'timeout'}, 'query_timeout' => $self->{'query_timeout'}) };
        $self->{'lastError'} .= ($self->{'lastError'} ? "\n" : '').$@ if $@;
        if ($self->{'client'}) {
            if ($$self{'user'}) {
                my $res = eval { $self->{'client'}->authenticate($$self{'db_name'}, $$self{'user'}, $$self{'passwd'}) };
                writeLog("reconnect - authenticate error: '$@'. Auth result: ".Dumper($res)) unless ref($res) eq 'HASH' and $$res{'ok'} and !$@;
            }
            $self->{'db'} = eval { $self->{'client'}->get_database($self->{'db_name'}) };
            writeLog('reconnect - DB selection error: '.Dumper($self->{'db'}), $$self{'logFile'}) unless ref($$self{'db'}) eq 'MongoDB::Database';
            writeLog('reconnect error: '.$@, $$self{'logFile'}) if $@;
        }
    }
    $MongoDB::Cursor::slave_okay = 1 unless isMaster($self);
    endProfiling('SO::MongoDatabase->reconnect');
}

sub addPartitionedCollection {
    my ($self, $name, $primaryKey) = @_;
    return unless isValidConnection($self) and $name and $primaryKey;
    my $res = eval { $self->run_command(['buildInfo' => 1]) };
    return if $@ or ref($res) ne 'HASH' or !exists $res->{'tokumxVersion'} or $res->{'tokumxVersion'} lt '1.5.0';
    eval { $self->run_command(['create' => $name, 'primaryKey' => $primaryKey, 'partitioned' => 1]) }
}

sub addCollection {
    my ($self, $name, $capped, $autoIndexId, $size, $max) = @_; $capped ||= 0; $autoIndexId ||= 0;
    return unless isValidConnection($self) and $name;
    my %h = ('create' => $name);
    if ($capped) {
        return unless $size;
        @h{'capped', 'size'} = ($capped, $size);
    }
    $h{'autoIndexId'} = $autoIndexId if $autoIndexId;
    $h{'size'} = $size if $size;
    $h{'max'} = $max if $max;
    eval { $self->run_command([%h]) }
}

sub dropCollection {
    my ($self, $name) = @_;
    return unless isValidConnection($self) and $name;
    eval { $self->run_command(['drop' => $name]) }
}

sub AUTOLOAD {
    my ($self, $program) = (shift(), our $AUTOLOAD);
    return unless isValidConnection($self);
    $program =~ s/.*:://;
    $self->{'db'}->$program(@_)
}

sub DESTROY {
    my $self = shift();
    return unless isValidConnection($self);
    delete $self->{'db'}
}

END { closelog() }
1;
