#!/usr/bin/perl
# -*- Encoding: utf-8; Mode: cperl -*-
# kate: space-indent on; indent-width 4; replace-tabs on;
#
package SO::RedisDatabase;
use base qw(Exporter);

use strict;
use warnings;

BEGIN { push @INC, '/usr/lib/perl5', '/usr/lib/perl5/x86_64-linux-thread-multi'  }

use Redis;
use SO::IniFile qw(loadIni);
use SO::LogFile;
use Net::HTTP;

our @EXPORT = qw(isValidConnection selectMaster $defaultIniFile $defaultIniFileSection);
our @EXPORT_OK = qw(isValidConnection isMaster selectMaster switchDB $defaultIniFile $defaultIniFileSection $errorsLogFile);
our $VERSION = 1.00;
our $defaultIniFile = '/etc/spamooborona/so-redisdb.ini';
our $defaultIniFileSection = 'redis';
our $errorsLogFile = '/var/log/so-logs/so-redisdb-errors.log';
my ($redisDefaultHost, $redisDefaultPort, $redisDefaultDatabase) = ('localhost', 6379, 0);

# Local functions

sub getHosts4Group {
    my ($group, @res) = shift();
    my $http = Net::HTTP->new('Host' => 'c.yandex-team.ru', 'HTTPVersion' => '1.0', 'KeepAlive' => 0, 'Timeout' => 1);
    unless ($http) {
        writeLog('getHosts4Group: cannot create HTTP object. Reason: '.$@);
        return wantarray() ? @res : \@res;
    }
    $http->write_request('GET' => '/api-cached/groups2hosts/'.$group);
    my ($code, $mess, %h) = eval { $http->read_response_headers; };
    my ($answer, $buf, $size) = ('', '', 0);
    if ($@) { writeLog("getHosts4Group: $@") }
    elsif ($code < 300) {
        while (1) {
            $size = $http->read_entity_body($buf, 1024);
            last unless $size;
            $answer .= $buf;
        }
        foreach (split /\n/, $answer) {
            next unless /(\S+)/;
            push @res, $1;
        }
    } else { writeLog("getHosts4Group: $code $mess") }
    wantarray() ? @res : \@res
}

# Methods of class

sub new {
    my $class = shift();
    my $self  = bless {}, $class;
    my ($iniFile, $q, $host) = (SO::IniFile->new(shift() || $defaultIniFile), {}, `hostname`); chomp $host;
    $iniFile->loadIni($q, $defaultIniFileSection);
    $SO::LogFile::defaultLogFile = $self->{'logFile'} = $$q{'errorsLogFile'} || $errorsLogFile;
    $self->{'db_number'} = $$q{'db'} || $redisDefaultDatabase;
    $self->{$_} = $q->{$_} foreach (grep {$_ ne 'db' and !$self->{$_}} keys %$q);
    $self->{'hosts'} ||= '';
    $self->{'hosts'} = join(' ', @{getHosts4Group($$q{'cluster'}) || []}) || $$q{'hosts'} if $$q{'cluster'};
    $self->{'host'}  ||= $redisDefaultHost;
    $self->{'port'}  ||= $redisDefaultPort;
    $self->{'role'}  ||= 'master';
    unless ($self->{'hosts'} and selectRole($self)) {
        $self->{'host'} = $self->{'master_host'} || $self->{'host'};
        writeLog(ucfirst($self->{'role'})." DB not found among '$$self{hosts}'. DB host '$$self{host}' will accepted.") if $self->{'hosts'};
        connectDB($self);
        $self->{'lastError'} = $@;
        writeLog("Error while establishing connection in SO::RedisDatabase::new: $@") if $@ or !$$self{'db'};
    }
    return $self;
}

sub isValidConnection {
    my $self = shift();
    ($self->{'db'} and ref($self->{'db'}) eq 'Redis' and ($self->{'db'}->ping() || '') eq 'PONG') ? 1 : 0
}

sub switchDB {
    my ($iniFileSection, $iniFilePath, $logFilePath) = @_;
    $errorsLogFile = $SO::LogFile::defaultLogFile = $logFilePath if $logFilePath;
    $defaultIniFile = $iniFilePath if $iniFilePath;
    $defaultIniFileSection = $iniFileSection if $iniFileSection;
}

sub connectDB {
    my $self = shift();
    $self->{'db'}->quit(), undef($self->{'db'}) if $self->{'db'};
    $self->{'db'} = eval { Redis->new(
        'server'    => $$self{'host'}.':'.$$self{'port'},
        'reconnect' => $$self{'reconnect'} || 30,
        'every'     => $$self{'every'} || 1000
    ) };
    if (isValidConnection($self)) {
        my $info = eval { $self->{'db'}->info('server') };
        $self->{'db'}->select($self->{'db_number'});
        $self->{'version'} = $info->{'redis_version'} || ''
    }
    $self->{'db'}
}

sub role {
    my $self = shift();
    return '' unless isValidConnection($self);
    if (join('.', map { sprintf("%03d", $_) } split(/\./, $self->{'version'})) ge '002.008.012') {
        my @res = $self->{'db'}->role();
        return wantarray() ? @res : $res[0]
    } else {
        my $info = eval { $self->{'db'}->info('replication') };
        return $info->{'role'} || $self->{'role'}
    }
}

sub selectRole {
    my ($self, $role, $hosts) = @_;
    $self->{'db'}->quit(), undef($self->{'db'}) if $self->{'db'};
    my ($host, $host_role) = `hostname`; chomp $host; $role ||= $$self{'role'}; $hosts ||= $$self{'hosts'} || $$self{'host'};
    foreach (split /,?\s+/, $hosts) {
        $self->{'host'} = ($_ eq $host ? 'localhost' : $_);
        writeLog('SO::RedisDatabase::selectRole error: "'.$@.'"'), next unless connectDB($self) and !$@;
        $host_role = role($self);
        if (role($self) eq $role) {
            writeLog("Host $_ have selected as a $role Redis DB");
            $self->{'db'}->select($self->{'db_number'}); $self->{'host'} = $_;
            last
        }
        $self->{'master_host'} = $self->{'host'} if $host_role eq 'master';
    }
    isValidConnection($self)
}

sub zIncr {
    my ($self, $key, $member, $expiretime, $addscore) = @_;
    return unless isValidConnection($self);
    $addscore ||= 1;
    eval {
        if ($self->{'db'}->exists($key)) { $self->{'db'}->zincrby($key, $addscore, $member) }
        else { zAdd($self, $key, $member, $addscore, $expiretime) }
    };
    writeLog("SO::RedisDatabase::zIncr: $@") if $@
}

sub zDecr {
    my ($self, $key, $member, $expiretime, $minus_score) = @_;
    return unless isValidConnection($self);
    $minus_score ||= 1;
    eval {
        if ($self->{'db'}->exists($key)) {
            $self->{'db'}->zincrby($key, -$minus_score, $member);
            $self->{'db'}->zrem($key, $member) if ($self->{'db'}->zscore($key, $member) || 0) < 1
        }
    };
    writeLog("SO::RedisDatabase::zDecr: $@") if $@
}

sub zAdd {
    my ($self, $key, $member, $addscore, $expiretime) = @_;
    return unless isValidConnection($self);
    eval {
        $self->{'db'}->zadd($key, $addscore, $member);
        $self->{'db'}->expireat($key, $expiretime) if $expiretime
    };
    writeLog("SO::RedisDatabase::zAdd: $@") if $@
}

sub AUTOLOAD {
    my ($self, $program) = (shift(), our $AUTOLOAD);
    return unless isValidConnection($self);
    $program =~ s/.*:://;
    $self->{'db'}->$program(@_)
}

sub DESTROY {
    my $self = shift();
    return unless isValidConnection($self);
    $self->{'db'}->quit(); delete $self->{'db'}
}

1;