# -*- encoding: koi8-r -*-
import argparse
import copy
from collections import abc
from os import listdir
import re
import itertools
import collections
import functools

delimeters_split = re.compile('[ 	()+\-|&!<>\n]')

max_cycle_len = 15


def prime_gen():
    # Initialize a list
    for possiblePrime in itertools.count(2):
        # Assume number is prime until shown it is not.
        isPrime = True
        for num in range(2, int(possiblePrime ** 0.5) + 1):
            if possiblePrime % num == 0:
                isPrime = False
                break
        if isPrime:
            yield possiblePrime


class THas(abc.MutableSet):
    first_primes = tuple([2, 3, 5, 7, 11, 13, 17, 19, 23, 29, 31, 37, 41, 43])

    def __init__(self):
        self.buckets = {}

    def add(self, val):
        val = hash(val)
        bucket_index, prime_index = divmod(val, len(THas.first_primes))

        prime = THas.first_primes[prime_index]

        bucket_val = self.buckets.setdefault(bucket_index, prime)

        if bucket_val % prime:
            self.buckets[bucket_index] = bucket_val * prime
            return True

        return False

    def has(self, val):
        val = hash(val)
        bucket_index, prime_index = divmod(val, len(THas.first_primes))

        return self.buckets.get(bucket_index, 1) % THas.first_primes[prime_index] == 0

    def discard(self, val):
        val = hash(val)
        bucket_index, prime_index = divmod(val, len(THas.first_primes))

        prime = THas.first_primes[prime_index]

        if val % prime:
            self.buckets.update({bucket_index, val / prime})

    def __len__(self):
        size = 0
        for bucket_index, bucket in self.buckets.items():
            for i, prime in enumerate(THas.first_primes):
                if bucket % prime == 0:
                    size += 1

        return size

    def __contains__(self, item):
        return self.has(item)

    def __iter__(self):
        for bucket_index, bucket in self.buckets.items():
            base = bucket_index * len(THas.first_primes)
            for i, prime in enumerate(THas.first_primes):
                if bucket % prime == 0:
                    yield base + i

    def __copy__(self):
        cp = THas()
        cp.buckets = copy.copy(self.buckets)
        return cp


class TRuleConfig(abc.Hashable):
    prime_generator = prime_gen()

    def __init__(self, path, line, name, deps, anti):

        self.path = path
        self.line = line
        self.name = name
        self.deps = [d for d in deps if not d.isdigit()]
        self.anti = anti
        self.prime = next(TRuleConfig.prime_generator)

    def __hash__(self):
        return self.prime

    def __repr__(self):
        return "{}:{} {}".format(self.path, self.line, self.name) + (" anti" if self.anti else "")

    def __lt__(self, other):
        return self.prime < other.prime


class TRule(abc.Hashable):
    def __init__(self, config:TRuleConfig):
        self.config = config
        self.masters = []
        self.deps = []

    def __repr__(self):
        return str(self.config)

    def __hash__(self):
        return hash(self.config)

    def __lt__(self, other):
        return self.config < self.config


def read_file(path)->[TRuleConfig]:
    with open(path, encoding='koi8-r') as f:
        it = iter(enumerate(f))
        for curline_index, line in it:
            if not line.startswith("rule"):
                continue

            parts = [p for p in delimeters_split.split(line) if len(p)]

            if len(parts) < 2:
                continue

            name = parts[1]
            anti = False
            deps = []
            try:
                if "R_ANTI" == parts[2]:
                    deps = parts[3:]
                    anti = True
                else:
                    _, line = next(it)
                    parts = [p for p in delimeters_split.split(line) if len(p)]
                    if parts[0] == "arithmetic" or parts[0] == "bexpr":
                        deps = parts[1:]

            except:
                pass
            yield TRuleConfig(path, curline_index, name, deps, anti)


def read_rules(root: str)->{str, TRule}:
    files = (root + f for f in listdir(root) if f.endswith(".rul") or f.endswith(".dlv"))

    configs = itertools.chain.from_iterable((read_file(p) for p in files))

    rules_by_name = {config.name: TRule(config) for config in configs}

    for name, rule in rules_by_name.items():
        if rule.config.anti:
            dep = rule.config.deps[0]

            rule.deps.append(rules_by_name.get(dep))
            rules_by_name.get(dep).masters.append(rule)

            for masterName in rule.config.deps[1:]:
                master = rules_by_name.get(masterName)
                if master is None:
                    print(masterName)
                    continue

                master.deps.append(rule)
                rule.masters.append(master)
        else:
            for depsName in rule.config.deps:
                dep = rules_by_name.get(depsName)

                if dep is None:
                    print(depsName)
                    continue
                rule.deps.append(dep)
                dep.masters.append(rule)

    return rules_by_name


class Task:
    __slots__ = ("fun", "args")

    def __init__(self, fun, args):
        self.fun = fun
        self.args = args


def detect_loops(rule: TRule, known_cycles: abc.MutableSet, stack: abc.MutableSequence, done_rules: THas)->[Task]:
    if done_rules.has(rule):
        min_stack = stack[stack.index(rule):]

        sorted_cycle = tuple(sorted(min_stack))

        if sorted_cycle not in known_cycles:
            known_cycles.add(sorted_cycle)
            if len(min_stack) < max_cycle_len:
                print("\n".join(map(str, min_stack)))
                print(rule)
                print()

        return
    else:
        done_rules.add(rule)

    stack.append(rule)

    yield from (Task(detect_loops, (d, known_cycles, copy.copy(stack), copy.copy(done_rules))) for d in rule.masters)


parser = argparse.ArgumentParser()
parser.add_argument('--max_len', default=max_cycle_len, type=int)
parser.add_argument('--rules_root')
args = parser.parse_args()

max_cycle_len = args.max_len

rules_by_name = read_rules(args.rules_root)

known_cycles = THas()
tasks = [Task(detect_loops, (rule, known_cycles, [], THas())) for rule in rules_by_name.values() if len(rule.deps) == 0]
while len(tasks) > 0:
    new_tasks = []
    for task in tasks:
        new_tasks.extend(task.fun(*task.args))
    tasks = new_tasks
