#!/usr/bin/python2
#-*- coding: utf-8 -*-
# kate: space-indent on; indent-width 4; replace-tabs on;
#
import os, os.path, re, time, dateutil.parser, imaplib
from datetime import datetime, timedelta
from email import message_from_string
from email.parser import HeaderParser
from email.header import decode_header
from collections import defaultdict
from log_utils import writelog
from smtplib import SMTP
from common import CFG, RE, default, to_utf8


def parseHeadersAsDict(msgheader):
    found = re.findall(r"(?:^|\n)([^:\s]+):[^\S\n]*(.*?)(?=\n\S+|\n\s*\n|$)", msgheader, re.S)
    return defaultdict(str, dict(map(lambda (k, v): (k.lower(), v.rstrip()), found)))


def getHeadersAsStr(msg):
    msgheader = ''
    for line in msg.splitlines():
        if not line:
            break
        else:
            msgheader += line + "\n"
    return msgheader


def getMxQueueID(msgheader):
    # For algorithm info see: https://st.yandex-team.ru/SODEV-1893
    context = {
        "domain": "",
        "ip":     "",
        "by":     "",
        "mx":     "",
        "queueid": "",
        "cmail":   "",
        "backend": "",
        "rcpt":    "",
        "forward": False
    }
    def processReceived(c, t):
        c["domain"], c["ip"], c["mx"], c["cmail"], c["backend"], c["queueid"] = t[0], t[1], t[2], t[3], t[4], t[5]
        if t[6]:
            if not c["rcpt"]:
                c["rcpt"] = t[6]
            elif c["rcpt"] != t[6]:
                c["forward"] = True

    for domain, ip, by, mx, cmail, backend, queueid, rcpt in RE["mx_queueid"].findall(msgheader):
        if RE["yandex_net"].match(domain):
            if not context["domain"] or context["domain"] == by:
                processReceived(context, (domain, ip, mx, cmail, backend, queueid, rcpt))
            else:
                break
        else:
            processReceived(context, (domain, ip, mx, cmail, backend, queueid, rcpt))
            break
    if context["backend"]:
        context["mx"] = context["backend"]
    elif not context["mx"]:
        context["mx"] = 'smtp'
    return context["mx"], context["queueid"], context["cmail"], context["domain"], context["ip"], context["forward"]


def getMsgID(headers):
    m = headers.get("message-id", "").strip().split()
    return m[0].strip("<>") if len(m) > 0 else ''


def getRoute(msgheader, mx, source):
    if mx.endswith("corp") or source == "exchange" or source == "so":
        return "corp"
    elif source == "fblin" or mx.startswith("smtp") or re.search(r'^X-Mailer: Yamail', msgheader, re.M | re.I):
        return "out"
    elif mx == "mxfront" or mx == "yaback" or mx == "outback":
        return "in"
    m = re.search("^X-Yandex-Front: ([a-z\-]+)[^\.]+\.mail\.yandex\.net", msgheader, re.M)
    if not m:
        m = re.search("^Authentication-Results: ([a-z\-]+)[^\.]+\.mail\.yandex\.net", msgheader, re.M)
    if m and m.group(1).endswith('corp'):
        return "corp"
    if m and m.group(1) == 'mxback' and (re.search("^Received: by [a-z0-9-]+\.qloud-c\.yandex\.net with HTTP;", msgheader, re.M) or mx == 'smtp' or mx == 'web'):
        return "out"
    return "in"


def getMsgTime(headers, params={}):
    timestamp = headers.get("x-yandex-timemark", "")
    try:
        if timestamp:
            params["data"] = timestamp
            return int(float(timestamp))
        elif "data" in params:
            return int(float(params.get("data", 0)))
        elif "date" in headers:
            return int(dateutil.parser.parse(headers["date"]).strftime("%s"))
        else:
            return 0
    except:
        return 0


def parseMsgAddresses(msg):
    match = re.search(r"^From:\s[^\n\r]*?<(%s)>" % RE['email'].pattern, msg, re.IGNORECASE | re.MULTILINE)
    if not match:
        match = re.search(r"^From:\s[^\n\r]*?(%s)" % RE['email'].pattern, msg, re.IGNORECASE | re.MULTILINE)
        if not match:
            match = re.search(r"^From:\s[^\n\r]*?(%s)" % RE['safe_email'].pattern, msg, re.IGNORECASE | re.MULTILINE)
    from_addr = match.group(1) if match else ""
    match = re.search(r"^To:\s[^\n\r]*?<(%s)>" % RE['email'].pattern, msg, re.IGNORECASE | re.MULTILINE)
    if not match:
        match = re.search(r"^To:\s[^\n\r]*?(%s)" % RE['email'].pattern, msg, re.IGNORECASE | re.MULTILINE)
        if not match:
            match = re.search(r"^To:\s[^\n\r]*?(%s)" % RE['safe_email'].pattern, msg, re.IGNORECASE | re.MULTILINE)
    to_addr = match.group(1) if match else ""
    return from_addr, to_addr


def getSoResolution(headers, suid):
    so_res = default(headers.get("x-yandex-spam", '-'), '-')
    if so_res == '-':
        so_res = default(headers.get("x-spam-flag", '-').lower(), '-')
        if so_res != '-':
            so_res = '4' if so_res == 'yes' else '1'
    if suid != '-' and 'x-yandex-suid-status' in headers:
        m = re.search(r'\b(\d+)\s+{0}\b'.format(suid), headers.get('x-yandex-suid-status', ''))
        if m:
            so_res = m.group(1)
    return so_res


def decodePart(part):
    try:
        if part[1]:
            return part[0].decode(part[1], "ignore")
    except:
        pass
    return part[0].decode("ascii", "ignore")


def decodeHeader(s):
    if not s:
        return ''
    try:
        parts = decode_header(s)
        if not parts:
            return s.decode("ascii", "ignore").strip()
        return " ".join(map(decodePart, parts)).strip()
    except:
        return s.decode("ascii", "ignore").strip()


def decodeEmail(email_field):
    if not email_field:
        return '', ''
    emails = RE["email_field"].findall(email_field)
    msgto = re.sub("\"|'|,", "", RE["email_field"].sub("", email_field)).strip()
    try:
        res = decodeHeader(msgto), ", ".join(map(lambda eml: eml.strip().strip("<>"), emails))
    except:
        res = msgto.replace("\n", " ").replace("\t", " "), ", ".join(map(lambda eml: eml.strip().strip("<>"), emails))
    return res


def unsignedHash(val):
    h = hash(val)
    if h < 0:
        return h + 2**64
    return h


def parseEmailAddress(s):
    match = re.search("({0})".format(RE['email'].pattern), s, re.IGNORECASE)
    try:
        return match.group(1) if match else decodeHeader(s.strip())
    except:
        return s.strip()


def getMsgParams(data, complrequest):
    msg = message_from_string(data)
    try:
        subj = decodeHeader(msg["subject"]).encode("utf-8", "ignore")
    except:
        pass
    from_label, from_email = map(lambda elem: to_utf8(elem), decodeEmail(msg["from"]))
    to_label, to_email = map(lambda elem: to_utf8(elem), decodeEmail(msg["to"]))
    try:
        date = int(dateutil.parser.parse(decodeHeader(msg["date"])).strftime("%s"))
    except:
        date = int(time.time())
    params = {"data_id": unsignedHash(complrequest), "subj": subj, "from_label": from_label,
              "from_email": from_email, "to_label": to_label, "to_email": to_email, "msgdate": date}
    return params


def sendEmail(msg, fromaddr=CFG['robot']['name'], toaddr=CFG['report'], log_fh=None):
    try:
        if toaddr:
            server = SMTP(CFG['smtp']['host'], CFG['smtp']['port'], timeout=CFG['smtp']['timeout'])
            server.sendmail(fromaddr, toaddr, msg)
            server.quit()
        else:
            s = "Sending email failed: to-addr must be set!"
            writelog(s, False, log_fh)
            return s
    except Exception, e:
        s = "Error in sendEmail: %s" % str(e)
        writelog(s, True, log_fh)
        return s
    return "OK"


def sendEmailSafe(msg_text, fromaddr, toaddr, log_fh=None):
    msg = message_from_string(msg_text)
    return sendEmail(msg.as_string(), msg['From'] if 'From' in msg and msg['From'] else fromaddr, msg['To'] if 'To' in msg and msg['To'] else toaddr, log_fh)


def sendMsgToIMAPFolder(msg, acttime=None, folder_name=CFG["imap_corp"]["folder"], isImap=False,
                        user=CFG['robot']['user'], password=CFG['robot']['password'], log_fh=None):
    status = ""
    if isImap:
        if not acttime:
            acttime = int(time.time())
        if not user or not password:
            s = "Invalid IMAP user's credentials: user=%s, passwd=%s." % (user, password)
            writelog(s, False, log_fh)
            status += ('\n' if status else '') + s
        try:
            imap = imaplib.IMAP4_SSL(CFG['imap_corp']['in_host'], CFG['imap_corp']['in_port'])
            loginInfo = imap.login(user, password)
            if loginInfo[0] == "NO":
                s = "Authenticating for user '%s' IMAP failed: %s." % (user, str(loginInfo))
                writelog(s, False, log_fh)
                status += ('\n' if status else '') + s
            else:
                folderInfo = imap.select(folder_name)
                if folderInfo[0] == "NO" and folderInfo[1][0].find("No such folder") > -1:
                    createInfo = imap.create(folder_name)
                    s = "Creating IMAP folder '%s' result: %s." % (folder_name, str(createInfo))
                    writelog(s, False, log_fh)
                    if createInfo[0] == "NO":
                        status += ('\n' if status else '') + s
                appendInfo = imap.append(folder_name, '', imaplib.Time2Internaldate(acttime), msg)
                if appendInfo[0] == "NO":
                    s = "Appending of message to IMAP folder '%s' failed: %s." % (folder_name, str(appendInfo))
                    writelog(s, False, log_fh)
                    status += ('\n' if status else '') + s
                imap.logout()
        except Exception, e:
            s = "Senging message to corp IMAP folder '%s' failed: %s." % (folder_name, str(e))
            writelog(s, True, log_fh)
            status += ('\n' if status else '') + s
    else:
        status = sendEmail(msg, toaddr="{0}@yandex-team.ru".format(folder_name), log_fh=log_fh)
    return status


def deleteMsgsFromIMAPFolder(stids, filter_header="X-Yandex-STID", folder_name=CFG["imap_corp"]["folder"],
                             user=CFG['robot']['user'], password=CFG['robot']['password'], log_fh=None):
    status = ""
    try:
        if not user or not password:
            s = "Invalid IMAP user's credentials: user=%s, passwd=%s." % (user, password)
            writelog(s, False, log_fh)
            status += ('\n' if status else '') + s
        imap = imaplib.IMAP4_SSL(CFG['imap_corp']['in_host'], CFG['imap_corp']['in_port'])
        imap.login(user, password)
        imap.select(folder_name)
        resp, data = imap.search(None, "ALL")
        uids = data[0].split()
        mailparser = HeaderParser()
        for uid in uids:
            resp, data = imap.fetch(uid, "(BODY.PEEK[HEADER])")
            msg = mailparser.parsestr(data[0][1])
            if filter_header in msg and msg[filter_header] in stids:
                imap.store(uid, '+FLAGS', '(\\Deleted)')
        imap.expunge()
        imap.logout()
    except Exception, e:
        writelog("Deleting messages from corp IMAP folder '%s' failed: %s." % (folder_name, str(e)), True, log_fh)
        status = "Error: '%s'" % str(e)
    return status
