#!/bin/bash

# ------------------------------------------------------------------------------
# Variables
# ------------------------------------------------------------------------------
# Antifraud rules reader
READER_PATH="./antifraud_rules_reader.sh"

# Repository's folder
repo_dir=$PWD

# Count of errors found in commits
errors=0

# Commit sha with all zeros
zero_commit='0000000000000000000000000000000000000000'

# Temp.file containing patch
patchfile=$(mktemp git.diff.XXXXXXXX)

# Temp.dir containing all files of the repository
temp_repo_dir=$(mktemp -d git.repo.new.state.XXXXXXXX)

# Variable with the result text from checking up process
result_text=""

# ------------------------------------------------------------------------------
# Pre-receive hook
# ------------------------------------------------------------------------------
while read oldrev newrev refname; do
  [ "$refname" != "refs/heads/master" ] && continue

  # # Debug payload
  # echo -e "${oldrev} ${newrev} ${refname}\n"

  # ----------------------------------------------------------------------------
  # Get the diff for all the commits
  # ----------------------------------------------------------------------------

  # Check if a zero sha
  if [ "${oldrev}" = "${zero_commit}" ]; then
    # List everything reachable from newrev but not any heads
    git diff $(git for-each-ref --format='%(refname)' refs/heads/* | sed 's/^/\^/') ${newrev} > $patchfile
  else
    git diff ${oldrev}..${newrev} > $patchfile
  fi

  # ----------------------------------------------------------------------------
  # Copy repository files to the side
  # ----------------------------------------------------------------------------
  mkdir -p $temp_repo_dir
  git --git-dir=$GIT_DIR --work-tree=$temp_repo_dir checkout master .

  # ----------------------------------------------------------------------------
  # Patch files of the copied repository
  # ----------------------------------------------------------------------------
  cd $temp_repo_dir
  patch -p1 < $patchfile

  # ----------------------------------------------------------------------------
  # Check repository up
  # ----------------------------------------------------------------------------
  result_text=$($READER_PATH $temp_repo_dir)
  [ "$?" -gt "0" ] && (( errors++ ))
  fault=$(cat $result_text | grep 'All fault rules: ' | sed -e 's#*All fault rules: ##')
  [ "$fault" -gt "0" ] && (( errors++ ))

  # ----------------------------------------------------------------------------
  # Clean temporary objects up
  # ----------------------------------------------------------------------------
  rm -rf $temp_repo_dir
  rm -f $patchfile
  cd $repo_dir
done

# ------------------------------------------------------------------------------
# Verify count of found errors
# ------------------------------------------------------------------------------
if [ "$errors" -gt "0" ]; then
  # Found errors, exit with error
  echo "[POLICY BLOCKED] You're trying to commit broken SO Antifraud rules."
  echo "Please, look for the possible reason in this:"
  echo "Ckecking process result: $result_text"
  exit 1
else
  # No errors found, exit with success
  exit 0
fi
