#!/usr/bin/python2
# encoding: utf-8
# kate: space-indent on; indent-width 4; replace-tabs on;
#
import os, os.path, sys, re, urlparse, time, ConfigParser, uwsgi
sys.path.insert(0, os.path.dirname(os.path.abspath(__file__)))
sys.path.insert(0, 'WORKING_DIR')
sys.path.insert(0, 'WORKING_DIR/web')
from IPy import IP, IPSet
from collections import defaultdict
from random import randrange
from log_utils import writelog
from common import RE, LOCK, getUUID
from db_utils import DB, getPGCredentials, getPGdb
from email_utils import sendEmailSafe


INI_FILE = '%s/so-fbl-out.ini' % os.path.dirname(os.path.abspath(__file__))
FROM_ADDR = "fbl-noreply@yandex.ru"
KASPER1_RE = re.compile(r'\blua[ -]profiles\b', re.I)
KASPER2_RE = re.compile(r'\bluacore\b', re.I)
MESS_RE = re.compile(r'<b>mess:</b>\s', re.I | re.S)
RFC822_RE = re.compile(r'\bmessage/rfc822\b', re.I)
HEADER_RE = re.compile(r'^([\w\-]+)\s*:\s+(.*)$', re.I)
HEADER_CONTINUE_RE = re.compile(r'^\s+(\S+.*)$')
BOUNDARY_RE = re.compile(r'\bboundary\s*=\s*(["\'])([^"\']*?)\1', re.I)
KASP_RE = re.compile(r'\br_nl:(?:</b>)?\s.*?\bTO_KASP_FBL\b', re.I | re.S)
IMAP_RE = re.compile(r'\s+with\s+IMAP\s+', re.I)
POP3_RE = re.compile(r'\s+with\s+POP3\s+', re.I)
QUEUEID_RE = re.compile(r'by ([a-z-]+)[0-9a-z-]+?\.mail\.yandex\.net.+?with E?SMTPS?A?\s+id\s+(\w+-\w+);')
YANDEX_SENDER_RE = re.compile(r'^X-Mailer: YandexSender', re.M | re.I)
FILE_NAME_RE = re.compile(r'(^.*\bfilename=")[\w\.\-]+(\.eml".*)$', re.S)
SOURCE_IMAP_RE = re.compile(r'\&source=imap(?:\&|\s)', re.I)
MOVE_RE = re.compile(r'\&move=1(?:\&|\s)', re.I)
SPAM_TYPE_RE = re.compile(r'\&type=([a-z]+)(?:\&|\s)', re.I)


def metric_inc(metric):
    if uwsgi.mule_id() > 0:
        uwsgi.metric_inc("mule.%s.%s" % (uwsgi.mule_id(), metric))
    else:
        uwsgi.metric_inc("worker.%s.%s" % (uwsgi.worker_id(), metric))


def loadSettings(iniFile):
    cfg = {}
    try:
        ini = ConfigParser.SafeConfigParser({})
        ini.optionxform = str
        ini.read(iniFile)
        for section in ini.sections():
            if section == "uwsgi":
                continue
            cfg[section] = dict(ini.items(section)) if ini.has_section(section) else {}
    except ConfigParser.Error, e:
        writelog("Error while parsing config file: '%s'" % str(e), True)
        sys.exit(1)
    return cfg


class FBLMsgContext:
    def __init__(self, msg, processingInfo, processingType):
        self.processingInfo = processingInfo
        self.processingType = processingType
        self.complType = "foo"
        self.route = "in"
        self.trusted = False
        self.selected = False if len(self.processingInfo.SelectDomains) > 0 and not self.processingInfo.sendOtherTo else True
        self.feedbackDomain = ""
        self.subdomain = ""
        self.xSentTo = False
        self.mx = ""
        self.allHeaders = False
        self.isValid = True
        self.boundarySepOutMsg = "=_NextPart_%s" % getUUID('.')
        self.originalTo = ''
        self.fields = {
            "subject":      "Fwd: ",
            "originalfrom": "",
            "originalto":   "",
            "originaldate": time.strftime("%a, %d %b %Y %H:%M:%S %z"), # Thu, 13 Aug 2020 20:00:18 +0300
            "from":         " ",
            "to":           " ",
            "content-type": 'multipart/report; boundary="%s"; report-type="feedback-report"' % self.boundarySepOutMsg,
            "message-id":   "<%s@fbl-out.yandex.ru>" % getUUID(''),
            "mime-version": "1.0"
        }
        self.fieldNames = {"from": "From", "to": "To", "subject": "Subject", "message-id": "Message-ID", "mime-version": "MIME-Version", "content-type": "Content-Type"}
        self.entities = []
        self.betweenHeadersAndBody = ""
        if processingType != "common" and self.processingInfo.allComplaintsReceivers[processingType]["allHeaders"]:
            self.allHeaders = True
        self.isDeliveryLogPresent = False
        self.isKasperskyByRules = True
        self.parseMsg(msg)

    def parseMsg(self, msg):
        tag, isHeader, boundarySep = "", True, ""
        for line in re.split(r"\r?\n", msg):
            if isHeader:
                m = HEADER_CONTINUE_RE.match(line)
                if m:
                    s += "\n\t" + m.group(1)
                    continue
                else:
                    if tag:
                        if tag == 'iy-complrequest':
                            m = SPAM_TYPE_RE.search(s)
                            if not m:
                                self.selected = False
                                break
                            self.complType = m.group(1)
                            if MOVE_RE.search(s) or SOURCE_IMAP_RE.search(s):
                                self.selected = False
                                break
                        elif tag == 'subject':
                            self.fields['subject'] = 'Fwd: ' + s
                        elif tag == 'mime-version':
                            self.fields['mime-version'] = s
                        elif tag == 'date':
                            self.fields['originaldate'] = s
                        elif tag == 'from':
                            m = RE['email'].search(s)
                            if m:
                                self.feedbackDomain = m.group(0)
                            if not self.feedbackDomain and self.processingType == "common": # delete complaint if From: doesn't contain valid email
                                writelog("Wrong sender (field From:): '%s'" % s, fh=self.processingInfo.error_log)
                                isValid = False
                                break
                            if RE['yandex_domain'].search(self.feedbackDomain):
                                self.route = 'out'
                            self.fields['originalfrom'] = s
                        elif tag == 'to':
                            self.fields['originalto'] = s
                            self.originalTo = s
                        elif tag == 'content-type':
                            m = BOUNDARY_RE.search(s)
                            boundarySep = m.group(2) if m else ''
                        elif tag == 'x-yandex-route':
                            if re.match(r'[a-z\-]+', s.split()[0]):
                                self.route = s.split()[0]
                    m = HEADER_RE.match(line)
                    if m:
                        tag = m.group(1).lower()
                        self.fieldNames[tag] = m.group(1)
                        s = m.group(2)
                        continue
                    elif re.match(r'^\s*$', line):
                        self.betweenHeadersAndBody += "%s\n" % line
                        if isHeader:
                            isHeader, tag = False, ''
                        continue
            if self.complType != 'foo' or not self.fields['originalfrom'] and self.processingType == "common":
                isValid = False
                break
            if line.startswith('--') and line.find(boundarySep) >= 0:
                self.entities.append([])
                continue
            if len(self.entities) < 1:
                self.betweenHeadersAndBody += line + "\n"
                continue
            self.entities[-1].append(line + "\n")


class FBLOutGenerator:
    def __init__(self, cfg, dbInfo):
        self.cfg = cfg.copy()
        self.dbInfo = dbInfo
        self.TrustedDomainsRE = []
        self.TrustedDomains = cfg['params']['trustedDomains'].split() if "params" in cfg and "testExcludeDomains" in cfg['params'] and cfg['params']['trustedDomains'] else []
        self.SelectDomains = cfg['params']['sendToDomains'].split() if "params" in cfg and 'sendToDomains' in cfg['params'] and cfg['params']['sendToDomains'] else []
        self.SelectDomainsRE = []
        self.ExcludeDomains = cfg['params']['excludedDomains'].split() if "params" in cfg and 'excludedDomains' in cfg['params'] and cfg['params']['excludedDomains'] else []
        self.ExcludeDomainsRE = []
        self.allHeadersDomainsRE = []
        self.allHeadersDomains = cfg['params']['allHeadersDomains'].split() if "params" in cfg and "allHeadersDomains" in cfg['params'] and cfg['params']['allHeadersDomains'] else []
        self.testExclDomains = {}
        self.sendOtherTo = cfg["params"]["sendOtherTo"] if "params" in cfg and "sendOtherTo" in cfg["params"] and cfg["params"]["sendOtherTo"] else ""
        self.testSendTo = cfg['params']['testSendTo'] if "params" in cfg and "testSendTo" in cfg["params"] and cfg["params"]["testSendTo"] else ""
        self.testSendTimeout = int(cfg['params']['testSendTimeout']) if "params" in cfg and cfg['params'].get('testSendTimeout', '').isdigit() else 1
        self.Domains = defaultdict(lambda: {})
        self.allComplaintsReceivers = {}
        for section in cfg:
            if section == "params" or section == "providers":
                continue
            self.allComplaintsReceivers[section] = {
                "email":      cfg[section]["email"] if "email" in cfg[section] else "",
                "allHeaders": True if "allHeaders" in cfg[section] and cfg[section]["allHeaders"] == "true" else False
            }
        self.yandexFrom = cfg['params']['yandexFrom'] if "params" in cfg and "yandexFrom" in cfg['params'] else ""
        self.replaceStr = cfg['params'].get('replaceStr', '') if "params" in cfg else ""
        self.replaceDomainStr = cfg['params'].get('replaceDomainStr', '') if "params" in cfg else ""
        self.logFilePath = cfg['params']['logFile'] if "params" in cfg and 'logFile' in cfg['params'] and cfg['params']['logFile'] else '/logs/current-fbl-out.log'
        self.error_log = self.fbl_log = None
        try:
            self.complaintsDb = getPGdb(dbInfo)
            self.error_log = open(cfg['params']['errorsLogFile'] if "params" in cfg and 'errorsLogFile' in cfg['params'] else '/logs/current-fbl-out-errors.log', "a+t")
            self.fbl_log = open(self.logFilePath, "a+t")
            for (k, v) in cfg['providers'].items():
                m = re.match(r'subnet_(.*)', k)
                if m:
                    if 'subnet' not in self.Domains[m.group(1)]:
                        self.Domains[m.group(1)]['subnet'] = IPSet([])
                    for subnet in v.split():
                        self.Domains[m.group(1)]['subnet'].add(IP(subnet))
                    continue
                self.Domains[k]['email'] = v
            if "params" in cfg and "domainsXSendTo" in cfg['params']:
                for domain in cfg['params']['domainsXSendTo'].split():
                    self.Domains[domain]['x-sent-to'] = 1
            if "params" in cfg and "testExcludeDomains" in cfg['params']:
                for domain in cfg['params']['testExcludeDomains'].split():
                    self.testExclDomains[domain] = 1
            for domain in self.SelectDomains:
                self.SelectDomainsRE.append(re.compile(r'\b>{0}\b'.format(re.escape(domain)), re.I))
            for domain in self.ExcludeDomains:
                self.ExcludeDomainsRE.append(re.compile(r'\b{0}\b'.format(re.escape(domain)), re.I))
            for domain in self.TrustedDomains:
                self.TrustedDomainsRE.append(re.compile(r'\b{0}\b'.format(re.escape(domain)), re.I))
            for domain in self.allHeadersDomains:
                self.allHeadersDomainsRE.append(re.compile(r'\b{0}\b'.format(re.escape(domain)), re.I))
        except Exception, e:
            writelog("FBLOutGenerator's initialization failed: %s" % str(e), True, self.error_log)

    def generateOutMsg(self, context):
        curDate = time.strftime("%a, %d %m %Y %H:%M:%S %z")
        while True:
            header, body, subdomain = '', '', ''
            # preparing of body
            for entity in context.entities:
                entity2 = self.parseEntity("".join(entity), context)
                if not entity2:
                    continue
                if not context.selected and context.processingType == "common":
                    break
                body += entity2
            if not context.selected and context.processingType == "common":
                break
            if len(context.entities) > 0:
                body += "\n--%s--\n" % context.boundarySepOutMsg
            if context.originalTo:
                m = re.search(r'\s*<?(%s)>?;?\s*' % RE['email'].pattern, context.originalTo)
                if m:
                    originalTo = m.group(1)
            if context.processingType != "common" and context.originalTo:
                body = body.replace(context.originalTo, self.replaceStr)
            # preparing of headers
            m = re.search(r'\s*<?(%s)>?;?\s*' % RE['email'].pattern, context.fields['originalfrom'])
            context.fields['sender'] = m.group(1) if m else '|%s|' % context.fields['originalfrom']
            m = re.search(r'\s*<?(%s)>?;?\s*' % RE['email'].pattern, context.fields['originalto'])
            context.fields['recipient'] = m.group(1) if m else '|%s|' % context.fields['originalto']
            if not (context.trusted or context.xSentTo):
                context.fields['originalto'] = re.sub(r'(?i)(\s*<?)(%s)(>?;?\s*)' % RE['yandex_email'].pattern, \
                    r'\1{0}\3'.format(self.replaceStr), context.fields['originalto'])
            context.fields['from'] = self.yandexFrom
            if context.processingType == "common":
                context.fields['to'] = self.Domains[context.feedbackDomain].get('email', '')
                if not context.fields['to']:
                    context.fields['to'] = self.sendOtherTo
                    #if not context.fields['to']:
                    #    context.fields['to'] = 'feedback@' + context.feedbackDomain
            else:
                context.fields['to'] = self.allComplaintsReceivers[context.processingType]["email"]
            if not context.fields['to']:
                context.isValid = False
            for h in ['from', 'to', 'subject', 'message-id', 'mime-version', 'content-type']:
                header += (context.fieldNames[h] if context.fieldNames[h] else h.capitalize()) + ": " \
                    + (context.fields[h] if context.fields[h] else '') + "\n"
            # generating of the outgoing FBL-message
            return header + (context.betweenHeadersAndBody if context.betweenHeadersAndBody else '') + """
--{0}
MIME-Version: 1.0
Content-Disposition: inline
Content-Transfer-Encoding: 7bit
Content-Type: text/plain; charset="US-ASCII"
Date: {1}

This is an email abuse report for an email message received from {2} on {3}

--{0}
MIME-Version: 1.0
Content-Disposition: inline
Content-Transfer-Encoding: 7bit
Content-Type: message/feedback-report

Feedback-Type: abuse
User-Agent: Yandex-Mail-Report/1.0
Version: 0.1
Original-Mail-From: {4}
{5}
Received-Date: {3}
Reported-Domain: {2}

""".format(context.boundarySepOutMsg, curDate, context.feedbackDomain, context.fields['originaldate'], context.fields['originalfrom'], \
        "Original-Rcpt-To: %s" % (context.fields['originalto'] if context.trusted else self.replaceStr)) \
                + (body if body else '')

    def sendOutMsg(self, msg, context):
        sendEmailSafe(msg, FROM_ADDR, context.fields['to'], self.error_log)
        metric_inc('fbl_out_sent')
        try:
            uwsgi.lock(LOCK['complaints_db'])
            db_cursor = self.complaintsDb.cursor()
            db_cursor.execute("SELECT value FROM settings WHERE key = 'fblLastSendTime'")
            try:
                row = db_cursor.fetchone()
            except:
                row = None
            t, t0 = int(time.time()), int(row[0]) if row else 0
            if self.testSendTo and context.feedbackDomain not in self.testExclDomains and (t0 + self.testSendTimeout <= t):
                sendEmailSafe(msg, FROM_ADDR, self.testSendTo, self.error_log)
                metric_inc('fbl_out_test_sent')
                db_cursor.execute("INSERT INTO settings (key, value) VALUES ('fblLastSendTime', '{0}') ON CONFLICT (key) DO UPDATE SET value = '{0}' WHERE settings.key = 'fblLastSendTime'".format(t))
                self.complaintsDb.commit()
            db_cursor.close()
            uwsgi.unlock(LOCK['complaints_db'])
        except Exception, e:
            writelog("DB error in sendOutMsg: %s" % str(e), True, self.error_log)
            self.complaintsDb = getPGdb(self.dbInfo)
            uwsgi.unlock(LOCK['complaints_db'])
        if self.logFilePath:
            try:
                if not context.subdomain:
                    context.subdomain = ""
                writelog(" ".join([time.strftime("%a, %d %m %Y %H:%M:%S %z"), context.feedbackDomain, context.subdomain, context.fields['sender'], context.fields['recipient'], context.fields['to']]), fh=self.fbl_log)
            except Exception, e:
                writelog("sendOutMsg failed: %s" % str(e), True, self.error_log)

    def processMsg(self, msg, processingType="common"):
        if msg:
            context = FBLMsgContext(msg, self, processingType)
            if context.isValid:
                if processingType != "common":
                    context.isValid = False
                outMsg = self.generateOutMsg(context)
                if context.isValid and (processingType == "common" and context.selected or processingType != "common" and context.isDeliveryLogPresent) \
                    and context.route == 'in' and (context.mx != 'yaback' or context.mx == 'yaback' and context.feedbackDomain == 'sender.yandex-team.ru'):
                        self.sendOutMsg(outMsg, context)

    def parseEntity(self, entity, context):
        body, header, isHeader, isHeader2, tag, s, isKaspersky = '', '', 1, 1, '', '', False
        if MESS_RE.search(entity):
            context.isDeliveryLogPresent = True
            if context.processingType != "common" and not KASP_RE.search(entity):
                context.isKasperskyByRules = False
            return ''
        entity = re.sub(r'^\s+', '', entity)
        if not entity:
            return ''
        fieldNames, tags, tagvals = {}, [], []
        context.validEntity = False
        for line in entity.split("\n"):
            if isHeader:
                m = HEADER_CONTINUE_RE.match(line)
                if m:
                    s += "\n\t%s" % m.group(1)
                    continue
                else:
                    if tag:
                        if tag == 'content-type':
                            if not RFC822_RE.search(s):
                                return ''
                            if YANDEX_SENDER_RE.search(entity):
                                context.feedbackDomain = 'sender.yandex-team.ru'
                        if tag == 'content-disposition':
                            s2 = getUUID('_');
                            m = FILE_NAME_RE.search(s)
                            s = (m.group(1) if m else 'attachment;\n\tfilename="') + s2 + (m.group(2) if m else '.eml"')
                        header += fieldNames[tag] + ': ' + s + "\n"
                    m = HEADER_RE.match(line)
                    if m:
                        tag = m.group(1).lower()
                        fieldNames[tag] = m.group(1)
                        s = m.group(2)
                        continue
                    elif re.match(r'^\s*$', line):
                        header += "\n"
                        if isHeader:
                            isHeader, tag = 0, ''
                        continue
            if isHeader2:
                if self.allHeadersDomains and not context.allHeaders:
                    for domain in self.allHeadersDomainsRE:
                        if re.search(r'\b{0}\b'.format(domain), context.feedbackDomain):
                            context.allHeaders = True
                m = HEADER_CONTINUE_RE.match(line)
                if m:
                    s += "\n\t" + m.group(1)
                    continue
                else:
                    if tag:
                        if tag == 'from' or tag == 'mime-version' or tag == 'date' or tag == 'subject' or tag == 'content-type' or tag == 'x-sent-to' or \
                                tag == 'content-transfer-encoding' or tag == 'reply-to' or tag == 'return-path' or tag == 'to' or tag == 'message-id':
                            tags.append(tag)
                            tagvals.append(s)
                        elif tag == 'received':
                            if (IMAP_RE.search(s) or POP3_RE.search(s)) and re.search(r'\b{0}\b'.format(RE['yandex_domain'].pattern), s, re.I):
                                context.selected = False
                                return ''
                            m = QUEUEID_RE.search(s)
                            if m:
                                context.mx = m.group(1)
                            if context.mx and context.mx != 'yaback' and context.feedbackDomain == 'sender.yandex-team.ru':
                                context.feedbackDomain = ''
                            if context.feedbackDomain == 'sender.yandex-team.ru':
                                if context.mx:
                                    context.selected = context.trusted = context.validEntity = True
                                    context.route = 'in'
                            elif context.processingType == "common":
                                s2 = s
                                s2 = re.sub(r'(?i)([\s\(]+)(%s(?:\s\(?\[%s\]\)?)?)([;:\)]?\s+)' % (RE['yandex_domain'].pattern, RE['ip'].pattern), r'\1{0}\3'.format(self.replaceDomainStr), s2)
                                s2 = re.sub(r'(?i)(\s+<?)(%s)(>?;?\s+)' % RE['yandex_email'].pattern, r'\1{0}\3'.format(self.replaceStr), s2)
                                s2 = s2.replace(context.originalTo, self.replaceStr)
                                m = re.search(r'from\s+(%s)\s*(?:\(?.*?\[(%s)\]\))?' % (RE['domain'].pattern, RE['ip'].pattern), s2, re.I)
                                if re.search(r'by\s+%s\b' % RE['yandex_domain'].pattern, s, re.I) and m:
                                    context.subdomain = m.group(1)
                                    self.verifyDomain(m.group(1), m.group(2), context, 'received')
                            else:
                                context.validEntity = True
                            tags.append('received')
                            tagvals.append(s)
                        elif tag == 'authentication-results':
                            tags.append('authentication-results')
                            tagvals.append(s)
                        elif (tag == 'x-yandex-pop-server' or tag == 'x-yandex-rpop-id' or tag == 'yandex-rpop-info') and context.processingType == "common":
                            context.selected = False
                            return ''
                        elif context.allHeaders and not tag.startswith("x-yandex"):
                            tags.append(tag)
                            tagvals.append(s)
                    m = HEADER_RE.match(line)
                    if m:
                        tag = m.group(1).lower()
                        fieldNames[tag] = m.group(1)
                        s = m.group(2)
                        continue
                    elif re.match(r'^\s*$', line):
                        body += "\n"
                        if isHeader2:
                            isHeader2, tag = 0, ''
                        continue
            if not context.trusted and context.processingType == "common":
                break
            body += line + "\n"
        if not context.validEntity:
            context.selected = False
            return ''
        if not context.subdomain:
            context.subdomain = ''
        for i in range(len(tags)):
            if context.processingType == "kaspersky":
                if (KASPER1_RE.search(tags[i]) or KASPER1_RE.search(tagvals[i]) or KASPER2_RE.search(tags[i]) or KASPER2_RE.search(tagvals[i])) and context.complType == 'foo':
                    isKaspersky = True
            if tags[i] == 'received':
                if not context.trusted:
                    tagvals[i] = re.sub(r'(?i)([\s\(]+)(%s(?:\s\(?\[%s\]\)?)?)([;:\)]?\s+)' % (RE['yandex_domain'].pattern, RE['ip'].pattern), r'\1{0}\3'.format(self.replaceDomainStr), tagvals[i])
                    tagvals[i] = re.sub(r'(?i)(\s+<?)(%s)(>?;?\s+)' % (RE['yandex_email'].pattern), r'\1{0}\3'.format(self.replaceStr), tagvals[i])
                    tagvals[i] = tagvals[i].replace(context.originalTo, self.replaceStr)
            elif tags[i] == 'authentication-results':
                if not context.trusted:
                    tagvals[i] = re.sub(r'(?i)([\s\(]*)(%s)([;:\)]?\s+)' % RE['yandex_domain'].pattern, r'\1{0}\3'.format(self.replaceDomainStr), tagvals[i])
                    tagvals[i] = re.sub(r'(?i)(\s+)(\S*?\b%s\b\S*?)(\s+)' % RE['yandex_domain'].pattern, r'\1{0}\3'.format(self.replaceDomainStr), tagvals[i])
                    tagvals[i] = tagvals[i].replace(context.originalTo, self.replaceStr)
            elif tags[i] == 'return-path':
                if not context.trusted:
                    tagvals[i] = re.sub(r'(?i)(\s*<?)(%s)(>?;?\s*)' % RE['yandex_email'].pattern, r'\1{0}\3'.format(self.replaceStr), tagvals[i])
                    tagvals[i] = re.sub(r'(?i)(\s*)(\S*?\b%s\b\S*?)(\s+)' % RE['yandex_domain'].pattern, r'\1{0}\3'.format(self.replaceDomainStr), tagvals[i])
                    tagvals[i] = tagvals[i].replace(context.originalTo, self.replaceStr)
            elif tags[i] == 'to' or tags[i] == 'cc' or tags[i] == 'bcc':
                if context.processingType != "common":
                    tagvals[i] = self.replaceStr
                elif not context.trusted:
                    tagvals[i] = re.sub(r'(?i)(\s*<?)(%s)(>?;?\s*)' % RE['yandex_email'].pattern, r'\1{0}\3'.format(self.replaceStr), tagvals[i])
            elif tags[i] == 'x-sent-to':
                if not context.xSentTo and context.processingType == "common":
                    continue
            elif tags[i] == 'message-id':
                if not context.trusted and not context.xSentTo and context.processingType == "common":
                    continue
            header += (fieldNames[tags[i]] if fieldNames[tags[i]] else tags[i].capitalize()) + ': ' + (tagvals[i] if tagvals[i] else '') + "\n"
        if context.processingType == "kaspersky" and isKaspersky and context.isKasperskyByRules:
            context.isValid = True
            metric_inc('fbl_out_kaspersky')
        if not context.trusted and context.processingType == "common":
            body += "<body removed>\n"
        return "--%s\n%s%s" % (context.boundarySepOutMsg, header, body)

    def verifyDomain(self, domain, ip, context, tag):
        if len(self.SelectDomains) < 1 or self.sendOtherTo:
            m = re.search(r'\b(%s)\b' % RE['domain'].pattern, domain, re.I)
            if tag == 'received' and m:
                context.feedbackDomain, context.validEntity = m.group(1), True
            for d in self.Domains.keys():
                if not ((re.search(r'\S+\.{0}\b'.format(d), context.feedbackDomain, re.I) or context.feedbackDomain == d) and \
                    ('subnet' not in self.Domains[d] or not self.Domains[d]['subnet']) or ip and 'subnet' in self.Domains[d] and self.Domains[d]['subnet'] \
                        and not self.Domains[d]['subnet'].isdisjoint(IPSet([IP(ip)]))):
                    continue
                context.feedbackDomain, context.validEntity = d, True
                if 'x-sent-to' in self.Domains[d] and self.Domains[d]['x-sent-to']:
                    context.xSentTo = True
                d = d[:d.find('.')]
                metric_inc('fbl_out_provider_{0}'.format('google' if d == 'gmail' or d == 'google' else d))
                break
        elif not context.selected:
            for i in range(len(self.SelectDomains)):
                if not((re.search(r'\S+\.{0}\b'.format(self.SelectDomainsRE[i].pattern), domain, re.I) or domain == self.SelectDomains[i]) and \
                    (self.SelectDomains[i] not in self.Domains or self.SelectDomains[i] in self.Domains and \
                    ('subnet' not in self.Domains[self.SelectDomains[i]] or not self.Domains[self.SelectDomains[i]]['subnet'])) or \
                    self.SelectDomains[i] in self.Domains and 'subnet' in self.Domains[self.SelectDomains[i]] and \
                    self.Domains[self.SelectDomains[i]]['subnet'] and not self.Domains[self.SelectDomains[i]]['subnet'].isdisjoint(IPSet([IP(ip)]))):
                        continue
                context.selected = True
                context.feedbackDomain = self.SelectDomains[i]
                context.validEntity = True
                if 'x-sent-to' in self.Domains[self.SelectDomains[i]] and self.Domains[self.SelectDomains[i]]['x-sent-to']:
                    context.xSentTo = True
                break
        if not context.trusted:
            for d in self.TrustedDomainsRE:
                if re.search(r'\b{0}\b'.format(d.pattern), context.feedbackDomain, re.I):
                    context.trusted = True
                    metric_inc('fbl_out_trusted')
        if len(self.ExcludeDomains) > 0:
            for d in self.ExcludeDomainsRE:
                if re.search(r'\b{0}\b'.format(d.pattern), context.feedbackDomain, re.I):
                    context.selected = False


def application(env, start_response):
    if not hasattr(application, "custom_init"):
        cfg = loadSettings(INI_FILE)
        getPGCredentials(DB)
        application.FBL = FBLOutGenerator(cfg, DB)
        application.custom_init = True
    request_method = env.get("REQUEST_METHOD", "")
    request_uri = env["REQUEST_URI"]
    params = dict(urlparse.parse_qsl(request_uri.split("?")[-1]))
    address = request_uri.split("?")[0].rstrip("/")

    if address == "/fbl-ping":
        start_response("200 OK", [("Content-type", "text/plain")])
        return []

    elif address == "/fbl-out" and request_method == "POST" and env['wsgi.input']:
        start_response("200 OK", [("Content-type", "text/plain")])
        msg = env['wsgi.input'].read(int(env['CONTENT_LENGTH']))
        if application.custom_init:
            application.FBL.processMsg(msg)
            metric_inc('fbl_out_total')
            if params.get("footype", "") == "foo":
                application.FBL.processMsg(msg, "kaspersky")
                metric_inc('fbl_out_kaspersky_total')
        return []

    elif address == "/fbl-kaspersky" and request_method == "POST" and env['wsgi.input']:
        start_response("200 OK", [("Content-type", "text/plain")])
        msg = env['wsgi.input'].read(int(env['CONTENT_LENGTH']))
        metric_inc('fbl_out_kaspersky_total')
        return [application.FBL.processMsg(msg, "kaspersky")] if application.custom_init else []

    else:
        start_response("404 Not Found", [("Content-Type","text/plain")])
        return [""]
