#!/usr/bin/env python
# encoding: utf-8
# kate: space-indent on; indent-width 4; replace-tabs on;
#
import os, re, urlparse, json, string, time, ConfigParser
from datetime import datetime
from bson.binary import Binary
from bson.objectid import ObjectId
from collections import OrderedDict
from log_utils import writelog
from db_utils import IMAPCHICK, redisReconnect, loadMongoDbCredentials, getMongoDB
from common import RE, metric_inc

INI_FILE = '%s/imappush.ini' % os.path.dirname(os.path.abspath(__file__))
BSON_DOC_MAXSIZE = 16777216
APPEND_SCRIPT = """
    local msgid = redis.call("hincrby", "info", KEYS[1].."_max_msgid", 1) - 1
    if msgid == 0 then
        redis.call("hset", "info", KEYS[1].."_min_msgid", 0)
    end
    local block_id = math.floor(msgid / ARGV[1])
    local block = KEYS[1].."_"..block_id
    redis.call("hset", "blocks", block, msgid % ARGV[1] + 1)
    redis.call("append", block, ARGV[2].."\\n")
    """


def timestamp_to_str(timestamp):
    dt = datetime.fromtimestamp(timestamp)
    return dt.strftime("%Y-%m-%d %H:%M:%S")


class Imapchick:
    def __init__(self, ini_file):
        self.cfg = None
        self.error_log = None
        self.redis_db = None
        self.collections = None
        self.imapchick_db = None
        try:
            loadMongoDbCredentials(IMAPCHICK['mongodb'])
            self.imapchick_db = getMongoDB(IMAPCHICK['mongodb'])
            self.collections = set(self.imapchick_db.collection_names(False))
            self.redis_db = redisReconnect(IMAPCHICK['redis'], self.error_log)
        except Exception, e:
            writelog("Connection to DB failed: %s" % str(e), True)
            #sys.exit(1)
        try:
            self.cfg = ConfigParser.SafeConfigParser({})
            self.cfg.optionxform = str
            self.cfg.read(ini_file)
        except ConfigParser.Error, e:
            writelog("Error while parsing config file: '%s'" % str(e), True)
            sys.exit(1)
        self.error_log = open(self.cfg.get('params', 'errorsLogFile') if self.cfg.has_option('params', 'errorsLogFile') else "ERRORLOG_DIR/imappush.log", "a+t")
        self.block_size = int(self.cfg.get('params', 'block_size') if self.cfg.has_option('params', 'block_size') else '1000')

    def redisAutoReconnect(self):
        try:
            if not self.redis_db or self.redis_db and self.redis_db.execute_command('role')[0] != 'master':
                self.redis_db = redisReconnect(IMAPCHICK['redis'], self.error_log)
        except:
            self.redis_db = redisReconnect(IMAPCHICK['redis'], self.error_log)

    def createPartitionedCollection(self, collection):
        if collection in self.collections:
            return
        try:
            params = OrderedDict()
            params["msgdate"] = 1
            params["_id"] = 1
            self.imapchick_db.create_collection(collection, partitioned=True, primaryKey=params)
            self.collections = set(self.imapchick_db.collection_names(False))
        except Exception, e:
            writelog("Exception in create_partitioned_collection: %s" % str(e), True, self.error_log)

    def requestAddMessage(self, env, start_response, params):
        self.redisAutoReconnect()
        body_len = int(env.get("CONTENT_LENGTH", "0"))
        data = env["wsgi.input"].read(body_len)
        collections = filter(len, map(string.strip, params.get("folder", "").replace("-", "_").split(",")))
        m = re.search(r'^<b>log\s+:</b> cl (\S+?)\s*(?:<br>)?\s*$', data, re.I | re.M)
        msg_type = m.group(1) if m else ''
        m = re.search(r'^<b>yaml:</b> \S+\s+(\d+)', data, re.I | re.M)
        signific = m.group(1) if m else '0'
        from_addr = params.get("from_email", "").strip()
        m = re.search(r'@({})$'.format(RE['domain'].pattern), from_addr)
        from_domen = m.group(1) if m else ""
        index = {
            'cmpldate':    timestamp_to_str(int(params.get("cmpldate", int(time.time())))),
            'msgdate':     timestamp_to_str(int(params.get("msgdate", "0"))),
            'to_label':    params.get("to_label", ""),
            'to_email':    params.get("to_email", ""),
            'from_label':  params.get("from_label", ""),
            'from_email':  from_addr,
            'from_domain': from_domen,
            'mfrm':        params.get("mfrm", ""),
            'ip':          params.get("ip", ""),
            'rdns':        params.get("rdns", ""),
            'subj':        params.get("subj", ""),
            "msg_type":    msg_type,
            "signific":    int(signific),
            "mn":          params.get("mn", ""),
            "queueid":     params.get("queueid", ""),
            "stid":        params.get("stid", "")
        }
        if index['msgdate'] > index['cmpldate']: index['msgdate'] = index['cmpldate']
        for collection in collections:
            self.createPartitionedCollection(collection)
            try:
                result = self.imapchick_db[collection].insert_one({'data': Binary(data[:BSON_DOC_MAXSIZE]), 'msgdate': index['msgdate']})
            except Exception, e:
                writelog("Inserting data to DB failed: %s" % str(e), True, self.error_log)
                start_response("409 Conflict", [""])
                return [str(e)]
            index["data_id"] = str(result.inserted_id)
            try:
                self.redis_db.eval(APPEND_SCRIPT, 1, collection, self.block_size, json.dumps(index))
            except Exception, e:
                self.redis_db = redisReconnect()
                self.redis_db.eval(APPEND_SCRIPT, 1, collection, self.block_size, json.dumps(index))
        start_response("200 OK", [])
        return [""]

def application(env, start_response):
    if not hasattr(application, "custom_init"):
        application.IMAP = Imapchick(INI_FILE)
        application.custom_init = True
    request_method = env.get("REQUEST_METHOD", "")
    request_uri = env["REQUEST_URI"]
    params = dict(urlparse.parse_qsl(request_uri.split("?")[-1]))
    address = request_uri.split("?")[0].rstrip("/")

    if address == "/ping":
        start_response("200 OK", [("Content-type", "text/plain")])
        return []

    elif address == "/imapchick/addmessage" and request_method == "POST" and env['wsgi.input']:
        return application.IMAP.requestAddMessage(env, start_response, params)

    start_response("200 OK", [])
    return [""]
