#!/usr/bin/python2
# encoding: utf-8
# kate: space-indent on; indent-width 4; replace-tabs on;
#
from __future__ import print_function
import os, os.path, sys, re, json, time
sys.path.insert(0, os.path.dirname(os.path.abspath(__file__)))
sys.path.insert(0, 'WORKING_DIR')
import pymongo, urllib2, smtplib
from urllib import urlopen
from traceback import format_exception
from shutil import move
from tempfile import mkstemp

LOGFILE = 'WORKING_DIR/logs/so-ml.log'
RETRY_COUNT = 3
MONGO = {
    'cluster': '',
    'port':    27018,
    'db':      'so_ml',
    'hosts':   ['sas-82j15sr5u4ezy36h.db.yandex.net', 'vla-ya6rl0p17o2nj3wl.db.yandex.net'],
    'ssl':     True
}
MODELS = {
    'folder':       'WORKING_DIR/so-ml/data/',
    'testing_slot': 6,
    'current_slot': 5
}
RULES = {
    'cluster':        'so_db',
    'port':           27017,
    'db':             'solog',
    'hosts':          ["db1%s.so.yandex.net" % dc for dc in "jmh"],
    'folder':         'WORKING_DIR/rules/',
    'folder_in':      'WORKING_DIR/rules/',
    'folder_out':     'WORKING_DIR/rules/outgoing/',
    'folder_corp':    'WORKING_DIR/rules/',
    'mn_folder_in':   'WORKING_DIR/rules/mn/',
    'mn_folder_out':  'WORKING_DIR/rules/outgoing/mn/',
    'mn_folder_corp': 'WORKING_DIR/rules/mn/',
    'mn_weight_in':   9.0,
    'mn_weight_out':  9.0,
    'mn_weight_corp': 9.0,
    'yt_dict_in':     '//home/so_fml/nirvana/rules_dict_in',
    'yt_dict_out':    '//home/so_fml/nirvana/rules_dict_out',
    'yt_dict_corp':   '//home/so_fml/nirvana/rules_dict_corp'
}
YT = {
    'token':                '',
    'proxy':                'hahn.yt.yandex.net',
    'spdaemon_config_path': '//home/so_fml/config/sp_daemon'
}
OAUTH_TOKEN = {'conductor': '', 'app': ''}
ID = {
    'main_workflow_in':               "a9cd048f-af06-462b-b5f2-0aaa8149c069",
    'main_workflow_instance_in':      "2e086478-a885-4334-ab31-640c9cdc65cf",
    'main_workflow_out':              "f7159da9-9e05-45d3-aea6-a378cd3fed0d",
    'main_workflow_instance_out':     "cdad7148-5892-46ca-b3c9-cd4a5ada565b",
    'compare_workflow':               "f7b79cf3-8ed0-4919-975a-9b1b8aae01ec",
    'compare_workflow_instance':      "49349a12-a739-476f-aed5-407f86467727",
    'make_so_models_bundle':          "6aca9e63-8afc-4d40-9956-37a08bb6f47d",
    'make_so_models_bundle_instance': "6b979640-8e2d-4373-a690-5b408a902a4c"
}
SIGNALS = {
    'Step1.Check_models_diff': 25,
    'Step2.Check_complaints':  26,
    'Step3.Check_done':        27
}
TIMERS = {
    'Step1.Check_models_diff': 5400,
    'Step2.Check_complaints':  21600,
    'Step3.Check_done':        3600
}
NIRVANA_API_HOST = 'https://nirvana.yandex-team.ru'
BUILD_PKG_URL = 'http://build-rhel6.so.yandex.net/make_matrixnet'   # http://sobuild01i.cmail.yandex.net/make_matrixnet, http://klimiky.cmail.yandex.net/test/save_files
SEND2SANDBOX_URL = 'http://spamcorp.search.yandex.net/make_infos'
NGINX_FOLDER = 'WORKING_DIR'
SMTP_TIMEOUT = 5.0

def get_traceback():
    exc_type, exc_value, exc_traceback = sys.exc_info()
    tb = ''
    for step in format_exception(exc_type, exc_value, exc_traceback):
        try:
            tb += "\t" + step.strip() + "\n"
        except:
            pass
    return tb

def writelog(msg, isTB = False):
    if not msg: return
    try:
        tb = "\n"
        if isTB:
            tb = get_traceback()
        f = open(LOGFILE, 'a')
        f.write(time.strftime("[%Y-%m-%d %H:%M:%S]: ") + msg + tb)
        f.close()
    except Exception, e:
        print("Writelog error: %s" % str(e), file = sys.stderr)

def doRequest(url, prompt):
    try:
        f = urlopen(url)
        if f.getcode() == 200:
            return f.read()
        else:
            writelog('{0} response HTTP code: {1}, body: {2}'.format(prompt, f.getcode(), f.info()))
    except Exception, e:
        writelog('%s HTTP request failed: %s.' % (prompt, str(e)), True)
    return ""

def cluster_hosts(conductor_group, default_hosts = []):
    if conductor_group:
        for i in range(RETRY_COUNT):
            r = doRequest("https://c.yandex-team.ru/api-cached/groups2hosts/%s" % conductor_group, "Get DB cluster hosts for group %s" % conductor_group)
            if r:
                hosts = map(str.strip, r.splitlines())
                return hosts if len(hosts) > 0 else default_hosts
            else: continue
    return default_hosts

def mongo_conn_str(cfg):
    prfx = ("%s:%s@" % (cfg['user'], cfg['passwd'])) if 'user' in cfg and cfg['user'] else ''
    return "mongodb://%s%s/%s" % (prfx, ','.join(cluster_hosts(cfg['cluster'], cfg['hosts'])), cfg['db'])

def loadDBCredentials(cfg):
    f, CURDIR = None, 'WORKING_DIR'
    try:
        if not os.path.exists('%s/.mongodb.%s' % (CURDIR, cfg['db'])):
            CURDIR = os.path.dirname(os.path.abspath(__file__))
        f = open('%s/.mongodb.%s' % (CURDIR, cfg['db']))
        for line in f:
            sf = line.split(':')
            if len(sf) == 2:
                cfg['user'], cfg['passwd'] = sf[0], sf[1]
                break
        f.close()
    except Exception, e:
        writelog("getCredentials exception: %s" % str(e), True)

def getDB(cfg):
    mongo_cfg = {
        'host':             mongo_conn_str(cfg),
        'port':             int(cfg['port']),
        'connectTimeoutMS': 10000,
        'socketTimeoutMS':  10000,
        'read_preference':  pymongo.read_preferences.ReadPreference.NEAREST
    }
    if cfg.get('ssl', False):
        HOME = 'WORKING_DIR'
        if not os.path.exists('%s/allCAs.pem' % HOME):
            HOME += '/.mongodb'
            if not os.path.exists('%s/allCAs.pem' % HOME):
                HOME = os.path.dirname(os.path.abspath(__file__))
        mongo_cfg.update({'ssl': True, 'ssl_ca_certs': '%s/allCAs.pem' % HOME})
    return pymongo.MongoClient(**mongo_cfg)[cfg['db']]

def loadCredentials():
    try:
        if os.path.exists(NGINX_FOLDER + '/.yt_token'):
            f = open(NGINX_FOLDER + '/.yt_token')
            YT['token'] = f.read().strip()
            f.close()
        if os.path.exists(NGINX_FOLDER + '/.conductor_token'):
            f = open(NGINX_FOLDER + '/.conductor_token')
            OAUTH_TOKEN['conductor'] = f.read().strip()
            f.close()
        if os.path.exists(NGINX_FOLDER + '/.app_token'):
            f = open(NGINX_FOLDER + '/.app_token')
            OAUTH_TOKEN['app'] = f.read().strip()
            f.close()
    except Exception, e:
        writelog("loadYTCredentials error: %s" % str(e), True)

def sendEmail(msg, fromaddr='robot-mailspam@yandex-team.ru', toaddr='so-report@yandex-team.ru'):
    try:
        if toaddr:
            server = smtplib.SMTP("outbound-relay.yandex.net", 25, timeout=SMTP_TIMEOUT)
            server.sendmail(fromaddr, toaddr, msg)
            server.quit()
    except Exception, e:
        writelog("Error in sendEmail: %s" % str(e), True)

def updateRulesRepo():
    ret = ''
    try:
        ret = subprocess.check_call('cd %s && git pull >/dev/null' % RULES['folder'], shell=True)
    except Exception, e:
        ret = "updateRulesRepo error: '%s'" % str(e)
    return ret

def makeCommit(msg):
    ret = updateRulesRepo()
    if not ret:
        print("Updating rules: %s" % ret, file=sys.stderr)
        if re.search(r'error|fatal|reject', str(ret)):
            return ret
    try:
        ret = subprocess.check_call('cd %s && git commit -am "%s" --author="robot-mailspam" && git push origin master >/dev/null' % (RULES['folder'], msg), shell=True)
    except Exception, e:
        ret = "makeCommit error: '%s'" % str(e)
    return ret

def assignRuleWeight(rule, weight, filename, route='in'):
    f2, tmp_filename = mkstemp()
    f_out = os.fdopen(f2, 'w')
    with open(RULES['folder_%s' % route] + filename) as f:
        for line in f:
            if re.match(r'rule\s+{0}'.format(rule), line):
                print(re.sub(r'^(rule\s+{0}\s+)(\-?\d\S*)'.format(rule), r'\g<1>{0}'.format(weight), line), end='', file=f_out)
            else:
                print(line, end = '', file=f_out)
    f_out.close()
    os.unlink(RULES['folder_%s' % route] + filename)
    move(tmp_filename, RULES['folder_%s' % route] + filename)

def enableProdSlotModel(enable=True, route='in'):
    weight = RULES['mn_weight_%s' % route] if enable else 0
    assignRuleWeight('MATRIXNET_HAM_ALL_C', -weight, 'ml.rul', route)
    assignRuleWeight('MATRIXNET_SPAM_ALL_C', weight, 'ml.rul', route)
    makeCommit('Automatic %sabling of prod model in prod slot' % ('en' if weight else 'dis'))

def requestNirvana(method, params):
    for i in range(RETRY_COUNT):
        try:
            r = urllib2.Request(url = '%s/api/public/v1/%s' % (NIRVANA_API_HOST, method),
                data = json.dumps({"jsonrpc": "2.0", "method": method, "id": params['workflowId'], "params": params}),
                headers = {'Content-Type': 'application/json; charset=utf-8', 'Authorization': 'OAuth %s' % OAUTH_TOKEN['app']})
            f = urllib2.urlopen(r)
            if f:
                return f.read()
            else:
                writelog('requestNirvana request #%d response is empty!' % (i + 1))
                continue
        except urllib2.URLError, e:
            writelog('requestNirvana HTTP request (attempt #%d) failed: %s\n' % (i + 1, e.reason), True)
            continue
        except urllib2.HTTPError, e:
            writelog('requestNirvana HTTP request (attempt #%d) failed (code=%s): %s\n' % (i + 1, e.code, e.reason), True)
            continue
        except Exception, e:
            writelog('requestNirvana HTTP request #%d failed: %s\n' % (i + 1, str(e)), True)
            continue
    return ""

def startNewWorkflow(workflow_id, workflow_instance_id, blocks=[], params={}):
    err_str = ''
    try:
        r = json.loads(requestNirvana('cloneWorkflow', {'workflowId': workflow_id, 'workflowInstanceId': workflow_instance_id}))
        if 'error' in r:
            err_str = 'NIRVANA cloneWorkflow error: code=%s, message="%s", data="%s"' % (r['error']['code'], r['error']['message'], r['error']['data'] if 'data' in r['error'] else '')
            writelog(err_str)
            return err_str
        new_workflow_id = r['result']
        r = json.loads(requestNirvana('getWorkflowMetaData', {'workflowId': new_workflow_id}))
        if 'error' in r:
            err_str = 'NIRVANA getWorkflowMetaData error: code=%s, message="%s", data="%s"' % (r['error']['code'], r['error']['message'], r['error']['data'] if 'data' in r['error'] else '')
            writelog(err_str)
            return err_str
        new_workflow_instance_id = r['result']['instanceId']
        writelog("New workflow: ID=%s, InstanceId=%s" % (new_workflow_id, new_workflow_instance_id))
        if len(blocks) > 0 and len(params.keys()) > 0:
            r = json.loads(requestNirvana('setBlockParameters', {
                "workflowId": new_workflow_id,
                "workflowInstanceId": new_workflow_instance_id,
                "blocks": map(lambda bid: {"code": bid}, blocks),
                "params": map(lambda (p, v): {"parameter": p, "value": v}, params.iteritems())
            }))
            if 'error' in r:
                err_str = 'NIRVANA setBlockParameters error: code=%s, message="%s", data="%s"' % (r['error']['code'], r['error']['message'], r['error']['data'] if 'data' in r['error'] else '')
                writelog(err_str)
                return err_str
            writelog("Workflow '%s' params saving response: %s" % (new_workflow_id, str(r['result'])))
        r = json.loads(requestNirvana('startWorkflow', {'workflowId': new_workflow_id, 'workflowInstanceId': new_workflow_instance_id}))
        if 'error' in r:
            err_str = 'NIRVANA startWorkflow error: code=%s, message="%s", data="%s"' % (r['error']['code'], r['error']['message'], r['error']['data'] if 'data' in r['error'] else '')
            writelog(err_str)
            return err_str
        writelog("Workflow '%s' starting response: %s" % (new_workflow_id, str(r['result'])))
    except Exception, e:
        err_str = "startNewWorkflow error: %s" % str(e)
        writelog(err_str, True)
    return err_str

def getFormulaInfo(formula_id):
    info = doRequest("https://fml.yandex-team.ru/rest/api/formula/fml/%s" % formula_id, 'getFormulaInfo')
    return json.loads(info) if info else {}

def getAcceptanceMetrics(formula_id):
    model = {}
    try:
        db = getDB(MONGO)
        rec = db['models'].find_one({'formula_id': int(formula_id)})
        #writelog('DB Record for formula %s: %s' % (formula_id, str(rec)))
        if rec:
            for k in rec:
                m = re.match(r'^am_(\w+)$', k)
                if m and m.group(1):
                    model[m.group(1)] = rec[k]
    except Exception, e:
        writelog("getAcceptanceMetrics error: %s" % str(e), True)
    return model

def getSetting(setting='', section='general', default="", db=None):
    try:
        if not db:
            db = getDB(MONGO)
        doc = db['settings'].find_one({'_id': section})
        if doc:
            return doc.get(setting, default) if setting else doc
    except Exception, e:
        writelog("getSetting DB error: %s.\n" % str(e), True)
    return default

def setSetting(setting, value, section='general', db=None):
    try:
        if not db:
            db = getDB(MONGO)
        db['settings'].update_one({'_id': section}, {'$set': {setting: value}})
    except Exception, e:
        writelog("setSetting DB error: %s.\n" % str(e), True)

def getModel(formula_id, isProdOnly=False, route='in', db=None):
    model, prod = {'formula_id': int(formula_id)}, None
    try:
        if not db:
            db = getDB(MONGO)
        rec = db['models'].find_one({'formula_id': int(formula_id)})
        if not rec:
            return {}
        for (k, v) in rec.items():
            if k == '_id' or k == 'formula_id':
                continue
            model[k] = v
        prod = db['models_prod_history'].find_one({'formula': formula_id, 'route': route}, sort=[('datetime', pymongo.DESCENDING)])
    except Exception, e:
        writelog("getModel DB error: %s" % str(e), True)
    if prod:
        for k in ['info', 'datetime']:
            if k in prod:
                model[k] = prod[k]
    try:
        prod = db['prod_history'].find_one({'route': route}, sort=[('datetime', pymongo.DESCENDING)])
    except Exception, e:
        writelog("getModel DB error: %s" % str(e), True)
    slots = []
    if prod and 'formulas' in prod:
        for (slot, formula) in prod['formulas'].iteritems():
            if model['formula_id'] == formula:
                slots.append(slot)
        if not isProdOnly or isProdOnly and len(slots) > 0:
            model['slot'] = ', '.join(slots)
    if 'creation_time' not in model:
        if 'start_time' not in model:
            info = getFormulaInfo(model['formula_id'])
            if 'dateTime' in info:
                model['start_time'] = ' '.join(info['dateTime'][:19].split('T'))
        model['creation_time'] = model['start_time'][:model['start_time'].rfind(':')]
    if 'name' not in model or not model['name']:
        model['name'] = (("%s: " % model['model_type']) if 'model_type' in model else '') + (model['comment'] if 'comment' in model else '')
        if not model['name']:
            model['name'] = "so_%s" % model['creation_time'].split()[0]
    return model

def setModelStatus(status, workflow_id='', workflow_instance_id='', formula_id=0, route='in', db=None):
    try:
        if not db:
            db = getDB(MONGO)
        cond, d = {'route': route}, {'status': status}
        if workflow_id:
            if formula_id:
                d['formula_id'] = int(formula_id)
                cond['$or'] = [{'workflow_id': workflow_id}, {'formula_id': int(formula_id)}]
            else:
                cond['workflow_id'] = workflow_id
        elif formula_id:
            if workflow_id:
                cond['$or'] = [{'workflow_id': workflow_id}, {'formula_id': int(formula_id)}]
            else:
                cond['formula_id'] = int(formula_id)
        if workflow_instance_id:
            if '$or' in cond:
                cond['$or'][0]['workflow_instance_id'] = workflow_instance_id
            else:
                cond['workflow_instance_id'] = workflow_instance_id
        if status == 'pool_gathering' or status.startswith('model'):
            d['start_time'], r = time.strftime("%Y-%m-%d %H:%M:%S"), {}
            if workflow_id:
                r = json.loads(requestNirvana('getWorkflowMetaData', {'workflowId': workflow_id}))
                if 'error' in r:
                    writelog('getWorkflowMetaData error: code=%s, message="%s", data="%s"' % (r['error']['code'], r['error']['message'], r['error']['data'] if 'data' in r['error'] else ''))
                else:
                    r = json.loads(requestNirvana('getExecutionState', {'workflowId': workflow_id, 'workflowInstanceId': r['result']['instanceId']}))
                    if 'error' in r:
                        writelog('getExecutionState error: code=%s, message="%s", data="%s"' % (r['error']['code'], r['error']['message'], r['error']['data'] if 'data' in r['error'] else ''))
                    elif 'result' in r:
                        d['start_time'] = ' '.join(r['result']['started'][:19].split('T'))
        doc = db['models'].find_one_and_update(cond, {'$set': d}, sort=[('start_time', pymongo.DESCENDING)], upsert=True)
    except Exception, e:
        writelog("setModelStatus DB error: %s.\n" % str(e), True)

def getModelParam(param, workflow_id='', workflow_instance_id='', formula_id=0, default='', db=None):
    doc = {}
    try:
        if not db:
            db = getDB(MONGO)
        cond = {}
        if workflow_id:
            cond['workflow_id'] = workflow_id
            if workflow_instance_id:
                cond['workflow_instance_id'] = workflow_instance_id
        elif formula_id:
            cond['formula_id'] = int(formula_id)
        doc = db['models'].find_one(cond, sort=[('start_time', pymongo.DESCENDING)])
    except Exception, e:
        writelog("getModelParam DB error: %s.\n" % str(e), True)
    return (doc[param] if param in doc else default)

def getModels(isProdOnly=False, route='in', db=None):
    models = []
    try:
        if not db:
            db = getDB(MONGO)
        for rec in db['models'].find({'route': route}, {'_id': False, 'formula_id': True}):
            if "formula_id" not in rec:
                continue
            models.append(getModel(int(rec['formula_id']), isProdOnly, route, db))
    except Exception, e:
        writelog("getModels error: %s" % str(e), True)
    return models

def getComparedModels(models=None, deepness=-1, route='in', db=None):
    compared_models = []
    if not db:
        db = getDB(MONGO)
    if not models:
        models = sorted(getModels(False, route, db), key=lambda m: m['start_time'], reverse=True)
    for model in models:
        if 'formula_id' not in model or ('am_f_measure' not in model and 'f1_score' not in model) or 'threshold' not in model:
            continue
        try:
            q = {'formula_id': model['formula_id'], 'op_type': 'pool_formula_test', 'name': 'output_messages'}
            if 'workflow_id' in model and model['workflow_id']:
                q['workflow_id'] = model['workflow_id']
            if 'workflow_instance_id' in model and model['workflow_instance_id']:
                q['workflow_instance_id'] = model['workflow_instance_id']
            rec = db['intermidiate_tables'].find_one(q)
            q['op_type'], q['name'] = 'pool_test', 'features'
            r = db['intermidiate_tables'].find_one(q)
            compared_models.append({
                "formula_id":   int(model['formula_id']),
                "path":         r['path'] if r and 'path' in r else '',
                "stats_folder": rec['path'][:rec['path'].rfind('/')] if rec and 'path' in rec else '',
                "name":         model['name'],
                "f_measure":    model['am_f_measure'] if 'am_f_measure' in model else model['f1_score'],
                "precision":    model['am_precision'] if 'am_precision' in model else '',
                "recall":       model['am_recall'] if 'am_recall' in model else '',
                "accuracy":     model['am_accuracy'] if 'am_accuracy' in model else '',
                "threshold":    model["threshold"]
            })
        except Exception, e:
            writelog('getComparedModels error "%s" for model: %s' % (str(e), str(model)), True)
    return compared_models[:deepness + 1]

def getComparedModelsInfo(formula_id, db=None):
    models = []
    try:
        if not db:
            db = getDB(MONGO)
        for model in db['comparing_models_info'].find({'$or': [{'formula1': int(formula_id)}, {'formula2': int(formula_id)}]}):
            if '_id' in model:
                del model['_id']
            models.append(model)
    except Exception, e:
        writelog("getComparedModelsInfo DB error: %s" % str(e), True)
    return models

def getProdModelsCurrent(route='in', db=None):
    prod_models = {}
    try:
        if not db:
            db = getDB(MONGO)
        prod = db['prod_history'].find_one({'route': route}, sort=[('datetime', pymongo.DESCENDING)], limit=1)
        slots = getSetting('slots_{0}'.format(route), db=db)
        if prod and 'formulas' in prod:
            for (slot, formula) in prod['formulas'].iteritems():
                rec = db['models_prod_history'].find_one({'formula': formula, 'slot': slot}, sort=[('datetime', pymongo.DESCENDING)], limit=1)
                if rec is None:
                    writelog('Unable find record with formula {0} and slot {1} in models_prod_history collection!'.format(formula, slot))
                    continue
                prod_models[slot] = {
                    "formula":   int(rec["formula"]),
                    "datetime":  re.sub(r'^(.*)(:\d\d)$', r'\1', rec["datetime"]),
                    "slot":      rec["slot"],
                    "threshold": float(rec["threshold"]),
                    "name":      rec["comment"],
                    "info":      rec["info"]
                }
        writelog('ProdModelsCurrent: %s' % json.dumps(prod_models))
    except Exception, e:
        writelog("getProdModelsCurrent DB error: %s" % str(e), True)
    return prod_models

def sendModelsToProd(models, status = 'sending_models2sandbox_stable', route='in', db=None, yt=None):
    output, formulas_by_slots = '', {}
    try:
        if not db:
            db = getDB(MONGO)
        sandbox_resource_ids, slots, formula_ids, thresholds, resource_id = [], [], [], [], ''
        for model in models:
            formula_ids.append(int(model['formula_id']))
            slots.append(model['slot'])
            formulas_by_slots[model['slot']] = int(model['formula_id'])
            m = getModel(model['formula_id'], route = route, db = db)
            if 'resource_id' not in m and 'workflow_id' in m and m['workflow_id'] and 'workflow_instance_id' in m and m['workflow_instance_id']:
                try:
                    workflow = json.loads(requestNirvana('getBlockResults', {'workflowId': m['workflow_id'], 'workflowInstanceId': m['workflow_instance_id'], 'blocks': [{'code': 'operation-1516696928277-121$773'}], 'outputs': ['resource_id']}))
                    if 'error' in workflow:
                        writelog('NIRVANA getBlockResults error: code=%s, message="%s", data="%s"' % (workflow['error']['code'], workflow['error']['message'], workflow['error']['data'] if 'data' in workflow['error'] else ''))
                    else:
                        writelog("Workflow info: %s" % str(workflow))
                        file_url = workflow['result'][0]['results'][0]['directStoragePath']
                        resource_id = int(doRequest(file_url, "Downloading Nirvana workflow block's result"))
                        writelog("For model %s with resource_url=%s resource_id=%s" % (model['formula_id'], file_url, resource_id))
                except Exception, e:
                    writelog("Exception while retrieving Nirvana workflow block's result: %s" % str(e), True)
            elif 'resource_id' in m and m['resource_id']:
                resource_id = int(m['resource_id'])
            if resource_id:
                sandbox_resource_ids.append(resource_id)
            thresholds.append(str(m['threshold']) if 'threshold' in m and m['threshold'] else 0)
        writelog("sendModelsToProd: slots=%s, sandbox_resource_ids=%s, formula_ids=%s, thresholds=%s, route=%s" % (slots, sandbox_resource_ids, formula_ids, thresholds, route))
        if len(sandbox_resource_ids) == len(slots):
            err_str = startNewWorkflow(ID['make_so_models_bundle'], ID['make_so_models_bundle_instance'], ['operation-1539363370008-3'],
                                       {'sandbox_resource_ids': sandbox_resource_ids, 'slots': slots, 'formula_ids': formula_ids, 'thresholds': thresholds, 'route': route})
            if err_str:
                writelog("Nirvana's startNewWorkflow error: %s" % err_str)
                return
            t = time.strftime("%Y-%m-%d %H:%M:%S")
            db['prod_history'].insert_one({'datetime': t, 'formulas': formulas_by_slots, 'status': status, 'route': route})
            for i, model in enumerate(models):
                db['models_prod_history'].insert_one({'datetime': t, "formula": int(model['formula_id']), "threshold": thresholds[i], "comment": model["name"], "info": model["comment"], "slot": model["slot"], "route": route})
        elif len(sandbox_resource_ids) < len(slots) and len(sandbox_resource_ids) > 0:
            writelog("Unable retrieve enough resource_ids while gathering data for starting workflow!")
        elif len(sandbox_resource_ids) == 0:
            writelog("Unable retrieve any resource_id while gathering data for starting workflow!")
    except Exception, e:
        writelog("sendModelsToProd error: %s" % str(e), True)
    return output

def doModelsOfflineTest(compared_models, deepness, db=None):
    S, formulas, r, c, N = [], [], [], [0.25, 0.25, 0.20, 0.15, 0.15], deepness + 1
    if not db:
        db = getDB(MONGO)
    for j in range(N):
        formulas.append(int(compared_models[j]['formula_id'])); S.append(0)
    writelog("OfflineTest input models: %s" % str(formulas))
    for i in range(N):
        F = []; r.append([])
        for j in range(N):
            if i != j:
                m = db['comparing_models_info'].find_one({'formula1': formulas[j], 'formula2': formulas[i]})
            else:
                m = getAcceptanceMetrics(formulas[j])
            if m:
                F.append([m["f_measure"], j + 1])
            else:
                writelog("Comparing formulas results is absent for [%s, %s]!" % (formulas[j], formulas[i]))
                return 0
        a = map(lambda f: f[1], sorted(F, key = lambda f: f[0], reverse=True))
        for j in range(N):
            r[i].append(a.index(j + 1) + 1)
    for j in range(N):
        for i in range(N):
            S[j] += c[i] * r[i][j]
    j = sorted(enumerate(S), key = lambda s: s[1])[0][0]
    return formulas[j]

def testModelOnline(formula_id, db=None, forceSendModelToProd=False, yt=None):
    try:
        if not db:
            db = getDB(MONGO)
        doc = db['models'].find_one_and_update({'formula_id': int(formula_id)}, {'$inc': {'cross_compared_count': 1}}, return_document = pymongo.ReturnDocument.AFTER)
        deepness = doc.get('cross_compared_total', 0)
        if getSetting('offline_test') and doc.get('cross_compared_count', 0) >= 2 * deepness:
            compared_models = getComparedModels(db = db)
            doc = db['prod_history'].find_one({}, sort = [('datetime', pymongo.DESCENDING)])
            current_formula_id = doc['formulas'][MODELS['current_slot'] - 1]
            writelog("Compared models: %s" % str(map(lambda m: m['formula_id'], compared_models)))
            if compared_models[0]['formula_id'] != current_formula_id:
                for i in range(1, len(compared_models)):
                    if compared_models[i]['formula_id'] != current_formula_id:
                        continue
                    if i != 1:
                        model = compared_models.pop(i)
                        compared_models.insert(1, model)
                    break
            won_formula_id = doModelsOfflineTest(compared_models, deepness, db)
            if won_formula_id:
                writelog("In offline testing of model %s won model: %s" % (formula_id, won_formula_id))
            elif forceSendModelToProd:
                won_formula_id = int(formula_id)
                writelog("Error in offline testing of model %s, but this model forced be sent to prod!" % formula_id)
            if won_formula_id:
                if won_formula_id != int(formula_id):
                    setModelStatus('model_rejected', formula_id = int(formula_id), db=db)
                if won_formula_id != current_formula_id:
                    enableProdSlotModel(True if getSetting('online_test') else False)
                    #sendModelToProdBranch(won_formula_id, threshold, MODELS['testing_slot'], 'prestable' if getSetting('online_test') else 'stable')
                    n, branch = MODELS['testing_slot'] - 1, 'prestable' if getSetting('online_test') else 'stable'
                    model = getModel(str(won_formula_id))
                    prod_models = getProdModelsCurrent(db)
                    prod_models[n]['formula'] = won_formula_id
                    prod_models[n]['datetime'] = time.strftime("%Y-%m-%d %H:%M:%S")
                    prod_models[n]['threshold'] = model['threshold']
                    prod_models[n]['comment'] = model['name']
                    prod_models[n]['info'] = ''
                    del prod_models[n]['name']
                    if branch == 'stable':
                        n = MODELS['current_slot'] - 1
                        prod_models[n]['formula'] = won_formula_id
                        prod_models[n]['datetime'] = time.strftime("%Y-%m-%d %H:%M:%S")
                        prod_models[n]['threshold'] = model['threshold']
                        prod_models[n]['comment'] = model['name']
                        prod_models[n]['info'] = ''
                        del prod_models[n]['name']
                    setModelStatus('building_package_%s' % branch, formula_id = won_formula_id, db=db)
                    writelog("Sending model %s in slot %s to branch '%s'" % (won_formula_id, n + 1, branch))
                    sendModelsToProd(prod_models, branch, won_formula_id, db, yt)
            else:
                writelog("Abort offline testing of model %s due to error." % formula_id)
    except Exception, e:
        writelog("testModelOnline error: %s" % str(e), True)

def saveAppliedPoolState(formula1, formula2, status, params, route='in', yt=None, db=None):
    try:
        writelog("Params: %s" % json.dumps(params))
        if not db:
            db = getDB(MONGO)
        record = {'status': status, 'datetime': time.strftime("%Y-%m-%d %H:%M:%S")}
        for p in params:
            if p == 'formula1' or p == 'formula2' or p == 'main_formula_id': continue
            record[p] = params[p] if p == "pool" or p.startswith("workflow_") else float(params[p])
        writelog("Status: %s, Record: %s" % (status, json.dumps(record)))
        try:
            db['comparing_models_info'].update_one({'formula1': int(formula1), 'formula2': int(formula2), 'route': route}, {'$set': record}, upsert=True)
        except Exception, e:
            writelog("saveAppliedPoolState DB error: %s" % str(e), True)
        if '_id' in record:
            del record['_id']
    except Exception, e:
        writelog("saveAppliedPoolState error: %s" % str(e), True)
    return {"result": record}

def startComparingModels(formula1, formula2, main_formula_id=0, route='in', db=None):
    if not (formula1.isdigit() and formula2.isdigit()):
        return {"error": "Formula ID is absent or empty or wrong!"}
    model1, model2, rec = getModel(formula1), getModel(formula2), {}
    if not (model1 and 'threshold' in model1):
        return {"error": "Main model's info is absent or it not contains threshold!"}
    if not (model2 and 'workflow_id' in model2):
        return {"error": "Model's info for the given pool is absent or it not contains workflow_id!"}
    try:
        if not db:
            db = getDB(MONGO)
        rec = db['comparing_models_info'].find_one({'formula1': int(formula1), 'formula2': int(formula2)})
        if rec and 'status' in rec and rec['status'] and (rec['status'] == 'started' or rec['status'] == 'completed'):
            del rec['_id']
            return {"result": rec}
        rec = db['intermidiate_tables'].find_one({'workflow_id': model2['workflow_id'], 'op_type': 'pool_test', 'name': 'features'})
        if not (rec and 'path' in rec):
            return {"error": "Unable find test pool info in DB!"}
        del rec['_id']
        err_str = startNewWorkflow(ID['compare_workflow'], ID['compare_workflow_instance'],
                                   ['operation-1487243011240-4', 'operation-1487243028775-15', 'operation-1487244504943-29', 'operation-1487243068748-22'],
                                   {'table': rec['path'], 'id': formula1, 'threshold': float(model1['threshold']), 'pool_formula': int(formula2),
                                    'main_formula': int(main_formula_id), 'timestamp': time.strftime("%Y-%m-%dT%H:%M:%S"), 'route': route})
        if err_str:
            return {"error": err_str}
        return saveAppliedPoolState(formula1, formula2, 'started', {'pool': rec['path'], 'workflow_id': workflow_id}, route)
    except Exception, e:
        rec = "startComparingModels error: %s" % str(e)
        writelog(rec, True)
    return {'error': rec}
