# coding: utf8
# kate: space-indent on; indent-width 4; replace-tabs on;
#
import json
import logging
from functools import wraps
from datetime import timedelta
from django.contrib.auth import get_user_model
from django.core.exceptions import PermissionDenied
from django.shortcuts import get_object_or_404
from django.utils.datetime_safe import datetime
from django.utils.decorators import method_decorator
from django_yauth.decorators import yalogin_required


LOGGER = logging.getLogger(__name__)
class_based_login_required = method_decorator(yalogin_required, name='dispatch')


def timed_cache(ttl):
    def decorator(func):
        cache = {}
        last_updates = {}

        @wraps(func)
        def wrapper(*args, **kwargs):
            now = datetime.now()
            key = json.dumps(args) + json.dumps(kwargs)
            last_update = last_updates.get(key)
            if last_update is None or last_update + ttl < now:
                value = func(*args, **kwargs)
                cache[key] = value
                last_updates[key] = now
            return cache[key]

        return wrapper

    return decorator


@timed_cache(timedelta(minutes=5))
def login_has_rights(login, pattern):
    user = get_object_or_404(get_user_model(), username=login)
    if user.is_superuser:
        return True

    return user.groups.filter(name__regex=pattern).exists()


def access_decorator(pattern: str):
    def decorator(func):
        @wraps(func)
        def inner(*args, **kwargs):
            request = kwargs.get('request') or args[0]

            if login_has_rights(request.yauser.login, pattern):
                return func(*args, **kwargs)
            raise PermissionDenied()

        return inner

    return decorator
