# -*- coding: utf-8 -*-
# kate: space-indent on; indent-width 4; replace-tabs on;
#
from datetime import datetime
import logging
# from typing import Optional
from functools import lru_cache
from tvmauth import BlackboxTvmId
from blackboxer import Blackbox, UnknownError
from blackboxer.exceptions import HTTPError
from tvm2.protocol import BLACKBOX_MAP

from .tvm2 import TVM2Mixin
from .exceptions import FailedDependencyException

LOGGER = logging.getLogger(__name__)


class BlackboxClient(Blackbox, TVM2Mixin):

    def __init__(self, blackbox_tvm2_client: BlackboxTvmId):
        Blackbox.__init__(self, BLACKBOX_MAP[blackbox_tvm2_client]["url"], timeout=2)
        TVM2Mixin.__init__(self, blackbox_tvm2_client, blackbox_tvm2_client.value)

    def _request(
            self,
            http_method: str = 'GET',
            blackbox_method: str = None,
            required_fields: dict = {},
            **kwargs) -> dict:

        LOGGER.warning(f'BlackboxClient: tvm_client_id={self.blackbox_tvm2_client}')
        if blackbox_method is None:
            raise ValueError('blackbox_method should be set')
        try:
            result = BlackboxClient.custom_method(
                self,
                http_method=http_method,
                blackbox_method=blackbox_method,
                headers=self.spec_headers(),
                required_fields=required_fields,
                **kwargs,
            )
        except UnknownError as e:
            if 'Expired ticket' in str(e):
                result = BlackboxClient.custom_method(
                    self,
                    http_method=http_method,
                    blackbox_method=blackbox_method,
                    headers=self.spec_headers(False),
                    required_fields=required_fields,
                    **kwargs,
                )
            raise
        return result

    @lru_cache
    def getDomainInfo(self, domain_name: str = None, admin_uid: str = None, with_aliases: bool = False) -> dict:
        if not domain_name and not admin_uid:
            raise ValueError('Specify domain_name or admin_uid')

        params = {}
        if domain_name:
            params['domain'] = domain_name
        if admin_uid:
            params['domain_admin'] = admin_uid

        try:
            domain_info = self._request(blackbox_method='hosted_domains', aliases=int(with_aliases), **params)
        except HTTPError as e:
            if e.response.code == 500:
                raise FailedDependencyException('blackbox')
            raise
        if 'hosted_domains' in domain_info and domain_info['hosted_domains']:
            info = domain_info['hosted_domains'][0]
            result = {
                'admin_id': int(info['admin']),
                'registration_date': datetime.strptime(
                    info['born_date'],
                    '%Y-%m-%d %H:%M:%S'
                ),
                'domain_id': info['domid'],
                'master_domain': info['master_domain'] or None,
                'mx': info['mx'],
                'blocked': info.get('ena', '1') == '0',
                'used_in_pdd': True  # https://st.yandex-team.ru/PASSPORTDUTY-659#61f8178aea9ef145afb9f111
            }
            if with_aliases:
                result['aliases'] = [_f for _f in info.get('slaves', '').split(',') if _f]
            return result

    @lru_cache
    def getUserInfo(self, key: str, value: str, ip: str = "127.0.0.1") -> dict:
        required_fields = {
            'userip': ip,
            key: value,
            'getemails': 'all',
            'email_attributes': '1'
        }
        if key == 'suid':
            required_fields["sid"] = '2'
        dbfields = [
            'userinfo.lang.uid',
            'userinfo.country.uid',
            'userinfo.reg_date.uid',
            'subscription.suid.2',
            'subscription.suid.669',
            'accounts.login.uid'
        ]
        try:
            raw_userinfo = self._request(
                blackbox_method='userinfo',
                required_fields=required_fields,
                dbfields=','.join(dbfields),
            )
            if 'users' not in raw_userinfo or len(raw_userinfo['users']) == 0:
                return None

            userinfo = raw_userinfo['users'][0]
            dbfields = userinfo.get('dbfields', {})
            emails = []
            for email_info in userinfo.get('emails', {}):
                if 'attributes' in email_info:
                    for _email in email_info['attributes'].values():
                        emails.append(_email)

            resp = {
                'login': dbfields['accounts.login.uid'] if dbfields.get('accounts.login.uid', '') else userinfo['login'],
                'uid': userinfo['uid']['value'] if userinfo['uid'].get('value', '') else userinfo['id'],
                'suid': dbfields.get('subscription.suid.2', ''),
                'emails': emails,
                'karma': userinfo['karma'].get('value', ''),
                'karma_status': userinfo['karma_status'].get('value', ''),
                'sid669': dbfields.get('subscription.suid.669', ''),
                'reg_date': dbfields.get('userinfo.reg_date.uid', ''),
                'country': dbfields.get('userinfo.country.uid', ''),
                'lang': dbfields.get('userinfo.lang.uid', ''),
                'attributes': userinfo.get('attributes', {})
            }
            return resp

        except HTTPError as exc:
            if exc.response.code == 500:
                raise FailedDependencyException('blackbox')
            raise


def get_blackbox_client(is_corp: bool = False) -> BlackboxClient:
    if is_corp:
        return BlackboxClient(BlackboxTvmId.ProdYateam)
    else:
        return BlackboxClient(BlackboxTvmId.Prod)
