# coding: utf8
# kate: space-indent on; indent-width 4; replace-tabs on;
#
from __future__ import unicode_literals
import logging
import warnings
from django.db.models import Q
from django.contrib.auth.models import Group
from django_idm_api.compat import get_user_model
from django_idm_api.exceptions import RoleNotFound, UserNotFound
from django_idm_api.hooks import AuthHooks

from .modules import SO_WEB_MODULES
from mail.so.spamstop.web.ui.web_tools.models import RoleInfo

LOGGER = logging.getLogger(__name__)


class IdmHooks(AuthHooks):
    default_lang : str

    def __init__(self, lang='ru', *args, **kwargs):
        self.default_lang = lang
        super(IdmHooks, self).__init__(*args, **kwargs)

    def info(self):
        info = {
            "code": 0,
            "roles": {
                "slug": "role",
                "name": "роль",
                "values": self.info_impl()
            },
        }
        return info

    def info_impl(self, **kwargs):
        roles = {'superuser': 'суперпользователь'}
        for group in self.get_group_queryset():
            role_info = RoleInfo.objects.filter(group_id=group.id, lang=self.default_lang).values()
            if role_info is None:
                LOGGER.warning(f"IdmHooks.info_impl: RoleInfo is None for groupId '{group.id}' and lang '{self.default_lang}'")
            else:
                roles[f'group-{group.name}'] = role_info[0]['title']
        self._get_roles(SO_WEB_MODULES, roles)
        return roles

    def add_role_impl(self, login, role, fields, **kwargs):
        roleInfo = role.get('role')
        if not roleInfo:
            raise RoleNotFound(f"Role is not provided in {role} for login '{login}' (fields: {fields})")

        role = roleInfo
        user = self._get_or_create_user(login)

        if role == 'superuser':
            user.is_staff = True
            user.is_superuser = True
            user.save(update_fields=('is_staff', 'is_superuser'))
        elif role.startswith('group-'):
            group = self._get_group_by_name(role[6:])
            user.is_staff = True
            user.groups.add(group)
            user.save(update_fields=('is_staff', 'is_superuser'))
        else:
            raise RoleNotFound('Role does not exist: %s' % role)

    def remove_role_impl(self, login, role, data, is_fired, **kwargs):
        roleInfo = role.get('role')
        if not roleInfo:
            raise RoleNotFound(f"Role is not provided in {role} for login '{login}' (is_fired={is_fired}, data={data})")

        role = roleInfo
        try:
            user = self._get_user(login)
        except UserNotFound:
            raise

        if role == 'superuser':
            user.is_superuser = False
            self._remove_is_staff_if_needed(user)
            user.save(update_fields=('is_staff', 'is_superuser'))
        elif role.startswith('group-'):
            group = self._get_group_by_name(role[6:])

            user.groups.remove(group)
            self._remove_is_staff_if_needed(user)
            user.save(update_fields=('is_staff', 'is_superuser'))
        else:
            raise RoleNotFound('Role does not exist: %s' % role)

    def get_user_roles_impl(self, login, **kwargs):
        warnings.warn('Method `get_user_roles_impl` is deprecated and will be removed soon.')
        user = self._get_user(login)
        roles = []

        for group in self.get_group_queryset().filter(user=user):
            roles.append(f'group-{group.name}')

        if user.is_superuser:
            roles.append('superuser')
        return [{'role': role} for role in roles]
    get_user_roles_impl.is_deprecated = True

    def get_all_roles_impl(self, **kwargs):
        users = (
            get_user_model().objects.
                filter(Q(groups__name__in=self.get_group_queryset().values('name')) | Q(is_superuser=True)).
                prefetch_related('groups').
                order_by('username').
                distinct()
        )
        all_roles = []
        for user in users:
            roles = []
            if user.is_superuser:
                roles.append({'role': 'superuser'})
            for group in user.groups.all():
                roles.append({'role': f'group-{group.name}'})
            all_roles.append((user.username, roles))
        return all_roles

    @staticmethod
    def _get_node_name(node, default_name='', default_lang='ru'):
        if isinstance(node, dict):
            if default_lang in node and node:
                return node[default_lang]
            elif "en" in node and node:
                return node["en"]
            else:
                return default_name
        else:
            return node

    @staticmethod
    def _get_roles(roles_info, roles={}, group_prefix=None, name_prefix=None, default_lang='ru'):
        if group_prefix is None:
            group_prefix = 'group'
        if name_prefix is None:
            name_prefix = ''
        for role, info in roles_info.items():
            if role == "set":
                continue
            group = group_prefix + '-' + role
            name = name_prefix
            if isinstance(info, dict):
                if "values" in info:
                    return IdmHooks._get_roles(info["values"], roles, group, name)
                elif "roles" in info:
                    if "name" in info:
                        name = ("{}: ".format(name) if name else '') + \
                            IdmHooks._get_node_name(info["name"], default_lang=default_lang)
                    if "values" in info["roles"]:
                        IdmHooks._get_roles(info["roles"]["values"], roles, group, name, default_lang)
                    else:
                        IdmHooks._get_roles(info["roles"], roles, group, name, default_lang)
                else:
                    group_name = role
                    if "name" in info:
                        group_name = IdmHooks._get_node_name(info["name"], group_name, default_lang)
                    else:
                        group_name = IdmHooks._get_node_name(info, group_name, default_lang)
                    roles[group] = '{}: {}'.format(name, group_name) if name else group_name
            else:
                roles[group] = '{}: {}'.format(name, info) if name else info
        return roles.copy()

    def _get_group_by_name(self, group_name):
        """Получить группу по id или кинуть эксепшн."""
        try:
            return self.get_group_queryset().get(name=group_name)
        except Group.DoesNotExist:
            raise RoleNotFound(f'Group does not exist: {group_name}')
