# -*- coding: utf-8 -*-
# kate: space-indent on; indent-width 4; replace-tabs on;
#
from __future__ import unicode_literals
import os
import os.path
import sys
import re
import fcntl
import logging
from bson.int64 import Int64
from functools import cmp_to_key
from subprocess import check_output, call, STDOUT, CalledProcessError
from time import mktime, localtime, strftime
from datetime import date, timedelta
from collections import OrderedDict
from shutil import rmtree
from glob import glob
from .mongodb import loadMongoDbCredentials, getMongoDB
from .common_utils import getUUID, get_traceback, getTraceback
from django.conf import settings

LOGGER = logging.getLogger(__name__)
RULES_TYPES = [
    [
        'In',
        u'Входящей почты',
        [
            f'{settings.CFG["so_rules"]["folder"]}/in/',
            f'{settings.CFG["so_rules"]["folder"]}/common/'
        ],
        [
            'spam', 'spampf', 'ham', 'hampf', 'malic', 'spampercent', 'nopf', 'total', 'cmpl_spam_nopf',
            'cmpl_spam_percent', 'cmpl_ham_nopf', 'cmpl_ham_percent'
        ]
    ], [
        'Out',
        u'Исходящей почты',
        [
            f'{settings.CFG["so_rules"]["folder"]}/outgoing/',
            f'{settings.CFG["so_rules"]["folder"]}/common/'
        ],
        [
            'spam', 'spampf', 'ham', 'hampf', 'malic', 'spampercent', 'nopf', 'total', 'cmpl_spam_nopf',
            'cmpl_spam_percent', 'cmpl_ham_nopf', 'cmpl_ham_percent'
        ]
    ], [
        'Corp',
        u'Корпоративной почты',
        [
            f'{settings.CFG["so_rules"]["folder"]}/in/',
            f'{settings.CFG["so_rules"]["folder"]}/local/',
            f'{settings.CFG["so_rules"]["folder"]}/common/'],
        [
            'spam', 'spampf', 'ham', 'hampf', 'malic', 'spampercent', 'nopf', 'total', 'cmpl_spam_nopf',
            'cmpl_spam_percent', 'cmpl_ham_nopf', 'cmpl_ham_percent'
        ]
    ], [
        'Sosearch',
        u'Поиска по почте',
        [f'{settings.CFG["so_rules"]["folder"]}/msearch/'],
        ['spam', 'ham', 'malic', 'spampercent', 'total']
    ], [
        'Sopassport',
        u'Паспорта',
        [f'{settings.CFG["so_rules"]["folder"]}/passport/'],
        ['spam', 'ham', 'malic', 'spampercent', 'total']
    ], [
        'Socheckform',
        u'Проверки форм',
        [f'{settings.CFG["so_rules"]["folder"]}/checkform/'],
        ['spam', 'ham', 'malic', 'spampercent', 'total']
    ], [
        'Socheckmessages',
        u'Проверки коротких сообщений',
        [f'{settings.CFG["so_rules"]["folder"]}/checkmessages/'],
        ['spam', 'ham', 'malic', 'spampercent', 'total']
    ]
]

RULE_PARAM = dict(map(lambda rt: (rt[0], rt[3]), RULES_TYPES))
RULES_DIRS = dict(map(lambda rt: (rt[0], rt[2]), RULES_TYPES))

PARAM = {
    'spam':              ['Spam',   u'Количество спама', 'Spam', '#FF0000'],
    'spampf':            ['SpamPF', u'Количество спама с учётом персональных фильтров', 'SpamPF', '#FF6600'],
    'ham':               ['Ham', u'Количество хама', 'Ham', '#00FF00'],
    'hampf':             ['HamPF', u'Количество хама  с учётом персональных фильтров', 'HamPF', '#B0DE09'],
    'malic':             ['Malicious', u'Malicious', 'Malic', '#960018'],
    'spampercent':       ['Spam%', u'Процент спама', 'SpamPerc', '#FF9100'],
    'nopf':              ['noPF%', u'Процент спама без учета персональных фильтров', 'noPFperc', '#FF00C3'],
    'total':             ['Total', u'Всего', 'Total', '#0D8ECF'],
    'cmpl_spam_nopf':    [
        u'Кол-во жалоб<br/>на спам без ПФ<br/>(с ПФ)',
        u'Количество жалоб на спам без персональных фильтров (количество жалоб на спам с персональными фильтрами)',
        'CmplSpam',
        '#550022'
    ],
    'cmpl_spam_percent': [
        u'% от Ham',
        u'Процент жалоб на спам без персональных фильтров по отношению к хаму',
        'CmplSpamPerc',
        '#9F0DCF'
    ],
    'cmpl_ham_nopf':     [
        u'Кол-во жалоб<br/>на хам без ПФ<br/>(с ПФ)',
        u'Количество жалоб на хам без персональных фильтров (количество жалоб на хам с персональными фильтрами)',
        'CmplHam',
        '#2A0CD0'
    ],
    'cmpl_ham_percent':  [
        u'% от Spam',
        u'Процент жалоб на хам без персональных фильтров по отношению к спаму', 'CmplHamPerc', '#9FD00C'
    ]
}


class CheckingRepoContext:
    def __init__(self, lockFile=settings.CFG["so_rules"]['check_lock_path']):
        self.LOCK = None
        self.lockFile = lockFile

    def __enter__(self):
        try:
            self.LOCK = open(self.lockFile, 'w')
        except Exception as e:
            LOGGER.error(f"Error while opening file '{self.lockFile}' for writing: {e}.")
        try:
            fcntl.flock(self.LOCK, fcntl.LOCK_EX)
        except Exception as e:
            LOGGER.error(
                f"Unable to lock file '{self.lockFile}' ({e}). Most likely web-hook process has already launched.")
        return self.LOCK

    def __exit__(self, exc_type, exc_value, exc_traceback):
        if self.LOCK:
            try:
                fcntl.lockf(self.LOCK, fcntl.LOCK_UN)
                self.LOCK.close()
            except Exception as e:
                LOGGER.error(f"Error occured in finalizing checkingRepoContext: {e}")
        tb = get_traceback(exc_type, exc_value, exc_traceback)
        if tb:
            LOGGER.error("Error occured in the context of repository checking." + tb)
        return True


def createTmpDir(subfolder: str, folder: str = settings.CFG['rules_check_tmp_dir'], inner_folder: str = 'rules'):
    newTmpDir = ''
    while os.path.exists(f"{folder}/{subfolder}"):
        subfolder = getUUID()
    newTmpDir = f"{folder}/{subfolder}"
    try:
        os.makedirs('{}{}'.format(newTmpDir, "/{}".format(inner_folder) if inner_folder else ''))
    except Exception as e:
        LOGGER.error(f"Error while creating temporary folder '{subfolder}': {e}.\t{getTraceback()}")
        sys.exit(1)
    LOGGER.warning(f"Creation of temp dir '{newTmpDir}' done")
    return subfolder


def verifyRules(rulesDir: str, rulesType: str = 'In') -> tuple[str, int]:
    faultRulesCnt = 0
    res = resText = ''
    dirName = createTmpDir(rulesDir, inner_folder=settings.CFG["so_rules"]["folder_name"])
    tmpRulesParentDir = f"{settings.CFG['rules_check_tmp_dir']}/{dirName}"
    tmpRulesDir = f'{tmpRulesParentDir}/{settings.CFG["so_rules"]["folder_name"]}/'

    for rulesFolder in RULES_DIRS[rulesType]:
        call(f"cp {rulesFolder}* {tmpRulesDir}", shell=True)
        if rulesType in ['In', 'Out', 'Corp']:
            call(f"cp -f {RULES_DIRS['In'][0]}common/* {tmpRulesDir}", shell=True)
        elif rulesType in ['Test_In', 'Test_Out', 'Test_Corp']:
            call(f"cp -f {RULES_DIRS['Test_In'][0]}common/* {tmpRulesDir}", shell=True)
    try:
        try:
            resText = check_output(
                f"{settings.CFG['so_rules']['reader_path']} -R {tmpRulesDir} -C {tmpRulesParentDir}/.rules_cache",
                stderr=STDOUT,
                shell=True,
                universal_newlines=True)
        except CalledProcessError as e:
            faultRulesCnt += 1
            if hasattr(e, 'output'):
                resText += f"\nRules Reader failed: {e.output}\n"
    except Exception as e:
        LOGGER.error(
            f'verifyRules: Exception while rules reader processing of rules {rulesType} in folder {tmpRulesDir}: {e}.'
            + f'\t{getTraceback()}')
    try:
        res = check_output(
            f"{settings.CFG['so_rules']['reader_path']} -R {tmpRulesDir} -C {tmpRulesParentDir}/.rules_cache | " +
                r'grep "All\|err\|Err\|Rules cs:"',
            stderr=STDOUT,
            shell=True,
            universal_newlines=True)
    except Exception as e:
        LOGGER.error(f'verifyRules: Exception while rules reader processing of rules {rulesType} (2nd iteration): '
                     + f'{e}.\t{getTraceback()}')
    m = re.search(r'All fault rules:\s+(\d+)', res)
    if m:
        faultRulesCnt += int(m.group(1)) if m.group(1) else 0
    try:
        rmtree(tmpRulesParentDir)
    except Exception as e:
        LOGGER.error(f"verifyRules: Exception while removing folder '{tmpRulesParentDir}': {e}.\t{getTraceback()}")
    return resText, faultRulesCnt


def verifyAFRules(rulesDir: str, submodule: str, repoName: str) -> tuple[str, int]:
    faultRulesCnt = 0
    res = resText = ''
    repoPath = settings.CFG["af_rules"]["folder"] + ("{}/".format(submodule) if submodule else "")
    dirName = createTmpDir(rulesDir, inner_folder=repoName)
    tmpRulesParentDir = f"{settings.CFG['rules_check_tmp_dir']}/{dirName}"
    tmpRulesDir = f'{tmpRulesParentDir}/{repoName}/'

    LOGGER.warning(f"verifyAFRules: copying rules from submodule '{submodule}' to temporary folder '{tmpRulesDir}'")
    call("cp -ar %s* %s" % (repoPath, tmpRulesDir), shell=True)
    try:
        try:
            resText = check_output(
                f"{settings.CFG['af_rules']['reader_path']} {tmpRulesDir}",
                stderr=STDOUT,
                shell=True,
                universal_newlines=True)
        except CalledProcessError as e:
            faultRulesCnt += 1
            if hasattr(e, 'output'):
                resText += f"\nRules Reader failed: {e.output}\n"
    except Exception as e:
        LOGGER.error(
            f'verifyAFRules: Exception while rules reader processing of folder {tmpRulesDir} (submodule ' +
                f'{submodule}): {e}.\t{getTraceback()}'
        )
    try:
        res = check_output(
            f'{settings.CFG["af_rules"]["reader_path"]} {tmpRulesDir} | ' + r'grep "All\|err\|Err\|Rules cs:"',
            stderr=STDOUT,
            shell=True,
            universal_newlines=True)
    except Exception as e:
        LOGGER.error(
            f'verifyAFRules: Exception while rules reader processing (2nd iteration for submodule {submodule}):' +
                f' {e}.\t{getTraceback()}')
    m = re.search(r'All fault rules:\s+(\d+)', res)
    if m:
        faultRulesCnt += int(m.group(1)) if m.group(1) else 0
    try:
        rmtree(tmpRulesParentDir)
    except Exception as e:
        LOGGER.error(f"verifyAFRules: Exception while removing folder '{tmpRulesParentDir}': {e}.\t{getTraceback()}")
    LOGGER.warning(
        f"verifyAFRules: checking of rules in submodule '{submodule}' done: errosCount={faultRulesCnt}, " +
            f"checkResult='{resText}'"
    )
    return resText, faultRulesCnt


def time2str(ts: int) -> str:
    return strftime('%Y-%m-%d %H:%M', localtime(ts))


def eFilter(s) -> str:
    if not s:
        s = ''
    return s.replace(r'&', r'&amp;').replace(r'<', r'&lt;').replace(r'>', r'&gt;').replace(r'"', r'&quot;')


def fmtValue(val, n=0) -> str:
    if not n:
        n = 2
    return format(val, f".{n}f").rstrip('0').rstrip('.')


def getRulesForPeriod(route, rules_filter) -> dict:
    MONGO_RULES = settings.DB["rules"].copy()
    loadMongoDbCredentials(MONGO_RULES)
    rules, rules_constraint, bornday_constraint = {}, {}, {}
    db = collection = None
    try:
        db = getMongoDB(MONGO_RULES)
        collection = db[f'Rules_{route}']
    except Exception as e:
        LOGGER.error(f"DB exception: {e}.\t{getTraceback()}")
    if 'rules' in rules_filter and isinstance(rules_filter['rules'], list):
        rules_constraint['rule'] = {'$in': rules_filter['rules']}
    if ('age_min' in rules_filter and rules_filter['age_min']) or \
            ('age_max' in rules_filter and rules_filter['age_max']):
        bornday1, bornday2 = int(rules_filter.get('age_min', 0)), int(rules_filter.get('age_max', 0))
        bornday_constraint['borndate'] = {}
        if rules_filter['age_min']:
            bornday_constraint['borndate']['$lte'] = (date.today() - timedelta(days=bornday1)).isoformat()
        if rules_filter['age_max']:
            bornday_constraint['borndate']['$gte'] = (date.today() - timedelta(days=bornday2)).isoformat()
    try:
        for doc in collection.aggregate([
                {'$match': rules_constraint},
                {'$project': {'date': 1, 'rule': 1}}, {'$group': {'_id': '$rule', 'borndate':   {'$min': '$date'}}},
                {'$match': bornday_constraint}]):
            rules[doc['_id']] = doc.get('borndate', '')
    except Exception as e:
        LOGGER.error(
            f"DB data aggregation for rules constraint '{rules_constraint}' and bornday constraint " +
                f"'{bornday_constraint}' failed: {e}.\t{getTraceback()}"
        )
    LOGGER.warning(f"getRulesForPeriod: obtained rules: {rules}")
    return rules


def gatherWeights(data: dict, filename: str, filetype: str, route: str) -> int:
    s, cnt, charset = '', 0, 'windows-1251' if route[:2] == 'So' else 'koi8-r'
    try:
        with open(filename, encoding=charset, errors='replace') as f:
            while True:
                row = f.readline()
                if not row:
                    break
                if row.startswith('#') or re.match(r'^\s*$', row):
                    continue
                m = re.match(r'^rule\s+(.*)$', row)
                if m:
                    s = m.group(1)
                    m = settings.RE["rule_weight"].match(s)
                    if not m:
                        continue
                    if m.group(1) is None or m.group(2) is None or m.group(1) not in data:
                        continue
                    s = m.group(1)
                    cnt += 1
                    data[s]['weight'] = m.group(2)
                    data[s]['type'] = filetype
                    data[s]['text'] = row   # .decode(charset, 'ignore')
                    data[s]['file'] = filename[len(settings.ROOT_DIR) + 1:] \
                        if filename.startswith(settings.ROOT_DIR) else filename
                    while True:
                        line = f.readline()
                        if not line or re.match(r'^\s*$', line):
                            break
                        data[s]['text'] += line     # line.decode(charset, 'ignore')
                        m = re.match(r'^describe\s+(.*)$', line)
                        if m:
                            data[s]['describe'] = m.group(1)    # m.group(1).decode(charset, 'ignore')
                            break
    except Exception as e:
        LOGGER.error(f"Exception: {e}.\t{getTraceback()}")
    return cnt


def parseRulesInfo(info: dict, s: str, f: str) -> dict:
    while s:
        m = re.match(r'^([a-z]\S*)(?:\s+(\d+))?(?:\s+(\d+))?', s, re.I)
        if m and m.group(1):
            s1, s2, s3 = m.group(1), m.group(2) if m.group(2) else '', m.group(3) if m.group(3) else ''
            info[f'{f}_{s1}_min'] = int(s2) if s2.isdigit() else None
            info[f'{f}_{s1}_max' % (f, s1)] = int(s3) if s3.isdigit() else None
            s = s[len(s1 + s2 + s3):]
            s = re.sub(r'^\s', r'', s)
        else:
            m = re.match(r'^([^a-z]\S*\s*)', s, re.I)
            if m:
                s = s[len(m.group(1) if m.group(1) else ''):]
    return info


def getRulesMonitoringInfo(rule: str, route: str, update_rules: bool) -> dict:
    if not route:
        return {}
    info = {}
    rules = {}
    for d in RULES_DIRS[route]:
        if route == 'Corp' and d.find('local') == -1:
            continue
        for f in ['daily', '10min']:
            try:
                with open(f"{d}.{f}.mon", errors='replace') as fh:
                    for row in fh:
                        if row.startswith('#'):
                            continue
                        m = re.match(r'^\s*\b{}\b(?:\s+(.+))?'.format(rule), row)
                        if rule and m:
                            info['rule'] = rule
                            info[f'{f}_path'] = f"{d}.{f}.mon"
                            parseRulesInfo(info, m.group(1), f)
                            break
                        elif not rule:
                            m = re.match(r'^\s*\b(\w+)\b(?:\s+(.+))?', row)
                            if m and m.group(1):
                                if m.group(1) not in rules:
                                    rules[m.group(1)] = {}
                                parseRulesInfo(rules[m.group(1)], m.group(2), f)
            except Exception as e:
                LOGGER.error(f"Exception while processing file '{d}.{f}.mon': {e}")
    return info if rule else rules


def getRuleData(
        rule: str,
        route: str,
        detailed: bool,
        day: str,
        period: int,
        sortorder: str,
        data: dict) -> dict:

    MONGO_RULES = settings.DB["rules"].copy()
    loadMongoDbCredentials(MONGO_RULES)
    regex, cnt, d, match = '=' if re.match(r'^[A-Z0-9_]+$', rule) else 'REGEXP', 0, '', {}
    y, m, md = map(int, day.split('-'))
    if detailed:
        dt, ts = 24 * 3600, int(mktime(date(*map(int, day.split('-'))).timetuple()))
        match['time'] = {'$gte': ts, '$lt': ts + dt}
    elif period != 99:
        match['date'] = {'$gte': data['dates']['from'], '$lte': data['dates']['to']}
    db = collection = None
    try:
        db = getMongoDB(MONGO_RULES)
        collection = db['{}Rules_{}'.format('detailed_' if detailed else '', route)]
    except Exception as e:
        LOGGER.error(f"DB exception: {e}.\t{getTraceback()}")
    data['isregexp'] = '(задано регулярное выражение)' if regex == 'REGEXP' else ''
    if regex == '=':
        rules = [rule]
    else:
        rules = collection.distinct('rule', {'rule': re.compile(f'{rule}')})
    data['rules_count'] = len(rules)
    data2 = {}
    if data['rules_count'] == 1:
        h = {
            '_id':   '${}'.format('time' if detailed else 'date'),
            'malic': {'$sum': {'$ifNull': ['$R256', 0]}},
            'spam':  {'$sum': {'$ifNull': ['$R4', 0]}}
        }
        if route[:2] == 'So':
            h['ham'] = {'$sum': '$R1'}
            h['total'] = {'$sum': {'$add': [{'$ifNull': ['$R1', 0]}, {'$ifNull': ['$R4', 0]}]}}
        else:
            h['ham'] = {'$sum': {'$add': [{'$ifNull': ['$R1', 0]}, {'$ifNull': ['$R2', 0]}]}}
            h['hampf'] = {'$sum': '$R8'}
            h['spampf'] = {'$sum': '$R127'}
            h['total'] = {'$sum': {'$add': [
                {'$ifNull': ['$R1', 0]},
                {'$ifNull': ['$R2', 0]},
                {'$ifNull': ['$R4', 0]},
                {'$ifNull': ['$R8', 0]},
                {'$ifNull': ['$R127', 0]},
                {'$ifNull': ['$R256', 0]}
            ]}}
            h['cmpl_spam'] = {'$sum': {'$ifNull': ['$cmpl_spam', 0]}}
            h['cmpl_spam_nopf'] = {'$sum': {'$ifNull': ['$cmpl_spam_nopf', 0]}}
            h['cmpl_ham'] = {'$sum': {'$ifNull': ['$cmpl_ham', 0]}}
            h['cmpl_ham_nopf'] = {'$sum': {'$ifNull': ['$cmpl_ham_nopf', 0]}}
        LOGGER.warning(f"getRuleData: We have 1 rule={rule}. Rules: {rules}. Filter: {h}. Sortorder: {sortorder}.")
        for r in rules:
            if r not in data2:
                data2[r] = {}
            m = match.copy()
            m['rule'] = r
            data2[r]['statistics'] = OrderedDict(list(map(
                lambda row: (time2str(row['_id']) if detailed else row['_id'], row),
                collection.aggregate([
                    {'$match': m},
                    {'$group': h},
                    {'$sort': {('_id' if sortorder == 'date' else sortorder): -1}}
                ])
            )))
            data2[r]['weight'] = None
            p = collection.aggregate([
                {'$match': {'rule': r}}, {'$project': {str('time' if detailed else 'date'): 1, 'rule': 1}},
                {'$group': {
                    '_id': '$rule',
                    'borndate': {'$min': '${}'.format('time' if detailed else 'date')},
                    'lastdate': {'$max': '${}'.format('time' if detailed else 'date')}}}
            ])
            if p and p.alive:
                p = p.next()
                data2[r]['borndate'] = time2str(p['borndate']) if detailed else p['borndate']
                if period == 99:
                    lastdate = time2str(p['lastdate']) if detailed else p['lastdate']
                    if 'from' not in data['dates']:
                        data['dates']['from'] = data2[r]['borndate']
                    else:
                        data['dates']['from'] = min(data['dates']['from'], data2[r]['borndate'])
                    if 'to' not in data['dates']:
                        data['dates']['to'] = lastdate
                    else:
                        data['dates']['to'] = min(data['dates']['to'], lastdate)
            for t, row in data2[r]['statistics'].items():
                if route[:2] == 'So':
                    row['spampercent'] = fmtValue(row['spam'] * 100.0 / row['total']) if row['total'] > 0 else '0'
                else:
                    row['spampercent'] = fmtValue((row['spam'] + row['spampf'] + row['malic']) * 100.0 / row['total']) \
                        if row['total'] > 0 else '0'
                    row['nopf'] = fmtValue((row['spam'] + row['hampf'] + row['malic']) * 100.0 / row['total']) \
                        if row['total'] > 0 else '0'
                    row['cmpl_spam_percent'] = fmtValue(row['cmpl_spam_nopf'] * 100.0 / row['ham'], 6) \
                        if row['ham'] > 0 else ''
                    row['cmpl_ham_percent'] = fmtValue(row['cmpl_ham_nopf'] * 100.0 / row['spam'], 6) \
                        if row['spam'] > 0 else ''
                    row['spampf'] = fmtValue(float(row['spampf']))
                    row['hampf'] = fmtValue(float(row['hampf']))
                    row['cmpl_spam'] = fmtValue(float(row['cmpl_spam']))
                    row['cmpl_ham'] = fmtValue(float(row['cmpl_ham']))
                row['ham'] = fmtValue(float(row['ham']))
                row['spam'] = fmtValue(float(row['spam']))
                row['ham'] = fmtValue(float(row['ham']))
                row['malic'] = fmtValue(float(row['malic']))
                row['total'] = fmtValue(float(row['total']))
        for d in RULES_DIRS[route]:
            for filename in glob(d + '*.rul'):
                if not os.path.isfile(filename):
                    continue
                cnt += gatherWeights(data2, filename, 'rul', route)
                if cnt >= data['rules_count']:
                    break
            for filename in glob(d + '*.dlv'):
                if not os.path.isfile(filename):
                    continue
                cnt += gatherWeights(data2, filename, 'dlv', route)
                if cnt >= data['rules_count']:
                    break
        for r in data2:
            for par in ['type', 'weight', 'text', 'file', 'borndate']:
                if par not in data2[r]:
                    data2[r][par] = ''
    else:
        for r in rules:
            data2[r] = {}
    LOGGER.warning(f"Result of getRuleData: {data2}.")
    return data2


def getRulesData(route: str, rules: dict, params: dict) -> dict:
    MONGO_RULES = settings.DB["rules"].copy()
    loadMongoDbCredentials(MONGO_RULES)
    cnt, rules_count, d = 0, len(rules), ''
    p = params['period'] * 30 if params['period'] > 0 and params['period'] < 7 else \
        (params['days_period'] if params['days_period'] else 90)
    today = date.today().isoformat()
    fromday = (date.today() - timedelta(days=p)).isoformat()
    db = collection = None
    try:
        db = getMongoDB(MONGO_RULES)
        collection = db[f'Rules_{route}']
    except Exception as e:
        LOGGER.error(f"DB exception: {e}.\t{getTraceback()}")
    h = {
        '_id':   '$rule',
        'malic': {'$sum': '$R256'},
        'spam':  {'$sum': '$R4'}
    }
    if route.startswith('So'):
        h['ham'] = {'$sum': '$R1'}
        h['total'] = {'$sum': {'$add': [{'$ifNull': ['$R1', Int64(0)]}, {'$ifNull': ['$R4', Int64(0)]}]}}
    else:
        h['ham'] = {'$sum': {'$add': [{'$ifNull': ['$R1', Int64(0)]}, {'$ifNull': ['$R2', Int64(0)]}]}}
        h['hampf'] = {'$sum': '$R8'}
        h['spampf'] = {'$sum': '$R127'}
        h['total'] = {'$sum': {'$add': [
            {'$ifNull': ['$R1', Int64(0)]},
            {'$ifNull': ['$R2', Int64(0)]},
            {'$ifNull': ['$R4', Int64(0)]},
            {'$ifNull': ['$R8', Int64(0)]},
            {'$ifNull': ['$R127', Int64(0)]},
            {'$ifNull': ['$R256', Int64(0)]}
        ]}}
        h['cmpl_spam'] = {'$sum': '$cmpl_spam'}
        h['cmpl_spam_nopf'] = {'$sum': '$cmpl_spam_nopf'}
        h['cmpl_ham'] = {'$sum': '$cmpl_ham'}
        h['cmpl_ham_nopf'] = {'$sum': '$cmpl_ham_nopf'}
    LOGGER.warning(f"getRulesData: We have rules: {rules}. Filter: {h}.")
    data = OrderedDict()
    for doc in collection.aggregate([
            {'$match':
                {'$and': [
                    {'date': {'$gte': fromday}},
                    {'date': {'$lte': today}},
                    {'rule': {'$in': list(rules.keys())}}
                ]}},
            {'$group': h}]):
        r = doc['_id']
        data[r] = {}
        for k, v in doc.items():
            if k != '_id':
                data[r][k] = v
    for r, rinfo in data.items():
        rinfo['weight'] = ''
        rinfo['borndate'] = rules[r] if r in rules else ''
        if route.startswith('So'):
            rinfo['spampercent'] = fmtValue(rinfo['spam'] * 100.0 / rinfo['total']) if rinfo['total'] > 0 else '0'
            continue
        rinfo['spampercent'] = fmtValue((rinfo['spam'] + rinfo['spampf'] + rinfo['malic']) * 100.0 / rinfo['total']) \
            if rinfo['total'] > 0 else '0'
        rinfo['nopf'] = fmtValue((rinfo['spam'] + rinfo['hampf'] + rinfo['malic']) * 100.0 / rinfo['total']) \
            if rinfo['total'] > 0 else '0'
        rinfo['cmpl_spam_percent'] = fmtValue(rinfo['cmpl_spam_nopf'] * 100.0 / rinfo['ham'], 6) \
            if rinfo['ham'] > 0 else ''
        rinfo['cmpl_ham_percent'] = fmtValue(rinfo['cmpl_ham_nopf'] * 100.0 / rinfo['spam'], 6) \
            if rinfo['spam'] > 0 else ''
    if params['choice'] == 2:
        for r in data:
            if params['sp_min'] and data[r]['spampercent'] < params['sp_min'] or params['sp_max'] and \
                    data[r]['spampercent'] > params['sp_max'] or params['nopf_min'] and \
                    data[r]['nopf'] < params['nopf_min'] or params['nopf_max'] and \
                    data[r]['nopf'] > params['nopf_max'] or params['total_min'] and \
                    data[r]['total'] < params['total_min'] or params['total_max'] and \
                    data[r]['total'] > params['total_max'] or params['cs_nopf_min'] and \
                    data[r]['cmpl_spam_nopf'] < params['cs_nopf_min'] or params['cs_nopf_max'] and \
                    data[r]['cmpl_spam_nopf'] > params['cs_nopf_max'] or params['ch_nopf_min'] and \
                    data[r]['cmpl_ham_nopf'] < params['ch_nopf_min'] or params['ch_nopf_max'] and \
                    data[r]['cmpl_ham_nopf'] > params['ch_nopf_max'] or params['cs_min'] and \
                    data[r]['cmpl_spam'] < params['cs_min'] or params['cs_max'] and \
                    data[r]['cmpl_spam'] > params['cs_max'] or params['ch_min'] and \
                    data[r]['cmpl_ham'] < params['ch_min'] or params['ch_max'] and \
                    data[r]['cmpl_ham'] > params['ch_max']:
                del data[r]
                del rules[r]
                rules_count -= 1
        rules_count = len(rules)
    for d in RULES_DIRS[route]:
        for filename in glob(d + '*.rul'):
            if not os.path.isfile(filename):
                continue
            cnt += gatherWeights(data, filename, 'rul', route)
            if cnt >= rules_count:
                break
        for filename in glob(d + '*.dlv'):
            if not os.path.isfile(filename):
                continue
            cnt += gatherWeights(data, filename, 'dlv', route)
            if cnt >= rules_count:
                break
    for r in data:
        for par in ['type', 'weight', 'text', 'file']:
            if par not in data[r]:
                data[r][par] = ''
    if params['choice'] == 2:
        text_re = re.compile(r'\bbexpr\b|\barithmetic\b|\bR_ANTI\b')
        w_re = re.compile(r'^\d+')
        for r in data:
            try:
                w_m = w_re.match(data[r]['weight'])
                w = float(data[r]['weight']) if w_m else 0
                if w_m and (params['heavy_min'] and w < params['heavy_min'] or params['heavy_max'] and
                            w > params['heavy_max']) or not w_m and (params['heavy_min'] or params['heavy_max']) or \
                            params['is_atomic'] and text_re.search(data[r]['text']):
                    del data[r]
                    del rules[r]
                    rules_count -= 1
            except Exception as e:
                LOGGER.error(f"Error while filtering DB data by weight and atomicity condition for rule {r}: " +
                             f"{e}.\t{getTraceback()}")
        rules_count = len(rules)
        if params['max_rules_count'] and rules_count > params['max_rules_count']:
            for i in range(rules_count):
                if i >= params['max_rules_count']:
                    del data[r]
                    del rules[r]
                    rules_count -= 1
                    i -= 1
    LOGGER.warning(f"getRulesData: result={data}.")
    return data


def _printRow(rule, route, ruledata):
    table = ""
    table += u"""<tr><td style="text-align: left;"><a href="{0}?rule={1}&route={2}">{1}</a></td>
        <td style="text-align:center;padding: 2px 0.2em;">{3}</td>
        <td style="text-align:center;white-space:nowrap;padding: 2px 0.4em;">{4}</td>
        <td style="text-align:center;padding: 2px 0.3em;">{5}</td>""".\
        format(settings.CFG['showrule']['path'], rule, route, ruledata[rule]['weight'], ruledata[rule]['borndate'],
               ruledata[rule]['spam'])
    if not route.startswith('So'):
        table += '<td style="text-align:center;padding: 2px 0.3em;">{}</td>'.format(ruledata[rule]['spampf'])
    table += u'<td style="text-align:center;padding: 2px 0.3em;">{}</td>'.format(ruledata[rule]['ham'])
    if not route.startswith('So'):
        table += u'<td style="text-align:center;padding: 2px 0.3em;">{}</td>'.format(ruledata[rule]['hampf'])
    table += u'<td style="text-align:center;padding: 2px 0.3em;">{}</td>'.format(ruledata[rule]['malic'])
    if route.startswith('So'):
        table += u'<td style="text-align:center;padding: 2px 0.3em;">{}</td>'.format(ruledata[rule]['spampercent'])
    else:
        table += u'<td style="text-align:center;padding: 2px 0.2em;">{} ({})</td>'.format(
            ruledata[rule]['spampercent'], ruledata[rule]['nopf'])
    table += u'<td style="text-align:center;padding: 2px 0.3em;">{0}</td>'.format(ruledata[rule]['total'])
    if not route.startswith('So'):
        table += u"""<td style="text-align:center;padding: 2px 0.2em;">{} ({})</td>
            <td style="text-align:center;padding: 2px 0.3em;">{}</td>
            <td style="text-align:center;padding: 2px 0.2em;">{} ({})</td>
            <td style="text-align:center;padding: 2px 0.3em;">{}</td>""".\
            format(ruledata[rule]['cmpl_spam_nopf'], ruledata[rule]['cmpl_spam'],
                   ruledata[rule]['cmpl_spam_percent'], ruledata[rule]['cmpl_ham_nopf'],
                   ruledata[rule]['cmpl_ham'], ruledata[rule]['cmpl_ham_percent'])
    table += u'<td style="text-align:left;">{0}</td></tr>'.format(eFilter(ruledata[rule].get('describe', '')))
    return table


def buildRulesTable(table_title: str, tp: str, sortorder: str, ruledata: dict, route: str) -> str:
    n = 1 if tp == 'rul' else (2 if tp == 'dlv' else 3)
    rules = list(filter(lambda r: ruledata[r]['type'] and ruledata[r]['type'] == tp, ruledata.keys())) \
        if n < 3 else list(filter(lambda r: not ruledata[r]['type'], ruledata.keys()))
    table_type = '' if len(rules) > 0 else 'style="display:none;"'
    table = f'<a name="table{n}"/>'
    if len(rules) > 0:
        table += u'<hr><h4>{}</h4>'.format(table_title)
    table += u"""<table cellspacing="0" cellpadding="2" border="1" id="table{0}" {1}> <thead><tr>
        <th rowspan=2><a class="headsort" id="rule" onClick="ChangeSortOrder(this,{0});" title="Имя правила"> Rule </a>
        </th>
        <th align="center" rowspan=2 style="padding: 2px 0.2em;"><a class="headsort" id="weight"
            onClick="ChangeSortOrder(this,{0});" title="Вес правила"> Weight </a></th>
        <th align="center" rowspan=2 style="padding: 2px 0.2em;"><a class="headsort" id="borndate"
            onClick="ChangeSortOrder(this,{0});" title="Дата появления правила в статистике"> BornDate </a></th>
        <th style="padding: 2px 0.2em;" rowspan=2><a class="headsort" id="spam" onClick="ChangeSortOrder(this,{0});"
            title="{2}"> {3} </a></th>""".format(n, table_type, PARAM['spam'][1], PARAM['spam'][0])
    if not route.startswith('So'):
        table += u"""<th style="padding: 2px 0.2em;" rowspan=2><a class="headsort" id="spamPF"
            onClick="ChangeSortOrder(this,{});" title="{}"> {} </a></th>""".\
            format(n, PARAM['spampf'][1], PARAM['spampf'][0])
    table += u"""<th style="padding: 2px 0.2em;" rowspan=2><a class="headsort" id="ham"
        onClick="ChangeSortOrder(this,{});" title="{}"> {} </a></th>""".format(n, PARAM['ham'][1], PARAM['ham'][0])
    if not route.startswith('So'):
        table += u"""<th style="padding: 2px 0.2em;" rowspan=2><a class="headsort" id="hamPF"
        onClick="ChangeSortOrder(this,{});" title="{}"> {} </a></th>""".format(n, PARAM['hampf'][1], PARAM['hampf'][0])
    table += u"""<th style="padding: 2px 0.2em;" rowspan=2><a class="headsort" id="malic"
        onClick="ChangeSortOrder(this,{});" title="{}"> {} </a></th>""".format(n, PARAM['malic'][1], PARAM['malic'][0])
    if route.startswith('So'):
        table += u"""<th align="center" rowspan=2 style="padding: 2px 0.2em;"><a class="headsort" id="spampercent"
            onClick="ChangeSortOrder(this,{});" title="{}"> {} </a></th>""".\
            format(n, PARAM['spampercent'][1], PARAM['spampercent'][0])
    else:
        table += u"""<th align="center" rowspan=2 style="padding: 2px 0.2em;"><a class="headsort" id="spampercent"
            onClick="ChangeSortOrder(this,{});" title="{} ({})"> {} ({}) </a></th>""".\
            format(n, PARAM['spampercent'][1], PARAM['nopf'][1], PARAM['spampercent'][0], PARAM['nopf'][0])
    table += u"""<th style="padding: 2px 0.2em;" rowspan=2><a class="headsort" id="total"
        onClick="ChangeSortOrder(this,{});" title="{}"> {} </a></th>""".format(n, PARAM['total'][1], PARAM['total'][0])
    if route.startswith('So'):
        table += u'<th rowspan=2> Rule\'s description </th> </tr><tr>'
    else:
        table += u"""<th style="padding: 2px 0.2em;" colspan=2 title="Жалобы на спам"> Cmpl_spam </th>
            <th style="padding: 2px 0.2em;" colspan=2 title="Жалобы на хам"> Cmpl_ham </th>
            <th rowspan=2> Rule's description </th> </tr><tr>
            <th><a class="headsort" id="cmpl_spam_nopf" onClick="ChangeSortOrder(this,{0});" title="{1}"> {2} </a></th>
            <th><a class="headsort" id="cmpl_spam_percent" onClick="ChangeSortOrder(this,{0});" title="{3}"> {4}
                </a></th>
            <th><a class="headsort" id="cmpl_ham_nopf" onClick="ChangeSortOrder(this,{0});" title="{5}"> {6} </a></th>
            <th><a class="headsort" id="cmpl_ham_percent" onClick="ChangeSortOrder(this,{0});" title="{7}"> {8}
                </a></th>""".\
            format(n, PARAM['cmpl_spam_nopf'][1], PARAM['cmpl_spam_nopf'][0], PARAM['cmpl_spam_percent'][1],
                   PARAM['cmpl_spam_percent'][0], PARAM['cmpl_ham_nopf'][1], PARAM['cmpl_ham_nopf'][0],
                   PARAM['cmpl_ham_percent'][1], PARAM['cmpl_ham_percent'][0])
    table += u'</tr></thead><tbody>'
    try:
        for r in sorted(
                filter(lambda x: ruledata[x][sortorder], rules),
                key=cmp_to_key(lambda a, b: (a <= b) if sortorder == 'rule' else
                               ((ruledata[b][sortorder] <= ruledata[a][sortorder]) if sortorder == 'borndate' else
                                   int((float(ruledata[b][sortorder]) - float(ruledata[a][sortorder])) * 1000)))):
            table += _printRow(r, route, ruledata)
        for r in sorted(filter(lambda x: not ruledata[x][sortorder], rules)):
            table += _printRow(r, route, ruledata)
    except Exception as e:
        LOGGER.error(f"Exception: {e}.\t{getTraceback()}")
    table += u'</tbody></table>'
    if len(rules) > 0:
        table += u'<br />'
    return table
