# -*- coding: utf-8 -*-
# kate: space-indent on; indent-width 4; replace-tabs on;
#
import requests
from typing import Optional
from tvmauth import BlackboxTvmId
from tvm2 import TVM2
from tvm2.aio.thread_tvm2 import TVM2 as TVM2Thread
from django.conf import settings


def get_tvm2_client(blackbox_client_id: BlackboxTvmId) -> TVM2Thread:
    return TVM2Thread(
        client_id=settings.TVM_CLIENT_ID,
        secret=settings.TVM_SECRET,
        blackbox_client=blackbox_client_id,
        allowed_clients=[settings.TVM_CLIENT_ID, ],
    )


# for some reason without this lines client.get_service_ticket
# in get_service_ticket resolves to
# https://a.yandex-team.ru/arc/trunk/arcadia/library/python/tvm2/tvm2/protocol.py?rev=r7035598#L92
# https://st.yandex-team.ru/CHEMODAN-76581
try:
    prod_client = get_tvm2_client(BlackboxTvmId.Prod)
    yateam_client = get_tvm2_client(BlackboxTvmId.ProdYateam)
except Exception:
    pass


async def get_service_ticket(blackbox_client_id: BlackboxTvmId, dst_tvm2_id: str) -> Optional[str]:
    client = get_tvm2_client(blackbox_client_id)
    return await client.get_service_ticket(dst_tvm2_id)


class TVM2Mixin(object):
    def __init__(self, blackbox_tvm2_client: BlackboxTvmId, dst_tvm2_id: str):
        self.blackbox_tvm2_client = blackbox_tvm2_client
        self.dst_tvm2_id = dst_tvm2_id
        self.tvm2_client = TVM2(
            client_id=settings.TVM_CLIENT_ID,
            secret=settings.TVM_SECRET,
            blackbox_client=blackbox_tvm2_client,
            destinations=(dst_tvm2_id,),
        )
        self._ticket = self._service_ticket(False)

    def _service_ticket(self, cached: bool = True) -> str:
        if cached and hasattr(self, '_ticket') and self._ticket:
            return self._ticket
        self._ticket = self.tvm2_client.get_service_ticket(self.dst_tvm2_id)
        return self._ticket

    def spec_headers(self, cached: bool = True) -> dict:
        return {'X-Ya-Service-Ticket': self._service_ticket(cached)}

    def request(self, url: str, **kwargs) -> requests.Response:
        s = requests.Session()
        s.headers.update(self.spec_headers())
        return s.get(url, **kwargs)
