# -*- coding: utf-8 -*-
# kate: space-indent on; indent-width 4; replace-tabs on;
#
from __future__ import unicode_literals
import logging
import requests
import asyncio
import json
import re
from collections.abc import Callable, Mapping
from concurrent.futures import ThreadPoolExecutor
from django.conf import settings

LOGGER = logging.getLogger(__name__)


def querySologger(session: requests.Session, route: str, params: str) -> tuple[str, str]:
    result = {"data": ""}
    try:
        url = "http://{}{}{}&route={}&getbyid=1".format(
            settings.CFG["sologger"]["host"],
            settings.CFG["sologger"]["uri"],
            params,
            route)
        LOGGER.warning(f"Query to Sologger: route={route}, params={params}, uri={url}")
        with session.get(url, timeout=settings.CFG["sologger"]["timeout"]) as response:
            result['data'] = response.text if response.text else ""
            if response.status_code != 200:
                LOGGER.error("Query to Sologger ({}) failed with code={}: {}".format(
                    url, response.status_code, response.reason))
                result["error"] = \
                    f"Sologger request failed for route={route}: {response.status_code} {response.reason}."
    except Exception as e:
        LOGGER.error(f"Query to Sologger (route={route}, uri={url}) failed: {e}")
        s = e.strerror if hasattr(e, 'strerror') else (e.reason if hasattr(e, 'reason') else str(e))
        if "error" in result:
            result["error"] += f" Also error: {s}."
        else:
            result["error"] = f"Sologger request failed for route={route}: {s}"
    return route, result


def querySologgerIndex(
        session: requests.Session,
        route: str,
        uid: str | int,
        ts: str | int,
        skip: str | int,
        limit: str | int) -> tuple[str, str]:
    result = {"data": ""}
    try:
        uid_field = "log_{}uid".format("" if route == "out" else "rcpt_")
        query = ("get=id,{0},log_queueid,log_code,log_ts&text={0}:{1}+AND+log_ts_day:{2}+AND+log_route:{3}" +
                 "&skip={4}&limit={5}").format(uid_field, uid, ts, route, skip, limit)
        url = "http://{}{}{}".format(settings.CFG["sologger"]["host"], settings.CFG["sologger"]["index_uri"], query)
        LOGGER.warning(f"Query to Sologger's index, uri: {url}")
        with session.get(url, timeout=settings.CFG["sologger"]["timeout"]) as response:
            result['data'] = response.text if response.text else ""
            if response.status_code != 200:
                LOGGER.error("Query to Sologger's index ({}) failed with code={}: {}".format(
                    url, response.status_code, response.reason))
                result["error"] = \
                    f"Request to Sologger's index failed for route={route}: {response.status_code} {response.reason}."
    except Exception as e:
        LOGGER.error(f"Query to Sologger's index ({url}) failed: {e}")
        s = e.strerror if hasattr(e, 'strerror') else (e.reason if hasattr(e, 'reason') else str(e))
        if "error" in result:
            result["error"] += f" Also error: {s}."
        else:
            result["error"] = f"Request to Sologger's index failed for route={route}: {s}"
    return route, result


def nextRow(payload):
    i = j = 0
    for c in payload:
        j += 1
        if c == "\n":
            yield payload[i:j]
            i = j
    if i < j:
        yield payload[i:j]


def initialParseDlvLog(data: str) -> tuple[str, str]:
    try:
        js = json.loads(data)
    except Exception as e:
        LOGGER.error(f"getDlvLog failed: '{e}'")
        js = {}
    if len(js) > 10:
        key, value = "", js
        if js[1][0] == 'locl':
            m = re.search(r'-(in|out|corp)-', js[1][1])
            if m:
                key = m.group(1)
        else:
            LOGGER.error(f"getDlvLog unable to find route for locl: {js[1]}")
        if js[0][0] == 'mess':
            m = re.match(r'\S+\s\d+\s\S+\s\S+\s(\d+)', js[0][1])
            if m:
                key += '_' + m.group(1)
        else:
            LOGGER.error(f"getDlvLog unable to find message's timestamp for mess: {js[0]}")
        return key, value
    else:
        LOGGER.error(f"getDlvLog failed to parse Sologger answer's row: '{data}'")
    return "", ""


async def getSologgerData(
        routes: list,
        querySologgerFunc: Callable[..., tuple[str, str]],
        *args) -> Mapping[str, str]:

    result = {"data": ""}
    with ThreadPoolExecutor(max_workers=8) as executor:
        with requests.Session() as session:
            loop = asyncio.get_event_loop()
            tasks = [
                loop.run_in_executor(executor, querySologgerFunc, *(session, route, *args)) for route in routes
            ]
            for future in asyncio.as_completed(tasks, timeout=settings.CFG["sologger"]["timeout"]):
                try:
                    route, response = await future
                    result[route] = response["data"]
                    if 'error' in response:
                        if 'error' in result:
                            result["error"] += f"\n{response['error']}"
                        else:
                            result["error"] = response['error']
                except Exception as e:
                    s = e.strerror if hasattr(e, 'strerror') else (e.reason if hasattr(e, 'reason') else str(e))
                    if not s:
                        s = 'скорее всего, следует сузить круг поиска'
                    if 'error' in result:
                        result["error"] += f"\nAsync requests to Sologger failed: {s}"
                    else:
                        result["error"] = f'Async requests to Sologger failed: {s}'
    return result


def querySologgerData(
        routes: list,
        querySologgerFunc: Callable[..., tuple[str, str]],
        *args) -> Mapping[str, str]:

    loop = asyncio.get_event_loop()
    LOGGER.warning(f"Query to Sologger for routes={routes}, args={args}")
    future = asyncio.ensure_future(getSologgerData(routes, querySologgerFunc, *args))
    loop.run_until_complete(future)
    return dict(future.result())
