# coding: utf8
# kate: space-indent on; indent-width 4; replace-tabs on;
#
import json
import logging
import requests
from django.conf import settings
from django.http import HttpRequest, JsonResponse, HttpResponseServerError, HttpResponse
from django.views.decorators.csrf import csrf_exempt
from django.views.generic import TemplateView
from tvm2 import TVM2
from tvmauth import BlackboxTvmId

from mail.so.libs.python.cast import to_longlong, to_ulonglong
from mail.so.spamstop.web.ui.web_tools.libs import shingle

LOGGER = logging.getLogger(__name__)
WEIGHTS = []


class ComplView(TemplateView):
    template_name = 'complshingler.html'


class UnbanView(TemplateView):
    template_name = 'unban.html'


def backend_session() -> requests.Session:
    tvm = TVM2(
        client_id=settings.TVM_CLIENT_ID,
        secret=settings.TVM_SECRET,
        blackbox_client=BlackboxTvmId.Prod,
        allowed_clients=tuple(),
        destinations=(settings.SENDERS_TVM_ID,),
    )
    s = requests.Session()
    s.headers.update(
        {'X-Ya-Service-Ticket': tvm.get_service_ticket(settings.SENDERS_TVM_ID)})
    return s


@csrf_exempt
def request_shingles(request: HttpRequest):
    data = json.loads(request.body)
    texts = data["texts"]
    shingles = [shingle(text) for text in texts]
    return JsonResponse({'shingles': shingles})


@csrf_exempt
def fetch_user_weights(request: HttpRequest):
    with backend_session() as s:
        response = s.get(settings.SOSEARCH_HOST + "/api/async/so/get-user-weights",
                         timeout=settings.SOSEARCH_TIMEOUT)
        response.raise_for_status()

        weights = response.json()

        for w in weights:
            w["shingle"] = hex(to_ulonglong(int(w["shingle"])))

        return JsonResponse({"weights": weights})


@csrf_exempt
def set_weights(request: HttpRequest):
    try:
        data = json.loads(request.body)
        shingles = data.pop("shingles")
        LOGGER.info(f'got data to ban:{data}')

        failed_shingles, status_code = [], 200
        with backend_session() as s:
            for sh in shingles:
                data["shingle"] = to_longlong(int(sh, 16))

                response = s.get(settings.SOSEARCH_HOST + "/api/async/so/update-user-weight",
                                 params=data,
                                 timeout=settings.SOSEARCH_TIMEOUT)

                LOGGER.info(f'got response:{response}')
                status_code = response.status_code
                if response.status_code != 200:
                    failed_shingles.append({sh: response.reason, "code": status_code})

        if failed_shingles:
            resp = JsonResponse({"failed": failed_shingles})
            resp.status_code = status_code
            LOGGER.error(f'Failed to process shingles: {failed_shingles}. Error: {resp.status_code} {resp.reason_phrase}')
            return resp
        else:
            return HttpResponse(content=f"Updated {len(shingles)} shingles", status=200)
    except Exception as e:
        LOGGER.error(f'Failed to set weights: {e}')
        return HttpResponseServerError(str(e))
