# coding: utf8
# kate: space-indent on; indent-width 4; replace-tabs on;
#
from __future__ import unicode_literals
import re
import json
import logging

from functools import cmp_to_key
from collections.abc import Mapping
from time import mktime, strptime, strftime, time, localtime
from urllib.parse import quote
from django.conf import settings
from django.views.generic import TemplateView
from django.http import HttpRequest, HttpResponse
from mail.so.spamstop.web.ui.web_tools.libs import ipReverse, get_blackbox_client, querySologgerData, querySologger


LOGGER = logging.getLogger(__name__)
routeOrder = {'in': 2, 'out': 1, 'corp': 0}
RE_RULE = re.compile(r'^(\S+)\s+(\S+)$')


def _renderCountry(short_country: str, long_country: str) -> str:
    href = '<img src="/images/flags/{0}.gif"></img>'.format(short_country.lower() if short_country else '_')
    return "{0} {1} ({2})".format(href, short_country, long_country if long_country else 'unknown')


def _ip_to_country(sIP):
    reversed_ip = ipReverse(sIP)
    if not reversed_ip:
        return None
    resolve = ''
    # try:
    #    resolve = subprocess.check_output('host -t txt {0}.geozone.yandex.ru georbl.so.yandex.net'.format(reversed_ip), shell=True)
    # except Exception as e:
    #    writelog("Web Get by ID ip_to_country exception: %s" % str(e), True)
    short_country = long_country = ''
    m = re.search(r'descriptive text "([A-Z]{2}) (\S.*?)"', resolve)
    if m:
        short_country, long_country = m.group(1), m.group(2)
    return _renderCountry(short_country, long_country)


def _ip_to_spamsource(sIP):
    reversed_ip = ipReverse(sIP)
    if not reversed_ip:
        return None
    resolve = ''
    # try:
    #    resolve = subprocess.check_output('host {0}.spamsource.yandex.ru georbl.so.yandex.net'.format(reversed_ip), shell=True)
    # except Exception as e:
    #    writelog("Web Get by ID ip_to_spamsource exception: %s" % str(e), True)
    return resolve


def _param2Timestamp(ts: str) -> int:
    return int(mktime(strptime(ts, "%d.%m.%Y %H:%M")))


def _parseRule(s: str) -> tuple[str, str]:
    a = s.split()
    return (a[0], a[1] if len(a) > 1 else "0")


def _parseRules(r_sp_s: str) -> dict:
    return dict(map(
        _parseRule,
        filter(lambda s: len(s.strip()), re.split(r'\s*[,;]\s*', r_sp_s[0]))
    )) if len(r_sp_s) > 0 else {}


def _urlizeRules(dlvlog: str, rule_header: str, rules: Mapping[str, str], uri: str) -> str:
    keys = sorted(rules.keys(), key=cmp_to_key(lambda a, b: float(rules[a]) - float(rules[b])))
    return re.sub(
        r'^{}:.+$'.format(rule_header),
        r'{}: {}\n'.format(
            rule_header,
            ',  '.join(map(lambda r: ('<a style="color:black; text-decoration:none;" class="blacklink" ' +
                                      'href="{0}{1}?rule={2}">{2}</a> {3}').
                                          format(uri, settings.CFG['showrule']['path'], r, rules[r]), keys))),
            dlvlog,
            flags=re.M)


def restoreDlvLog(dataStr: str) -> str:
    if dataStr[0] == '[':       # expecting of array in JSON format
        log, data = "", []
        try:
            data = json.loads(dataStr)
        except:
            try:
                data = json.loads(dataStr.encode('utf8', 'replace'))
            except Exception as e:
                LOGGER.error(f"Parsing of JSON failed: {e}")
        for t in data:
            log += t[0] + ": " + t[1] + "\n"
        return log
    elif dataStr[0] == '{':     # expecting of dict in JSON format
        log, data = "", {}
        try:
            data = json.loads(dataStr)
        except Exception as e:
            LOGGER.error(f"Parsing of JSON failed: {e}")
        for k, v in data.items():
            if isinstance(v, list):
                for it in v:
                    log += k + ": " + it + "\n"
            else:
                log += k + ": " + v + "\n"
        return log
    else:
        return dataStr


class GetByIdView(TemplateView):
    template_name = 'getbyid.html'

    def setup(self, request: HttpRequest, *args, **kwargs) -> None:
        super(GetByIdView, self).setup(request, *args, **kwargs)
        self.ban = False
        self.ip_geo = ''
        self.logs = []
        self.data = []
        self.params = {}
        self.routes = ['in', 'out', 'corp']
        self.records_count = 0
        self.initBlackboxClients()

    def initBlackboxClients(self) -> None:
        self.blackboxProd = get_blackbox_client()
        self.blackboxYaTeam = get_blackbox_client(True)

    def getUserInfo(self, key_type: str, key: str) -> tuple[str, str, str]:
        info = self.blackboxProd.getUserInfo(key_type, key)
        if not info["uid"]:
            info = self.blackboxYaTeam.getUserInfo(key_type, key)
        return info["uid"], info["suid"], info["login"]

    def dispatch(self, request, *args, **kwargs) -> HttpResponse:
        self.uri = request.META['REQUEST_URI'] if 'REQUEST_URI' in request.META else ''
        self.proto = request.scheme
        self.uri = re.sub(r'/?[^/]+$', '', self.uri)
        self.key = request.GET.get("key", "")
        self.key_type = request.GET.get("key_type", "")
        self.uid = request.GET.get("uid", "")
        self.suid = request.GET.get("suid", "")
        self.rcpt_uid = request.GET.get("rcpt_uid", "")
        self.login = request.GET.get("login", "")
        self.fromaddr = request.GET.get("from", "")
        self.msgid = request.GET.get("msgid", "")
        self.qid = request.GET.get("qid", "")
        self.ip = request.GET.get("ip", "")
        self.reg_exp = request.GET.get("reg_exp", "")
        self.limit = request.GET.get("limit", "20")
        self.skip = request.GET.get("skip", "0")
        self.startDate = request.GET.get("start_date", "")
        self.endDate = request.GET.get("end_date", "")
        user = request.yauser if hasattr(request, 'yauser') else request.user
        if hasattr(user, 'login'):
            user = user.login
        LOGGER.warning(f'Input parameters: url={self.proto}://{request.get_host()}{request.path}, args={args}, kwargs={kwargs}, user={user}')
        if self.uid:
            self.key_type, self.key = 'uid', self.uid
        elif self.rcpt_uid:
            self.key_type, self.key = 'rcpt_uid', self.rcpt_uid
        elif self.suid:
            self.key_type, self.key = 'suid', self.suid
        elif self.login:
            self.key_type, self.key = 'login', self.login
        elif self.fromaddr:
            self.key_type, self.key = 'fromaddr', self.fromaddr
        elif self.msgid:
            self.key_type, self.key = 'msgid', self.msgid
        elif self.qid:
            self.key_type, self.key = 'queueid', self.qid
        elif self.ip:
            self.key_type, self.key = 'source_ip', self.ip
        if self.key:
            if request.GET.getlist("cb_route", []):
                self.routes = request.GET.getlist("cb_route", ['in', 'out', 'corp'])
            self.params = {
                'mintime': _param2Timestamp(self.startDate) if self.startDate else 0,
                'limit':   self.limit,
                'skip':    self.skip
            }
            if self.endDate:
                ts = _param2Timestamp(self.endDate)
                if ts >= self.params['mintime']:
                    self.params['maxtime'] = ts
            if request.GET.getlist("cb_filter", []):
                codes = sorted(request.GET.getlist("cb_filter", []), key=int)
                if ','.join(codes) != "1,2,4,8,127,256":
                    self.params['code'] = ','.join(codes)
            if self.key_type == 'uid' or self.key_type == 'suid' or self.key_type == 'login':
                self.uid, self.suid, self.login = self.getUserInfo(self.key_type, self.key)
                self.params["uid"] = self.uid
            elif self.key_type == "rcpt_uid":
                self.rcpt_uid, self.suid, self.login = self.getUserInfo('uid', self.key)
                self.params["rcpt_uid"] = self.key
            elif self.key_type == "msgid":
                self.key = self.key.strip(" \t\n")
                self.params["msgid"] = quote(self.key)
            elif self.key_type == "ip":
                if re.search(r'has address 127.0.0.8$', _ip_to_spamsource(self.key)):
                    self.ban = True
                self.ip_geo = _ip_to_country(self.key)
                self.params["source_ip"] = self.key
            elif self.key_type == "qid":
                self.params["queueid"] = self.key
            elif self.key_type == "from":
                self.params["fromaddr"] = quote(self.key)
            else:
                self.params[self.key_type] = quote(self.key)
            result = []
            LOGGER.warning(f"GetByIdView: routes={self.routes}, params={self.params}.")
            sologgersResult = querySologgerData(
                self.routes,
                querySologger,
                '&'.join(map(lambda it: '{0}={1}'.format(*it), self.params.items())))
            if 'error' in sologgersResult:
                self.error = sologgersResult['error']
            for route, rawLogs in sologgersResult.items():
                if not rawLogs or route == "error":
                    continue
                logsArray = []
                for rawLog in rawLogs.split("\n"):
                    if rawLog:
                        logsArray.append(restoreDlvLog(rawLog))
                for log in logsArray:
                    m = re.search(r"mess:[^-]+-\s*(\d+):", log[:log.find("\n")])
                    ts = int(m.group(1) if m else time())
                    result.append([strftime("%Y-%m-%d", localtime(ts)), routeOrder[route], ts, log])
            if result:
                self.records_count = len(result)
                try:
                    self.data = map(lambda block: block[3], sorted(result, reverse=True))
                    # self.records_count = len(self.data)
                except Exception as e:
                    LOGGER.error(f"Decoding JSON exception: {e}")
            else:
                LOGGER.warning('Search has empty result!')
            self.limit = int(self.limit) if self.limit and self.limit.isdigit() and int(self.limit) > 0 else 0
            for i, dlvlog in enumerate(self.data, 1):
                if self.reg_exp and not re.search(self.reg_exp, dlvlog) or self.limit and i > self.limit or \
                        not re.match(r'^mess:', dlvlog.strip()):
                    continue
                m, country = re.search(r'^iy-geozone:\s+(\w+)\s+(\w+)', dlvlog, re.M), ''
                if m and m.group(1):
                    country = _renderCountry(m.group(1), m.group(2))
                else:
                    m = re.search(r'source ip =\s*(\S+)', dlvlog)
                    if m and m.group(1):
                        country = _ip_to_country(m.group(1))
                dlvlog = re.sub(r'&', r'&amp;', dlvlog)
                dlvlog = re.sub(r'<', r'&lt;', dlvlog)
                dlvlog = re.sub(r'>', r'&gt;', dlvlog)
                r_sp = _parseRules(re.findall(r'^r_sp:\s(.+)$', dlvlog, re.M))
                r_dl = _parseRules(re.findall(r'^r_dl:\s(.+)$', dlvlog, re.M))
                r_nl = _parseRules(re.findall(r'^r_nl:\s(.+)$', dlvlog, re.M))
                m = re.match(r'^mess: (.*)$', dlvlog, re.M)
                sdh = m.group(1) if m else ''
                sdh = re.sub(
                    r'\bid=(\d+)',
                    r"id=<a target='_blank' href='{}/{}?suid=\1'>\1</a>".format(
                        self.uri,
                        settings.CFG['users_abuse']['path']),
                    sdh)
                sdh = re.sub(
                    r'\buid=(\d+)',
                    r"uid=<a target='_blank' href='{}/{}?uid=\1'>\1</a>".format(
                        self.uri,
                        settings.CFG['users_abuse']['path']),
                    sdh)
                dlvlog = re.sub(r'^mess: (.*)$', r"mess: {}".format(sdh), dlvlog, flags=re.M)
                dlvlog = _urlizeRules(dlvlog, 'r_sp', r_sp, self.uri)
                dlvlog = _urlizeRules(dlvlog, 'r_dl', r_dl, self.uri)
                dlvlog = _urlizeRules(dlvlog, 'r_nl', r_nl, self.uri)
                dlvlog = (u'<a href="{}{}?rules={}" target="_blank"><B>[Rules\' statistics]</B></a>\n{}').\
                    format(
                        self.uri,
                        settings.CFG['statrules']['path'],
                        ','.join([','.join(r_sp.keys()), ','.join(r_dl.keys()), ','.join(r_nl.keys())]),
                        dlvlog)
                dlvlog = re.sub(r'^([^:]+:)', r'<b>\1</b>', dlvlog, flags=re.M)   # bold fieldnames
                if country:
                    dlvlog = u'<h4>geo: {}</h4>{}'.format(country, dlvlog)
                s0 = ''
                for s in dlvlog.splitlines():
                    s0 += u"{}<br/>\n".format(s)
                self.logs.append(u"<p><tt>{}</tt></p>".format(s0))
        else:
            LOGGER.warning('Unknown key for start search!')
        return super(GetByIdView, self).dispatch(request, *args, **kwargs)

    def get_context_data(self, **kwargs):
        context = super(GetByIdView, self).get_context_data(**kwargs)
        context['uri'] = self.uri
        context['key'] = self.key
        context['key_type'] = self.key_type
        context['uid'] = self.uid
        context['suid'] = self.suid
        context['rcpt_uid'] = self.rcpt_uid
        context['login'] = self.login
        context['from'] = self.fromaddr
        context['msgid'] = self.msgid
        context['qid'] = self.qid
        context['ip'] = self.ip
        context['reg_exp'] = self.reg_exp
        context['records_count'] = self.records_count
        context['skip'] = self.skip
        context['limit'] = self.limit
        context['start_date'] = self.startDate
        context['end_date'] = self.endDate
        context['cb_route'] = self.routes
        context["cb_filter"] = self.params.get('code', '').split(',') \
            if self.params.get('code', '') else ['1', '2', '4', '8', '127', '256']
        context['error'] = self.error if hasattr(self, 'error') and self.error else ''
        return context
