# coding: utf8
# kate: space-indent on; indent-width 4; replace-tabs on;
#
from __future__ import unicode_literals
import re
import json
import logging
from django.conf import settings
from django.views.generic import TemplateView
from django.http import HttpRequest, HttpResponse
from datetime import date, timedelta
from mail.so.spamstop.web.ui.web_tools.libs import (
    class_based_login_required, RULES_TYPES, RULE_PARAM, PARAM, getRuleData, encodeBase64Url, eFilter, getTraceback
)

LOGGER = logging.getLogger(__name__)


@class_based_login_required
class ShowRuleView(TemplateView):
    template_name = "showrule.html"

    def setup(self, request: HttpRequest, *args, **kwargs) -> None:
        super(ShowRuleView, self).setup(request, *args, **kwargs)
        self.routes = list(map(lambda rt: (rt[0], rt[1]), RULES_TYPES))
        self.rules = []
        self.rule_info = {}
        self.rule_data = {}
        self.rule_params = []
        self.rule_params_info = dict(map(
            lambda it: (it[0], re.sub(r'\s+\(.+?\)', r'', re.sub(r'<br/>', r' ', it[1][0]))),
            PARAM.items()))
        self.PARAM = PARAM
        self.showrule_path = settings.CFG['showrule']['path']

    def dispatch(self, request, *args, **kwargs) -> HttpResponse:
        LOGGER.info(f"Request: {request}")
        self.uri = request.META['REQUEST_URI'] if 'REQUEST_URI' in request.META else ''
        self.proto = request.scheme
        self.uri = re.sub(r'/?[^/]+$', '', self.uri)
        self.action = request.GET.get("action", "show")
        self.route = request.GET.get("route", "In")
        self.rule = request.GET.get("rule", "")
        self.period = int(request.GET.get("period", "3"))
        self.sortorder = request.GET.get("sortorder", "date")
        self.update_rules = request.GET.get("update_rules", "on")
        self.isregexp = request.GET.get("isregexp", "")
        self.detailed = int(request.GET.get("detailed", "0"))
        self.day = request.GET.get("day", date.today().isoformat())
        self.view_type = request.GET.get("view", "table")
        self.data = {'dates': {}, 'isregexp': self.isregexp, 'rules_count': 0}
        self.rule_params = RULE_PARAM[self.route]
        if self.action == 'show':
            if self.rule:
                y, m, md = map(int, self.day.split('-'))
                if self.detailed:
                    mdc = {
                        1: 31,
                        2: 31,
                        3: 29 if y % 400 == 0 or y % 4 == 0 and y % 100 != 0 else 28,
                        4: 31,
                        5: 30,
                        6: 31,
                        7: 30,
                        8: 31,
                        9: 31,
                        10: 30,
                        11: 31,
                        12: 30
                    }
                    self.data['dates']['day'] = self.day
                    self.data['dates']['to'] = (date(y, m, md) + timedelta(days=1)).isoformat()
                    self.data['dates']['tomorrow'] = self.data['dates']['to']
                    self.data['dates']['from'] = f"{self.data['dates']['day']} 00:00"
                    self.data['dates']['to'] = f"{self.data['dates']['to']} 00:00"
                    self.data['dates']['yesterday'] = (date(y, m, md) - timedelta(days=1)).isoformat()
                    self.data['dates']['today'] = date.today().isoformat()
                    self.data['dates']['month_ago'] = (date(y, m, md) - timedelta(days=mdc[m])).isoformat()
                    self.data['dates']['period'] = f"сутки {self.data['dates']['day']}"
                else:
                    if self.period == 99:
                        self.data['dates']['period'] = u'всё время'
                    else:
                        p1 = self.period * 30
                        self.data['dates']['to'] = date.today().isoformat()
                        self.data['dates']['from'] = (date(y, m, md) - timedelta(days=p1)).isoformat()
                        self.data['dates']['period'] = f"период с {self.data['dates']['from']} по " + \
                            f"{self.data['dates']['to']}"
                self.rule_data.update(getRuleData(
                    self.rule,
                    self.route,
                    self.detailed,
                    self.day,
                    self.period,
                    self.sortorder,
                    self.data))
                if self.data['rules_count'] == 1:
                    self.data['dates']['from_z'] = self.data['dates']['from'].replace(' ', '+') \
                        if self.detailed else (self.data['dates']['from'] + '+00:00')
                    self.data['dates']['to_z'] = self.data['dates']['to'].replace(' ', '+') \
                        if self.detailed else (self.data['dates']['to'] + '+00:00')
                    try:
                        if self.view_type == 'graph':
                            metric = {
                                'spampercent': "(so.{0}rules.{1}.{2}.spam + so.{0}rules.{1}.{2}.spampf + so.{0}rules.{1}.{2}.malic) * 100.0 / so.{0}rules.{1}.{2}.total",
                                'nopf': "(so.{0}rules.{1}.{2}.spam + so.{0}rules.{1}.{2}.hampf + so.{0}rules.{1}.{2}.malic) * 100.0 / so.{0}rules.{1}.{2}.total",
                                'cmpl_spam_percent': "so.{0}rules.{1}.{2}.cmpl_spam_nopf * 100.0 / so.{0}rules.{1}.{2}.ham",
                                'cmpl_ham_percent': "so.{0}rules.{1}.{2}.cmpl_ham_nopf * 100.0 / so.{0}rules.{1}.{2}.spam",
                                '_default_': "so.{0}rules.{1}.{2}.{3}"
                            }
                            self.data['chart'] = encodeBase64Url(json.dumps({
                                'charts': [{
                                    'graphs': list(map(lambda p: {
                                        'color':  PARAM[p][3],
                                        'metric': metric[p].format(
                                            'detail' if self.detailed else '',
                                            self.rule,
                                            self.route.lower())
                                                if p in metric else metric['_default_'].format(
                                                    'detail' if self.detailed else '',
                                                    self.rule,
                                                    self.route.lower(),
                                                    p),
                                        'title': PARAM[p][2],
                                        'fill': 0
                                    }, RULE_PARAM[self.route])),
                                    'height':        '480px',
                                    'width':         '98%',
                                    'marginLeft':    80,
                                    'marginRight':   60,
                                    'stacktype':     'none',
                                    'balloonformat': 'YYYY-MM-DD JJ:NN' if self.detailed else 'YYYY-MM-DD',
                                    'aggregation':   'AVG',
                                    'valuegrid':     1,
                                    'minperiod':     600 if self.detailed else 86400
                                }]
                            }))
                        for r, rd in self.rule_data.items():
                            if 'text' in rd:
                                rd['text'] = eFilter(rd['text']).replace(u'\n', r'</p><p>').replace(u'\t', r'&nbsp;')
                                rd['text'] = re.sub(r'^rule\s+([A-Z0-9_]+)', r'rule <b>\1</b>', rd['text'])
                            else:
                                rd['text'] = ''
                    except Exception as e:
                        LOGGER.error(f"Error: {e}.\t{getTraceback()}")
                else:
                    for r in self.rule_data.keys():
                        self.rules.append(r)
                self.rule_info = self.rule_data[self.rule] if self.rule in self.rule_data else {}
            else:
                LOGGER.warning("Unknown value for rule!")
        else:
            LOGGER.warning(f"Unacceptable value for 'action': '{self.action}'.")
        LOGGER.warning(f"Data: {self.data}. RuleData: {self.rule_data}. RuleInfo: {self.rule_info}. ViewType: {self.view_type}.")
        return super(ShowRuleView, self).dispatch(request, *args, **kwargs)

    def get_context_data(self, **kwargs):
        context = super(ShowRuleView, self).get_context_data(**kwargs)
        context["uri"] = f"{settings.CFG['showrule']['path']}?rule={self.rule}&route={self.route}&sortorder={self.sortorder}"
        context["route"] = self.route
        context["route_prefix"] = self.route[:2]
        context["data"] = self.data
        context["action"] = self.action
        context["routes"] = self.routes
        context["rule_data"] = self.rule_data
        context["rules"] = sorted(self.rules)
        context["rule"] = self.rule
        context["rule_info"] = self.rule_info
        context["detailed"] = self.detailed
        context["period"] = self.period
        context["day"] = self.day
        context["sortorder"] = self.sortorder
        context["isregexp"] = self.isregexp
        context["rule_params"] = self.rule_params
        context["rule_params_info"] = self.rule_params_info
        context["showrule_path"] = self.showrule_path
        context["view_type"] = self.view_type
        context["PARAM"] = PARAM
        return context
