# coding: utf8
# kate: space-indent on; indent-width 4; replace-tabs on;
#
from __future__ import unicode_literals
import re
import logging
from datetime import datetime
from functools import cmp_to_key
from django.conf import settings
from django.views.generic import TemplateView
from django.http import HttpRequest, HttpResponse
from mail.so.spamstop.web.ui.web_tools.libs import (
    class_based_login_required, getPGCredentials, getPGdb, get_blackbox_client
)

LOGGER = logging.getLogger(__name__)
CHUNK_SIZE = 10000

cmplSrc = {
    'wmi':      'Web',
    'imap':     'IMAP',
    'mobile':   'Mobile',
    'otrs':     'OTRS',
    'crm':      'CRM',
    'furita':   'Furita',
    'fastsrv':  'FastSrv',
    'node_js':  'NodeJS',
    'exchange': 'MSExchange',
    'fblin':    'FBL'
}
fooSrc = ['complaint on ham', 'complaint on spam', 'unsubscribe', 'move to another folder', 'set flag']


@class_based_login_required
class UserAbusesView(TemplateView):
    template_name = "user_abuses.html"

    def setup(self, request: HttpRequest, *args, **kwargs) -> None:
        super(UserAbusesView, self).setup(request, *args, **kwargs)
        self.initBlackboxClients()

    def initBlackboxClients(self) -> None:
        self.blackboxProd = get_blackbox_client()
        self.blackboxYaTeam = get_blackbox_client(True)

    def getUserInfo(self, key_type: str, key: str) -> tuple[str, str, str]:
        info = self.blackboxProd.getUserInfo(self.key_type, self.key)
        if not info["uid"]:
            info = self.blackboxYaTeam.getUserInfo(self.key_type, self.key)
        return info["uid"], info["suid"], info["login"]

    def dispatch(self, request, *args, **kwargs) -> HttpResponse:
        LOGGER.info(f"Request: {request}")
        self.uri = request.META['REQUEST_URI'] if 'REQUEST_URI' in request.META else ''
        self.proto = request.scheme
        self.uri = re.sub(r'/?[^/]+$', '', self.uri)
        self.key = request.GET.get("key", "")
        self.key_type = request.GET.get("key_type", "")
        self.data = {}
        self.uid = request.GET.get("uid", "")
        self.suid = request.GET.get("suid", "")
        self.login = request.GET.get("login", "")
        if self.uid:
            self.key_type, self.key = 'uid', self.uid
        elif self.suid:
            self.key_type, self.key = 'suid', self.suid
        elif self.login:
            self.key_type, self.key = 'login', self.login
        if self.key:
            cbAbuse = request.GET.getlist('cb_abuse')
            cbSource = request.GET.getlist('cb_source')
            abFilter = {}
            cbAbuseNeg = set(range(len(fooSrc))) - set(map(int, cbAbuse))
            cbSourceNeg = set(cmplSrc.keys()) - set(cbSource)
            if self.key_type == "uid" or self.key_type == "suid" or self.key_type == "login":
                self.uid, self.suid, self.login = self.getUserInfo(self.key_type, self.key)
            if len(cbAbuse) != 0 and len(cbSource) != 0:
                if self.uid:
                    abFilter["uid"] = "uid={0}".format(self.uid)
                elif self.suid:
                    abFilter["suid"] = "suid={0}".format(self.suid)
                if len(cbAbuse) < len(cbAbuseNeg):
                    abFilter['foo'] = 'foo IN (%s)' % ','.join(map(str, cbAbuse))
                elif len(cbAbuseNeg) > 0:
                    abFilter['foo'] = 'foo NOT IN (%s)' % ','.join(map(str, cbAbuseNeg))
                if len(cbSource) < len(cbSourceNeg):
                    abFilter['source'] = ' AND '.join(map(lambda s: "source LIKE '{0}%'".format(s), cbSource))
                elif len(cbSourceNeg) > 0:
                    abFilter['source'] = ' AND '.join(map(lambda s: "source NOT LIKE '{0}%'".format(s), cbSourceNeg))
                pg, PG = {}, settings.DB["complaints"].copy()
                getPGCredentials(PG)
                try:
                    pg = getPGdb(PG)
                    pg_cursor = pg.cursor()
                    pg_cursor.execute(
                        "SELECT id, data, cmpldata, source, foo, move, stid, cnt FROM abuses WHERE %s " %
                        'ORDER BY cmpldata DESC AND '.join(abFilter.values()))
                    while True:
                        n = 0
                        for row in pg_cursor.fetchmany(CHUNK_SIZE):
                            n += 1
                            self.data[row[0]] = {
                                'data':     datetime.fromtimestamp(row[1]).strftime("%Y-%m-%d %H:%M:%S"),
                                'cmpldata': datetime.fromtimestamp(row[2]).strftime("%Y-%m-%d %H:%M:%S"),
                                'source':   row[3],
                                'foo':      row[4],
                                'move':     row[5],
                                'stid':     row[6],
                                'cnt':      row[7]
                            }
                        if n < 1:
                            break
                    pg_cursor.close()
                except Exception as e:
                    LOGGER.error(f"DB exception: {e}")
        return super(UserAbusesView, self).dispatch(request, *args, **kwargs)

    def get_context_data(self, **kwargs):
        context = super(UserAbusesView, self).get_context_data(**kwargs)
        context["uri"] = self.uri
        context["foosrc"] = fooSrc
        context["cmplsrc"] = cmplSrc
        context["uid"] = self.uid
        context["suid"] = self.suid
        context["login"] = self.login
        context["data"] = sorted(self.data.values(), key=cmp_to_key(lambda a, b: a['data'] - b['data']))
        return context
