#pragma once

#include <boost/function.hpp>
#include <boost/asio.hpp>
#include <boost/noncopyable.hpp>
#include <boost/shared_ptr.hpp>
#include <boost/make_shared.hpp>
#include <boost/enable_shared_from_this.hpp>

#include <memory>

#include <nwsmtp/avir_options.h>
#include <nwsmtp/buffers.h>

namespace NNwSmtp
{

class avir_client
        : public std::enable_shared_from_this<avir_client>
        , private boost::noncopyable
{
  public:

    using socket = boost::asio::ip::tcp::socket;
    using buffers_ptr = std::shared_ptr<writable_buffers<socket> >;

    enum status
    {
        unknown,
        clean,
        infected
    };

    avir_client(boost::asio::io_service& ios,
            const avir_client_options& opt);

    virtual ~avir_client() = default;

    typedef boost::function<void(boost::system::error_code, status)> handler_t;
    typedef boost::function<void(const char*)> logger_t;

    template <typename ConstBuffers>
    void start(handler_t handler, logger_t logger, const ConstBuffers& msg, std::size_t size);

    virtual void check(handler_t h, logger_t l, buffers_ptr msg, std::size_t size);

    void stop();

    static std::string status_to_str(status stat);

  private:
    boost::asio::io_service& ios_;
    const avir_client_options opt_;
};

typedef std::shared_ptr<avir_client> avir_client_ptr;

template <typename ConstBuffers>
inline void avir_client::start(handler_t h, logger_t l, const ConstBuffers& msg, std::size_t size)
{
    auto buf = make_writable_buffers_ptr<socket>(msg);
    check(std::move(h), std::move(l), std::move(buf), size);
}

}   // namespace NNwSmtp
