#pragma once

#include <boost/asio.hpp>
#include <boost/function.hpp>


// Simple container reference wrapper modeling ConstBufferSequence and
// MutableBufferSequence concepts.
template <typename T>
class buffers_reference_wrapper
{
public:
    typedef typename T::const_iterator const_iterator;
    explicit buffers_reference_wrapper(T& x) : x_(boost::addressof(x)) {}
    const_iterator begin() const { return x_->begin(); }
    const_iterator end() const { return x_->end(); }
private:
    T* x_;
};


#if BOOST_VERSION >= 106600
namespace boost {
namespace asio {

template <typename T>
struct is_const_buffer_sequence<buffers_reference_wrapper<T>> : std::true_type{};

}  // namespace asio
}  // namespace boost
#endif


template <typename T>
inline buffers_reference_wrapper<const T> make_buffers_cref(const T& x)
{
    return buffers_reference_wrapper<const T>(x);
}

template <typename Stream>
class writable_buffers
{
public:
    typedef boost::function<void (boost::system::error_code, std::size_t)> handler_t;
    typedef Stream stream_t;

    virtual void async_write(stream_t&, handler_t&&) = 0;
    virtual void async_write(stream_t&, handler_t&&, std::size_t) = 0;

    virtual ~writable_buffers() {}
};

template <typename Stream, typename Buffers>
class writable_buffers_impl : public writable_buffers<Stream>
{
public:
    using handler_t = typename writable_buffers<Stream>::handler_t;
    using stream_t = typename writable_buffers<Stream>::stream_t;

    explicit writable_buffers_impl(const Buffers& buffers) : buffers_(buffers) {}

    void async_write(stream_t& s, handler_t&& h) override
    {
        boost::asio::async_write(s, make_buffers_cref(buffers_), std::move(h));
    }

    void async_write(stream_t& s, handler_t&& h, std::size_t min_size) override
    {
        boost::asio::async_write(s, make_buffers_cref(buffers_),
                boost::asio::transfer_at_least(min_size), std::move(h));
    }
private:
    Buffers buffers_;
};

template <typename Stream, typename Buffers>
inline std::shared_ptr<writable_buffers<Stream> > make_writable_buffers_ptr(const Buffers& b)
{
    return std::make_shared<writable_buffers_impl<Stream, Buffers> >(b);
}


