#include <iostream>
#include <iomanip>
#include <boost/asio.hpp>
#include <boost/bind.hpp>
#include <boost/format.hpp>
#include <boost/date_time/posix_time/posix_time.hpp>
#include <boost/algorithm/string/predicate.hpp>
#include <boost/shared_array.hpp>

#include <yplatform/time_traits.h>

#include <nwsmtp/avir_client.h>
#include <nwsmtp/hosts_resolver.h>
#include <nwsmtp/detail/request.hpp>

#ifdef NWSMTP_ENABLE_PA
#include <pa/async.h>
#endif

#include <boost/asio/yield.hpp>

/* -- copy from http://www.corpit.ru/pipermail/avcheck/2004q2/000974.html */
#define DRWEBD_SCAN_CMD          0x0001
#define DERR_IS_CLEAN            (1<<20) /*= 0x00100000 */
#define DERR_KNOWN_VIRUS         0x0020
#define DERR_UNKNOWN_VIRUS       0x0040
#define DRWEBD_IS_MAIL           (1<<19)

using boost::asio::ip::tcp;

namespace NNwSmtp
{

using clock = yplatform::time_traits::clock;
using time_point = yplatform::time_traits::time_point;

struct avir_client_impl
        : public std::enable_shared_from_this<avir_client_impl>
{
    boost::asio::io_service& ios_;
    const avir_client_options& opt_;
    resolver_t resolver_;
    boost::asio::deadline_timer timer_;
    std::shared_ptr<avir_client::socket> socket_;
    boost::asio::io_service::strand strand_;
    avir_client::handler_t handler_;
    avir_client::logger_t logger_;
    avir_client::buffers_ptr msg_;
    std::size_t msg_size_;

    avir_client::handler_t ohandler_;
    int i_;
    time_point startedAt;
#ifdef NWSMTP_ENABLE_PA
    pa::stimer_t pa_;
#endif
    bool connected_; // for logging
    detail::request_t req_;

    avir_client_impl(
        boost::asio::io_service& ios,
        const avir_client_options& opt,
        avir_client::handler_t&& handler,
        avir_client::logger_t&& logger,
        avir_client::buffers_ptr&& msg,
        std::size_t size)
            : ios_(ios)
            , opt_(opt)
            , resolver_(ios)
            , timer_(ios)
            , socket_()
            , strand_(ios)
            , handler_()
            , logger_(logger)
            , msg_(msg)
            , msg_size_(size)
            , ohandler_(handler)
            , i_(0)
#ifdef NWSMTP_ENABLE_PA
            , pa_()
#endif
            , connected_(false)
            , req_()
    {
    }

    ~avir_client_impl()
    {
    }

    enum
    {
        buf_size = 64
    };
    char buf_[buf_size];

    boost::asio::streambuf request_buf_;
    boost::asio::streambuf response_buf_;
    boost::shared_array<char> discard_buf_;
};

typedef std::shared_ptr<avir_client_impl> impl_ptr;

namespace {

struct handle_stop
{
    impl_ptr impl_;

    explicit handle_stop(const impl_ptr& rh)
            : impl_(rh)
    {
    }

    void operator()() const
    {
        try
        {
            detail::end_request(impl_->req_);

            impl_->resolver_.cancel();
            impl_->timer_.cancel();
            if (impl_->socket_)
            {
                impl_->socket_->shutdown(avir_client::socket::shutdown_both);
                impl_->socket_->close();
            }
        }
        catch (...)
        {
        }
    }
};

struct handle_done
{
    impl_ptr impl_;

    explicit handle_done(const impl_ptr& rh)
            : impl_(rh)
    {
    }

    void operator()(const boost::system::error_code& ec,
            avir_client::status status) const
    {
        if (ec == boost::asio::error::operation_aborted)
            return;

        handle_stop stop(impl_);
        stop();

        avir_client::handler_t h;
        impl_->handler_.swap(h);

        if (h)
            h(ec, status);
    }
};

struct handle_timeout
{
    impl_ptr impl_;
    detail::request_monitor_t req_;

    handle_timeout(const impl_ptr& impl, const detail::request_t& req, int seconds)
            : impl_(impl)
            , req_(req)
    {
        impl_->timer_.expires_from_now(boost::posix_time::seconds(seconds));
        impl_->timer_.async_wait(impl_->strand_.wrap(*this));
    }

    void operator()(const boost::system::error_code& ec) const
    {
        if (ec == boost::asio::error::operation_aborted
                || detail::request_expired(req_))
            return;

        handle_done hd(impl_);
        hd(make_error_code(boost::asio::error::timed_out), avir_client::unknown);
    }
};

struct handle_io : public boost::asio::coroutine
{
    impl_ptr impl_;
    detail::request_monitor_t req_;
    std::shared_ptr<avir_client::socket> socket_;

    explicit handle_io(const impl_ptr& impl, const detail::request_t& req)
            : impl_(impl)
            , req_(req)
            , socket_(impl->socket_)
    {
        char* pd = impl_->buf_;

        uint32_t i = htonl(DRWEBD_SCAN_CMD);
        memcpy(pd, &i, sizeof i);
        pd += sizeof i;

        i = htonl(DRWEBD_IS_MAIL);
        memcpy(pd, &i, sizeof i);
        pd += sizeof i;

        i = 0;
        memcpy(pd, &i, sizeof i);
        pd += sizeof i;

        i = htonl(impl_->msg_size_);
        memcpy(pd, &i, sizeof i);
        pd += sizeof i;

        boost::asio::async_write(*socket_,
                boost::asio::buffer(impl_->buf_, pd-impl_->buf_),
                impl_->strand_.wrap(*this));
    }

    void operator()(const boost::system::error_code& ec, std::size_t /*sz*/ = 0)
    {
        if (ec == boost::asio::error::operation_aborted
                || detail::request_expired(req_))
            return;

        int status = 0;

        if (ec)
        {
            handle_done h(impl_);
            return h(ec, avir_client::unknown);
        }
        else reenter(this)
        {
            yield return impl_->msg_->async_write(*socket_,
                    impl_->strand_.wrap(*this));

            yield return
                    boost::asio::async_read(*socket_,
                            boost::asio::buffer(impl_->buf_),
                            boost::asio::transfer_at_least(4),
                            impl_->strand_.wrap(*this));
            memcpy(&status, impl_->buf_, sizeof status);
            status = ntohl(status);

            yield return
                    handle_done(impl_)(ec, (status & DERR_IS_CLEAN)
                            ? avir_client::clean
                            : (status & (DERR_KNOWN_VIRUS | DERR_UNKNOWN_VIRUS)
                                    ? avir_client::infected
                                    : avir_client::unknown));
        }
    }
};

enum avir_icap_errors
{
    malformed_status_line = 1,
    bad_status_code
};

class avir_icap_category : public boost::system::error_category
{
public:
    const char* name() const BOOST_NOEXCEPT override
    {
        return "nwsmtp.avir.icap";
    }

    std::string message(int e) const override
    {
        switch (e)
        {
        case malformed_status_line:
            return "Malformed status line";
        case bad_status_code:
            return "Bad status code";
        default:
            return "Unknown ICAP error";
        }
    }
};

const boost::system::error_category& get_avir_icap_category()
{
    static avir_icap_category instance;
    return instance;
}

inline boost::system::error_code make_error_code(avir_icap_errors e)
{
    return boost::system::error_code(static_cast<int>(e),
        get_avir_icap_category());
}

struct handle_icap_io : public boost::asio::coroutine
{
    impl_ptr impl_;
    detail::request_monitor_t req_;
    std::shared_ptr<avir_client::socket> socket_;
    avir_client::status status_;
    bool reading_rest_;

    handle_icap_io(const impl_ptr& impl, const detail::request_t& req) :
        impl_(impl),
        req_(req),
        socket_(impl->socket_),
        status_(avir_client::unknown),
        reading_rest_(false)
    {}

    void operator ()(boost::system::error_code ec, std::size_t sz = 0)
    {
        Y_UNUSED(sz);

        if (ec == boost::asio::error::operation_aborted
                || detail::request_expired(req_))
            return;

        if (ec && (!reading_rest_ || ec != boost::asio::error::eof))
        {
            handle_done h(impl_);
            return h(ec, avir_client::unknown);
        }
        else
        {
            reenter(this)
            {
                // Send ICAP request and headers.
                prepare_request();
                yield return boost::asio::async_write(*socket_,
                    impl_->request_buf_, impl_->strand_.wrap(*this));

                // Send actual message.
                yield return impl_->msg_->async_write(*socket_,
                    impl_->strand_.wrap(*this));

                // Send chunked transfer encoding terminator.
                yield return boost::asio::async_write(*socket_,
                    boost::asio::buffer("\r\n0\r\n\r\n"),
                    impl_->strand_.wrap(*this));

                // Read response status line.
                yield return boost::asio::async_read_until(*socket_,
                    impl_->response_buf_, "\r\n",
                    impl_->strand_.wrap(*this));

                parse_response_status(ec);
                if (ec)
                    return handle_done(impl_)(ec, avir_client::unknown);

                // Read response headers.
                yield return boost::asio::async_read_until(*socket_,
                    impl_->response_buf_, "\r\n\r\n",
                    impl_->strand_.wrap(*this));

                status_ = parse_response_headers();
                if (status_ == avir_client::unknown)
                {
                    // Read HTTP response status.
                    yield return boost::asio::async_read_until(*socket_,
                        impl_->response_buf_, "\r\n",
                        impl_->strand_.wrap(*this));
                    status_ = parse_http_response_status();
                }

                // Read the remaining response.
                reading_rest_ = true;
                impl_->discard_buf_.reset(new char[1024]);
                while (ec != boost::asio::error::eof)
                {
                    yield return boost::asio::async_read(*socket_,
                        boost::asio::buffer(impl_->discard_buf_.get(), 1024),
                        impl_->strand_.wrap(*this));
                }
                if (ec == boost::asio::error::eof)
                    ec = boost::system::error_code();

                return handle_done(impl_)(ec, status_);
            }
        }
    }

    void prepare_request()
    {
        const remote_point& host = (impl_->i_ < impl_->opt_.retries)
            ? impl_->opt_.primary : impl_->opt_.secondary;
        impl_->request_buf_.consume(impl_->request_buf_.size());
        impl_->response_buf_.consume(impl_->response_buf_.size());
        impl_->discard_buf_.reset();
        std::ostream os(&impl_->request_buf_);
        os << "RESPMOD icap://" << host.host_name_ << ":" << host.port_
            << "/" << impl_->opt_.icap_service << " ICAP/1.0\r\n";
        os << "Host: " << host.host_name_ << "\r\n";
        os << "Allow: 204\r\n";
        os << "Encapsulated: res-body=0\r\n";
        os << "Connection: close\r\n";
        os << "\r\n";
        os << std::hex << impl_->msg_size_ << "\r\n";
    }

    void parse_response_status(boost::system::error_code& ec)
    {
        std::istream is(&impl_->response_buf_);
        std::string version;
        is >> version;
        unsigned int code = 0;
        is >> code;
        std::string message;
        std::getline(is, message);
        if (!is || version.compare(0, 5, "ICAP/") != 0)
            ec = make_error_code(malformed_status_line);
        else if (code != 204 && code != 200)
            ec = make_error_code(bad_status_code);
    }

    avir_client::status parse_response_headers()
    {
        std::istream is(&impl_->response_buf_);
        std::string header;
        bool res_hdr = false;
        while (std::getline(is, header) && header != "\r")
        {
            if (boost::istarts_with(header, "X-Infection-Found")
                    || boost::istarts_with(header, "X-Violations-Found")
                    || boost::istarts_with(header, "X-Virus-ID"))
                return avir_client::infected;

            if (boost::istarts_with(header, "Encapsulated"))
                res_hdr = (header.find("res-hdr=", 12) != std::string::npos);
        }

        // If we have res-hdr present, we have to inspect it to determine
        // the result.
        if (res_hdr)
            return avir_client::unknown;
        return avir_client::clean;
    }

    avir_client::status parse_http_response_status()
    {
        std::istream is(&impl_->response_buf_);
        std::string version;
        is >> version;
        unsigned int code = 0;
        is >> code;
        std::string message;
        std::getline(is, message);
        if (!is || version.compare(0, 5, "HTTP/") != 0)
            return avir_client::clean;
        return (code == 403) ? avir_client::infected : avir_client::clean;
    }
};

struct handle_connect
{
    impl_ptr impl_;
    detail::request_monitor_t req_;

    handle_connect(const impl_ptr& impl, const detail::request_t& req,
            const tcp::endpoint& endpoint)
            : impl_(impl)
            , req_(req)
    {
        boost::system::error_code ec;
        impl_->socket_.reset(new avir_client::socket(impl_->timer_.get_io_service()));
        impl_->socket_->async_connect(endpoint, impl_->strand_.wrap(*this));
    }

    void operator()(const boost::system::error_code& ec, std::size_t /*sz*/ = 0)
    {
        if (ec == boost::asio::error::operation_aborted
                || detail::request_expired(req_))
            return;

        if (!ec)
        {
            impl_->connected_ = true;
            detail::start_request(impl_->req_);
            handle_timeout ht(impl_, impl_->req_, impl_->opt_.timeout);
            if (impl_->opt_.protocol == avir_client_options::drweb)
            {
                handle_io h(impl_, impl_->req_);
            }
            else
            {
                handle_icap_io h(impl_, impl_->req_);
                h(boost::system::error_code());
            }
        }
        else
        {
            handle_done h(impl_);
            h(ec, avir_client::unknown);
        }
    }
};

const char* get_status_descr(avir_client::status status)
{
    switch (status)
    {
        case avir_client::infected:
            return "infected";
        case avir_client::clean:
            return "clean";
        case avir_client::unknown:
        default:
            return "unknown";
    }
}

struct handle_multiple_retries : private boost::asio::coroutine
{
    impl_ptr impl_;

    explicit handle_multiple_retries(const impl_ptr& impl)
            : impl_(impl)
    {
        impl_->handler_ = *this;
        impl_->i_ = 0;
    }

    const remote_point& get_host()
    {
        return (impl_->i_ < impl_->opt_.retries)
                ? impl_->opt_.primary
                : impl_->opt_.secondary;
    }

    void operator()(
        const boost::system::error_code& ec = boost::system::error_code(),
        avir_client::status status = avir_client::unknown)
    {
        if (ec == boost::asio::error::operation_aborted)
            return;

        reenter(this)
        for (impl_->i_=0;
             impl_->i_< 2*impl_->opt_.retries;
             ++impl_->i_)
        {
            yield
            {
                impl_->handler_ = *this;
                impl_->startedAt = clock::now();
                impl_->connected_ = false;
                detail::start_request(impl_->req_);
                handle_timeout ht(impl_, impl_->req_,
                        impl_->opt_.connect_timeout);

                const auto& host = get_host();
                boost::system::error_code addr_ec;
                auto addr = boost::asio::ip::make_address(host.host_name_, addr_ec);

                if (!addr_ec) {
                    handle_connect h(impl_, impl_->req_, tcp::endpoint(addr, host.port_));
                } else {
                    THostsResolver::AsyncResolve(
                        host.host_name_,
                        impl_->resolver_,
                        impl_->strand_,
                        [impl = impl_, port = host.port_](std::string address) {
                            if (address.empty()) {
                                auto ec = boost::system::errc::make_error_code(boost::system::errc::host_unreachable);
                                handle_done h(impl);
                                h(ec, avir_client::unknown);
                            } else if (!detail::request_expired(impl->req_)) {
                                handle_connect h(impl, impl->req_, tcp::endpoint(boost::asio::ip::make_address(address), port));
                            }
                            // Если request_expired, то значит уже был вызван handle_stop, вместе с которым вызывается iml_->handler
                            // и поток выполнения возваращается в handle_multiple_retries
                            // где вызывается пользовательский callback impl_->ohandler
                        }
                    );
                }

                return;
            }

            if (impl_->logger_)
                impl_->logger_(boost::str(
                    boost::format("ravatt connect=%1%, check=%2%, "
                            "host='%3%:%4%', delay=%5%, size=%6%, "
                            "status='%7%', msg='%8%', try=%9%")
                    % (impl_->connected_ ? "ok" : "error")
                    % (!ec ? "ok" : "error")
                    % get_host().host_name_
                    % get_host().port_
                    % yplatform::time_traits::to_string(clock::now() - impl_->startedAt)
                    % impl_->msg_size_
                    % get_status_descr(status)
                    % ec.message()
                    % (impl_->i_+1)).c_str());

            if (!ec)
                break;
        }

        impl_->ios_.post(std::bind(impl_->ohandler_, ec, status));
    }
};
}

void avir_client::check(handler_t h, logger_t l, buffers_ptr msg, std::size_t size)
{
    auto impl = std::make_shared<avir_client_impl>(
            ios_,
            opt_,
            std::move(h),
            std::move(l),
            std::move(msg),
            size);

    handle_multiple_retries mh(impl);
    ios_.post(mh);
}

void avir_client::stop()
{

}

std::string avir_client::status_to_str(status stat)
{
    if (stat == infected)
        return "infected";
    else if (stat == clean)
        return "clean";
    return "unknown";
}

avir_client::avir_client(boost::asio::io_service& ios,
        const avir_client_options& opt)
        : ios_(ios)
        , opt_(opt)
{
}

}   // namespace NNwSmtp
