#include <nwsmtp/hosts_resolver.h>

#include <yplatform/coroutine.h>
#include <yplatform/yield.h>

#include <functional>
#include <variant>

class THostResolver {
public:
    using TResolver = THostsResolver::TResolver;
    using TStrand = THostsResolver::TStrand;
    using TCallback = THostsResolver::TCallback;
    using TIterVariant = std::variant<TResolver::iterator_aaaa, TResolver::iterator_a>;
    using TYieldCtx = yplatform::yield_context<THostResolver>;

    THostResolver(std::string hostname, TResolver& resolver, TStrand strand, TCallback callback);

    void operator()(
        TYieldCtx yieldCtx,
        const boost::system::error_code& ec = boost::system::error_code(),
        TIterVariant iter = TIterVariant()
    );

private:
    std::string Hostname;
    TResolver& Resolver;
    TStrand Strand;
    TCallback Callback;
};

THostResolver::THostResolver(
    std::string hostname,
    TResolver& resolver,
    TStrand strand,
    TCallback callback
)
    : Hostname(std::move(hostname))
    , Resolver(resolver)
    , Strand(std::move(strand))
    , Callback(std::move(callback))
{}

void THostResolver::operator()(
    TYieldCtx yieldCtx,
    const boost::system::error_code& ec,
    TIterVariant iter)
{
    if (ec == boost::asio::error::operation_aborted) {
        return;
    }

    std::string ipAddress;

    reenter(yieldCtx) {
        yield Resolver.async_resolve_aaaa(Hostname, Strand.wrap(yieldCtx));

        if (!ec) {
            auto& ipv6Iter = std::get<TResolver::iterator_aaaa>(iter);
            if (ipv6Iter != TResolver::iterator_aaaa()) {
                ipAddress = *ipv6Iter;
            }
        }

        if (!ipAddress.empty()) {
            yield break;
        }

        yield Resolver.async_resolve_a(Hostname, Strand.wrap(yieldCtx));

        if (!ec) {
            auto& ipv4Iter = std::get<TResolver::iterator_a>(iter);
            if (ipv4Iter != TResolver::iterator_a()) {
                ipAddress = *ipv4Iter;
            }
        }
    }

    if (yieldCtx.is_complete()) {
        if (ipAddress.empty()) {
            YLOG_GLOBAL(error) << "Fail to resolve host: " << Hostname;
        }
        Callback(std::move(ipAddress));
    }
}

THostsResolver::THostsResolver(const resolver_options& resolverOptions)
    : IoService()
    , Strand(IoService)
    , Resolver(IoService, resolverOptions)
{}

void THostsResolver::AsyncResolve(
    std::string hostname,
    TResolver& resolver,
    TStrand strand,
    TCallback callback)
{
    auto hostResolver = std::make_shared<THostResolver>(
        std::move(hostname),
        resolver,
        std::move(strand),
        std::move(callback));

    yplatform::spawn(hostResolver);
}
