#include <nwsmtp/remote_point.h>

#include <boost/algorithm/string/predicate.hpp>
#include <boost/lambda/lambda.hpp>
#include <boost/spirit/include/qi.hpp>
#include <boost/spirit/include/qi_omit.hpp>
#include <boost/spirit/include/qi_raw.hpp>

std::string remote_point::str() const {
    std::ostringstream url;
    if (!proto_.empty()) {
        url << proto_ << "://";
    }
    url << host_name_ << ":" << port_ << url_;
    return url.str();
}

remote_point remote_point::from_string(const std::string& url)
{
    remote_point rp;
    if (!parse_remotepoint(url, rp))
        throw std::runtime_error("failed to parse url: " + url);

    if (rp.port_ == 0)
    {
        if (rp.proto_ == "https")
            rp.port_ = 443;
        else if (rp.proto_ == "http")
            rp.port_ = 80;
    }

    if (rp.port_ == 0)
        throw std::runtime_error(
            "missing port number or service name: " + url);

    return rp;
}

bool parse_remotepoint(const std::string_view in, remote_point& out)
{
    namespace qi = boost::spirit::qi;
    namespace ascii = boost::spirit::ascii;

    using boost::lambda::var;
    using boost::lambda::_1;
    using qi::attr;
    using qi::char_;
    using qi::int_;
    using qi::raw;
    using qi::lit;
    using qi::omit;
    using qi::alpha;
    using ascii::space;
    using ascii::space_type;

    remote_point rp;

    if (!qi::parse(
                in.begin()
              , in.end()
              , (+alpha >> omit[lit("://")] | attr(""))
                >> (omit[char_('[')] >> +(~char_(']')) >> omit[char_(']')]
                        | +(~char_(":/")))
                >> (omit[char_(':')] >> int_
                        | omit[char_('/')] >> attr(0) | attr(0))
                >> (+char_ | attr(""))
              , rp.proto_
              , rp.host_name_
              , rp.port_
              , rp.url_))
        return false;

    if (!boost::starts_with(rp.url_, "/")) {
        rp.url_ = std::string("/").append(rp.url_);
    }

    out = rp;
    return true;
}
