#pragma once

#include <string>
#include <boost/asio/spawn.hpp>
#include <yamail/expected.h>
#include <mail/ymod_queuedb/include/types.h>
#include <mail/webmail/corgi/include/types.h>


namespace spaniel {

using namespace corgi;

BOOST_STRONG_TYPEDEF(std::int64_t, SearchId)
BOOST_STRONG_TYPEDEF(std::int64_t, Id)


inline Id makeId(const std::string& val) {
    return Id(std::stoll(val));
}

struct SearchResult {
    Uid uid;
    Id id;
    std::time_t received_date;

    bool operator==(const SearchResult&) const = default;
};
using SearchResults = std::vector<SearchResult>;

using QueryJson = std::string;
using Notice = std::string;
using SearchName = std::string;

enum class SearchState {
    in_progress,
    partially_complete,
    archived,
    error,
    complete
};

struct Query {
    std::optional<bool> hasAttachments;
    std::optional<std::vector<std::string>> text;
    std::optional<std::vector<std::string>> subject;
    std::optional<std::vector<std::string>> from;
    std::optional<std::vector<std::string>> to;
    std::optional<std::vector<std::string>> cc;
    std::optional<std::vector<std::string>> bcc;
    std::optional<std::vector<std::string>> toCcBcc;
    std::optional<std::vector<std::string>> scope;
    std::optional<std::string> request;
    ResolveUsers resolveUsers;
};

struct Search {
    OrgId orgId;
    SearchId searchId;
    Uid adminUid;
    Uids requestedUids;
    std::optional<Uids> foundUids;
    Query query;
    SearchState state;
    SearchName name;
    std::time_t dateFrom;
    std::time_t dateTo;
    std::time_t created;
    std::time_t updated;
    std::optional<std::time_t> minSearchDate;
};

struct MessagesAccessParams {
    Uid uid;
    std::vector<std::string> mids;
    SearchId searchId;
};

struct SingleMessageAccessParams {
    Uid uid;
    std::string mid;
    SearchId searchId;
};

struct SwitchOrganizationParams {
    ymod_queuedb::TaskId taskId;
};

struct SearchCreateParams {
    Query queryParams;
    std::optional<SearchName> name;
    std::time_t dateFrom;
    std::time_t dateTo;
};

struct SearchShowParams {
    SearchId searchId;
};

struct SearchArchiveParams {
    SearchId searchId;
};

struct SearchRenameParams {
    SearchId searchId;
    std::string name;
};

struct SearchListParams {
    PageParams page;
};

struct AsyncSearchParams {
    SearchId searchId;
    std::time_t dateFrom;
    std::time_t dateTo;
};

struct OrganizationUpdateParams {
    Uids all;
    Uids newbie;

    auto operator<=>(const OrganizationUpdateParams&) const = default;
};

struct MailSearchCommonParams {
    std::time_t dateFrom;
    std::time_t dateTo;
    unsigned length;
};

struct MessagesBySearchParams {
    SearchId searchId;
    PageParams page;

    yamail::expected<void> validate(const boost::optional<Search>& search) const;
};

struct MessagesBySearchAndUidParams : public MessagesBySearchParams {
    Uid uid;

    yamail::expected<void> validate(const boost::optional<Search>& search) const;
};

struct SendShareParams {
    std::string to;
};

enum class ActionHistoryType {
    create_search,
    send_share
};

struct ActionHistoryInfo {
    struct SendShare {
        std::string to;
        Uid uid;
        std::string mid;
        SearchId searchId;
    };

    struct CreateSearch {
        SearchId searchId;
    };

    std::optional<SendShare> sendShare;
    std::optional<CreateSearch> createSearch;
};

struct ActionHistoryItem {
    Id actionId;
    Uid adminUid;
    ActionHistoryType type;
    ActionHistoryInfo info;
    std::time_t date;
};

struct ActionHistoryParams {
    PageParams page;
};

}
