import os

import yaml

from library.python import resource
from mail.devpack.lib import helpers
from mail.devpack.lib.components.base import DogComponent
from mail.pg.queuedb.devpack.components.queuedb import QueueDb
from mail.pg.spanieldb.devpack.components.spanieldb import SpanielDb
from mail.webmail_config.lib.make_config import make_config


class SpanielBase(DogComponent):
    def __init__(self, env, components):
        super(SpanielBase, self).__init__(env, components, binary_name='spaniel', custom_path='spaniel')

    def _generate_config(self, env):
        base = resource.find('spaniel/base.yml')
        service_yaml = yaml.safe_load(resource.find('spaniel/service.yaml'))
        common_yaml = yaml.safe_load(resource.find('webmail_config/common.yaml'))
        queuedb_yaml = yaml.safe_load(resource.find('ymod_queuedb/service.yaml'))
        spanieldb_yaml = yaml.safe_load(resource.find('ymod_db/service.yaml'))
        cfg = make_config(env, base, service_yaml, common_yaml, queuedb_yaml, spanieldb_yaml, silent=True)
        config = self.yhttp.format_config(cfg)
        helpers.write2file(config, os.path.join(self.etc_path, 'spaniel.yml'))

    def replace_config(self):
        pass

    def init_root(self):
        self.yhttp.init_root()

        helpers.write2file(resource.find('spaniel/tvm_secret'), os.path.join(self.etc_path, 'tvm_secret'))

        etc_queuedb_path = os.path.join(self.get_root(), 'etc', 'queuedb')
        helpers.mkdir_recursive(etc_queuedb_path)
        helpers.write2file(resource.find('ymod_queuedb/queuedb.conf'), os.path.join(etc_queuedb_path, 'queuedb.conf'))

        etc_spanieldb_path = os.path.join(self.get_root(), 'etc', 'spanieldb')
        helpers.mkdir_recursive(etc_spanieldb_path)
        helpers.write2file(resource.find('ymod_db/spanieldb.conf'), os.path.join(etc_spanieldb_path, 'spanieldb.conf'))

        self.replace_config()

        self.yhttp.init_pgpass(self.get_root())

        self._generate_config('development')

    def start(self):
        self.yhttp.put_pgpassfile_in_env(self.get_root())
        self.yhttp.start('pong')


class SpanielLocalQueuedb(SpanielBase):
    NAME = 'spaniel_local_queuedb'
    DEPS = [QueueDb, SpanielDb]

    def __init__(self, env, components):
        super(SpanielLocalQueuedb, self).__init__(env, components)

    def replace_config(self):
        replace = {
            'queuedb_connection_string': f'host=localhost port={self.components[QueueDb].port()} user=spaniel dbname=queuedb',
            'spanieldb_connection_string': f'host=localhost port={self.components[SpanielDb].port()} user=spaniel dbname=spanieldb',
        }
        devpack = self.yhttp.format_config(resource.find('spaniel/local_queuedb.yml'), **replace)
        helpers.write2file(devpack, os.path.join(self.etc_path, 'spaniel-devpack.yml'))
