#include <ymod_webserver/server.h>
#include <ymod_webserver/response.h>
#include <ymod_webserver/request.h>

#include <yplatform/find.h>
#include <yplatform/module_registration.h>
#include <yplatform/loader.h>

#include <tvm_guard/module.h>

#include <mail/spaniel/core/include/log.h>
#include <mail/spaniel/core/include/parse.h>
#include <mail/spaniel/core/include/types_error.h>

#include <mail/spaniel/service/include/handlers/action_history.h>
#include <mail/spaniel/service/include/handlers/attach_sid.h>
#include <mail/spaniel/service/include/handlers/filter_search.h>
#include <mail/spaniel/service/include/handlers/search.h>
#include <mail/spaniel/service/include/handlers/messages_by_search.h>
#include <mail/spaniel/service/include/handlers/organization.h>
#include <mail/spaniel/service/include/handlers/send_share.h>
#include <mail/spaniel/service/include/handlers/message.h>
#include <mail/spaniel/service/include/handlers/user.h>
#include <mail/spaniel/service/include/request_context.h>

#include <mail/spaniel/http_api/include/parse.h>
#include <mail/spaniel/http_api/include/response.h>


#include <mail/webmail/http_api_helpers/include/context.h>
#include <mail/webmail/http_api_helpers/include/ping.h>
#include <mail/webmail/http_api_helpers/include/handler_helpers.h>
#include <mail/webmail/corgi/include/http_api_handlers.h>

#include <mail/http_getter/client/include/endpoint_reflection.h>
#include <mail/http_getter/client/include/module.h>

#include <yamail/data/deserialization/ptree_reader.h>

#include <boost/range/algorithm/transform.hpp>

#include <mail/ymod_queuedb/include/queue.h>

#include <pa/async.h>


namespace spaniel {

using http_getter::TypedEndpoint;
namespace ydd = yamail::data::deserialization;

template<class F>
auto handler(ConfigPtr config, F f) {
    return [=] (ymod_webserver::response_ptr s, boost::asio::yield_context yield) {
        http_api::Context ctx {s->request()};
        CommonParams common = parseCommonParams(ctx);

        auto uid = std::to_string(common.adminUid.t);
        auto log = http_getter::withLog(config->getter->httpLogger(uid, common.requestId));
        auto httpClient = config->getter->create(*s->request(), log);

        if (!corgi::isAdmin(common, *httpClient, config->resolverConfig, yield).value_or_throw()) {
            throw mail_errors::system_error(make_error(corgi::AccessError::accessDenied, "is not admin"));
        }

        RequestContext req(httpClient, config, common);
        return f(std::move(common), std::move(req), std::move(ctx), std::move(yield));
    };
}

struct HttpApi: public yplatform::module {
    void init(const yplatform::ptree& cfg) {
        using http_api::findDependency;

        http_api::LogArgsAsTskvConfig logArgsCfg {
            .tskvFormat="mail-spaniel-args-log",
            .argsLogger=getTskvArgsLogger(cfg.get<std::string>("logger.args"))
        };
        boost::transform(
            cfg.equal_range("params_to_hide"), std::inserter(logArgsCfg.paramsToHide, logArgsCfg.paramsToHide.end()),
            [] (auto&& val) { return val.second.template get<std::string>("name"); }
        );

        http_api::BindInfo<Response> info {
            .guarded=findDependency<tvm_guard::Module>(cfg, "dependencies.guard"),
            .server=findDependency<ymod_webserver::server>(cfg, "dependencies.server"),
            .attributes=boost::coroutines::attributes(cfg.get<unsigned>("coroutine_stack_size")),
            .reactor=findDependency<yplatform::reactor>(cfg, "dependencies.reactor"),
        };

        auto asyncSearchTimeout = yplatform::time_traits::duration_cast<std::chrono::seconds>(
            cfg.get<yplatform::time_traits::duration>("async_search.timeout_dur")
        );

        auto ogranizationSwitchTimeout = yplatform::time_traits::duration_cast<std::chrono::seconds>(
            cfg.get<yplatform::time_traits::duration>("organization_switch.timeout_dur")
        );

        auto intervalBeforeRemovingOrganization = yplatform::time_traits::duration_cast<std::chrono::seconds>(
            cfg.get<yplatform::time_traits::duration>("interval_before_removing_organization_dur")
        );

        auto config = std::make_shared<Config> (Config {
            .getter=findDependency<http_getter::TypedClientModule>(cfg, "dependencies.http_getter"),
            .resolverConfig=corgi::fromPtree(cfg),
            .hound=ydd::fromPtree<TypedEndpoint>(cfg.get_child("hound")),
            .attach_sid=ydd::fromPtree<TypedEndpoint>(cfg.get_child("attach_sid")),
            .mbody=ydd::fromPtree<TypedEndpoint>(cfg.get_child("mbody")),
            .sendbernar=ydd::fromPtree<TypedEndpoint>(cfg.get_child("sendbernar")),
            .syncSearch=ydd::fromPtree<TypedEndpoint>(cfg.get_child("sync_search")),
            .settings=ydd::fromPtree<TypedEndpoint>(cfg.get_child("settings")),
            .createHiddenTrash=ydd::fromPtree<TypedEndpoint>(cfg.get_child("hidden_trash.create")),
            .purgeHiddenTrash=ydd::fromPtree<TypedEndpoint>(cfg.get_child("hidden_trash.purge")),
            .asyncSearchTaskTimeout=ymod_queuedb::Timeout(asyncSearchTimeout),
            .ogranizationSwitchTaskTimeout=ymod_queuedb::Timeout(ogranizationSwitchTimeout),
            .repo=findDependency<Repository>(cfg, "dependencies.spanieldb"),
            .queuedb=findDependency<ymod_queuedb::Queue>(cfg, "dependencies.queuedb"),
            .intervalBeforeRemovingOrganization=intervalBeforeRemovingOrganization.count(),
        });

        pa::async_profiler::init(500000, 16, cfg.get<std::string>("profiler_log"));

        http_api::bindPingHandler(info);
        organizations(info, logArgsCfg, config);
        user(info, logArgsCfg, config);
        search(info, logArgsCfg, config);
        messagesBySearch(info, logArgsCfg, config);
        proxy(info, logArgsCfg, config);
        sib(info, logArgsCfg, config);

        LOGDOG_(getModuleLogger(), notice, log::message="spaniel::HttpApi loaded");
    }

    void organizations(const http_api::BindInfo<Response>& info, const http_api::LogArgsAsTskvConfig& logArgsCfg, const ConfigPtr& config) const {

        bindPOST<OrganizationEnableResult>(info, logArgsCfg, "/v1/organization/enable",
            corgi::organizationHandler(config->getter, [=] (OrganizationParams common, http_getter::TypedClientPtr, boost::asio::yield_context yield) {
                return organizationEnable(common, config, yield);
            }));

        bindPOST<OrganizationDisableResult>(info, logArgsCfg, "/v1/organization/disable",
            corgi::organizationHandler(config->getter, [=] (OrganizationParams common, http_getter::TypedClientPtr, boost::asio::yield_context yield) {
                return organizationDisable(common, config, yield);
            }));

        bindPOST<OrganizationActivateResult>(info, logArgsCfg, "/v1/organization/activate",
            handler(config, [=] (CommonParams common, RequestContext, http_api::Context, boost::asio::yield_context yield) {
                return organizationActivate(common, config, yield);
            }));

        bindPOST<OrganizationDeactivateResult>(info, logArgsCfg, "/v1/organization/deactivate",
            handler(config, [=] (CommonParams common, RequestContext, http_api::Context, boost::asio::yield_context yield) {
                return organizationDeactivate(common, config, yield);
            }));

        bindGET<OrganizationShowResult>(info, "/v1/organization/show",
            corgi::organizationHandler(config->getter, [=] (OrganizationParams common, http_getter::TypedClientPtr, boost::asio::yield_context yield) {
                return organizationShow(common, config, yield);
            }));

        bindGET<EmptyResult>(info, "/v1/organization/update",
            [=] (ymod_webserver::response_ptr s, boost::asio::yield_context yield) {
                http_api::Context ctx {s->request()};

                return organizationUpdateAll(ctx.optionalHeader("x-request-id").value_or(""), config, std::move(yield));
        });
    }

    void user(const http_api::BindInfo<Response>& info, const http_api::LogArgsAsTskvConfig& logArgsCfg, const ConfigPtr& config) const {

        bindPOST<UserSettingResult>(info, logArgsCfg, "/v1/user/enable",
            corgi::userHandler(config->getter, [=] (SettingParams params, http_getter::TypedClientPtr client, boost::asio::yield_context yield) {
                return userDisableEnable(true, std::move(params), std::move(client), config, std::move(yield));
            }));

        bindPOST<UserSettingResult>(info, logArgsCfg, "/v1/user/disable",
            corgi::userHandler(config->getter, [=] (SettingParams params, http_getter::TypedClientPtr client, boost::asio::yield_context yield) {
                return userDisableEnable(false, std::move(params), std::move(client), config, std::move(yield));
            }));
    }

    void search(const http_api::BindInfo<Response>& info, const http_api::LogArgsAsTskvConfig& logArgsCfg, const ConfigPtr& config) const {

        bindPOST<SearchCreateResult>(info, logArgsCfg, "/v1/search/create",
            handler(config, [=] (CommonParams common, RequestContext req, http_api::Context ctx, boost::asio::yield_context yield) {
                return searchCreate(common, parseSearchCreateParams(ctx), req, config, yield);
            }));

        bindGET<SearchShowResult>(info, "/v1/search/show",
            handler(config, [=] (CommonParams common, RequestContext, http_api::Context ctx, boost::asio::yield_context yield) {
                return searchShow(common, parseSearchShowParams(ctx), config, yield);
            }));

        bindPOST<SearchRenameResult>(info, logArgsCfg, "/v1/search/rename",
            handler(config, [=] (CommonParams common, RequestContext, http_api::Context ctx, boost::asio::yield_context yield) {
                return searchRename(common, parseSearchRenameParams(ctx), config, yield);
            }));

        bindGET<SearchListResult>(info, "/v1/search/list",
            handler(config, [=] (CommonParams common, RequestContext, http_api::Context ctx, boost::asio::yield_context yield) {
                return searchList(common, parseSearchListParams(ctx), config, yield);
            }));

        bindGET<SearchLastIdResult>(info, "/v1/search/last_id",
            handler(config, [=] (CommonParams common, RequestContext, http_api::Context, boost::asio::yield_context yield) {
                return searchLastId(common, config, yield);
            }));

        bindPOST<SearchArchiveResult>(info, logArgsCfg, "/v1/search/archive",
            handler(config, [=] (CommonParams common, RequestContext, http_api::Context ctx, boost::asio::yield_context yield) {
                return searchArchive(common, parseSearchArchiveParams(ctx), config, yield);
            }));
    }

    void messagesBySearch(const http_api::BindInfo<Response>& info, const http_api::LogArgsAsTskvConfig&, const ConfigPtr& config) const {

        bindGET<MessagesBySearchResult>(info, "/v1/messages_by_search_and_uid",
            handler(config, [=] (CommonParams common, RequestContext req, http_api::Context ctx, boost::asio::yield_context yield) {
                return messagesBySearchAndUid(common, parseMessagesBySearchAndUidParams(ctx), req, config, yield);
            }));

        bindGET<MessagesBySearchAsyncResult>(info, "/v1/messages_by_search",
            handler(config, [=] (CommonParams common, RequestContext req, http_api::Context ctx, boost::asio::yield_context yield) {
                return spaniel::messagesBySearch(common, parseMessagesBySearchParams(ctx), req, config, yield);
            }));
    }

    void proxy(const http_api::BindInfo<Response>& info, const http_api::LogArgsAsTskvConfig& logArgsCfg, const ConfigPtr& config) const {

        bindPOST<SendShareResult>(info, logArgsCfg, "/v1/send_share",
            handler(config, [=] (CommonParams common, RequestContext req, http_api::Context ctx, boost::asio::yield_context yield) {
                return sendShare(common, parseSingleMessageAccessParams(ctx), parseSendShareParams(ctx), req, config, ctx, yield);
            }));

        bindGET<MessageResult>(info, "/v1/message",
            handler(config, [=] (CommonParams common, RequestContext req, http_api::Context ctx, boost::asio::yield_context yield) {
                return message(common, parseSingleMessageAccessParams(ctx), req, config, ctx, yield);
            }));

        bindGET<FilterSearchResult>(info, "/v1/filter_search",
            handler(config, [=] (CommonParams common, RequestContext req, http_api::Context ctx, boost::asio::yield_context yield) {
                return filterSearch(common, parseMessagesAccessParams(ctx), req, config, ctx, yield);
            }));

        bindGET<AttachSidResult>(info, "/v1/attach_sid",
            handler(config, [=] (CommonParams common, RequestContext req, http_api::Context ctx, boost::asio::yield_context yield) {
                return attachSid(common, parseSingleMessageAccessParams(ctx), req, config, ctx, yield);
            }));
    }

    void sib(const http_api::BindInfo<Response>& info, const http_api::LogArgsAsTskvConfig&, const ConfigPtr& config) const {

        bindGET<ActionHistoryResult>(info, "/v1/history",
            handler(config, [=] (CommonParams common, RequestContext, http_api::Context ctx, boost::asio::yield_context yield) {
                return actionHistory(common, parseActionHistoryParams(ctx), config, yield);
            }));
    }
};

}

DEFINE_SERVICE_OBJECT(spaniel::HttpApi)
