#include <mail/spaniel/http_api/include/response.h>
#include <mail/spaniel/http_api/include/result_reflection.h>
#include <mail/spaniel/core/include/types_error.h>
#include <mail/spaniel/core/include/types_reflection.h>
#include <mail/webmail/http_api_helpers/include/context.h>

#include <boost/algorithm/string/join.hpp>
#include <boost/range/adaptor/transformed.hpp>

namespace spaniel {

void Response::with(SearchCreateResult&& resp) const {
    defaultJsonResponse(std::move(resp));
}

void Response::with(SearchShowResult&& resp) const {
    defaultJsonResponse(std::move(resp));
}

void Response::with(SearchListResult&& resp) const {
    defaultJsonResponse(std::move(resp));
}

void Response::with(SearchLastIdResult&& resp) const {
    defaultJsonResponse(std::move(resp));
}

void Response::with(ActionHistoryResult&& resp) const {
    defaultJsonResponse(std::move(resp));
}

void Response::with(OrganizationShowResult&& resp) const {
    defaultJsonResponse(std::move(resp));
}

void Response::with(MessagesBySearchResult&& resp) const {
    std::string body = R"({"data":{)";

    auto transformedRange = resp.data | boost::adaptors::transformed([] (const auto& p) {
        return fmt::format(R"("{}":{})", std::to_string(p.first), p.second);
    });
    body.append(boost::join(transformedRange, ", "));
    if (resp.next) {
        body.append(R"(},"next":")");
        body.append(*resp.next);
        body.append(R"("})");
    } else {
        body.append("}}");
    }

    try {
        logSuccessResponse(body);
    } catch (const std::exception& ex) {
        logExceptionToDefaultLogger(ex);
    }

    stream->set_code(ymod_webserver::codes::ok);
    stream->set_content_type("application", "json");
    stream->result_body(body);
}

void Response::with(MessagesBySearchAsyncResult&& resp) const {
    std::visit([this] (auto&& r) {with(std::move(r));}, std::move(resp));
}

void Response::with(JsonProxyResult&& resp) const {
    defaultProxyResponse(std::move(resp));
}

void Response::defaultProxyResponse(JsonProxyResult&& resp) const {
    try {
        logSuccessResponse(resp.body);
    } catch (const std::exception& ex) {
        logExceptionToDefaultLogger(ex);
    }

    stream->set_code(resp.code);
    stream->set_content_type("application", "json");
    if (const auto& contentEncoding = resp.headers.find("content-encoding"); contentEncoding != resp.headers.end()) {
        stream->add_header("Content-Encoding", contentEncoding->second);
    }
    if (resp.headers.contains("transfer-encoding")) {
        stream->result_chunked()->client_stream() << resp.body;
    } else {
        stream->result_body(resp.body);
    }
}

void Response::logSuccessResponse(const std::string& body) const {
    http_api::Context ctx{stream->request()};

    const auto contextLogger = getContextLogger(
        ctx.optionalArg("adminUid"), ctx.optionalArg("orgId"), ctx.optionalHeader("x-request-id")
    );

    LOGDOG_(contextLogger, notice,
            log::where_name=stream->request()->url.make_full_path(),
            log::message=body);
}

void Response::logErrorResponse(const std::string& body) const {
    http_api::Context ctx{stream->request()};

    const auto contextLogger = getContextLogger(
        ctx.optionalArg("adminUid"), ctx.optionalArg("orgId"), ctx.optionalHeader("x-request-id")
    );

    LOGDOG_(contextLogger, error,
            log::where_name=stream->request()->url.make_full_path(),
            log::message=body);
}

bool Response::errorCodeToResponseCode(const mail_errors::error_code& ec, ymod_webserver::codes::code& code) const {
    if (corgi::errorCodeToResponseCode(ec, code)) {
        return true;
    } else if (ec.category() == getServiceCategory()) {
        switch(ServiceError(ec.value())) {
            case ServiceError::noSearchWithSearchId:
                code = ymod_webserver::codes::bad_request; break;
        }
        return true;
    } else if (ec.category() == getRemoteServiceCategory()) {
        switch (RemoteServiceError(ec.value())) {
            case RemoteServiceError::search:
            case RemoteServiceError::proxy:
                code = ymod_webserver::codes::internal_server_error; break;
        }
        return true;
    } else {
        return false;
    }
}

}
