#include <mail/spaniel/service/include/handlers/messages_by_search.h>
#include <mail/spaniel/service/include/handlers/search.h>
#include <mail/spaniel/service/include/message_access.h>
#include <mail/spaniel/service/include/search.h>
#include <mail/spaniel/service/include/hound.h>
#include <mail/spaniel/core/include/types_error.h>


namespace spaniel {
namespace {
template<class Params>
Search searchById(const CommonParams& common, const Params& params, ConfigPtr cfg, boost::asio::yield_context yield) {
    const boost::optional<Search> search = cfg->repo->searchById(common, params.searchId, io_result::make_yield_context(yield));
    params.validate(search).value_or_throw();

    return *search;
}

SearchResults cachedResults(const CommonParams& common, const MessagesBySearchParams& params, const Search& search, ConfigPtr cfg, boost::asio::yield_context yield) {
    auto range = cfg->repo->searchByAllUsers(common, search, params.page, io_result::make_yield_context(yield));
    return SearchResults(std::make_move_iterator(range.begin()), std::make_move_iterator(range.end()));
}

SearchResults cachedResults(const CommonParams& common, const MessagesBySearchAndUidParams& params, const Search& search, ConfigPtr cfg, boost::asio::yield_context yield) {
    auto range = cfg->repo->searchByOneUser(common, search, params.uid, params.page, io_result::make_yield_context(yield));
    return SearchResults(std::make_move_iterator(range.begin()), std::make_move_iterator(range.end()));
}

template<class Params>
bool needToAddResults(const SearchResults& results, const Params& params, const Search& search) {
    return results.size() < params.page.count() && search.state == SearchState::partially_complete;
}

template<class Params>
MessagesBySearchResult saturateWithEnvelopes(const SearchResults& results, const Params& params, RequestContext req, ConfigPtr cfg, boost::asio::yield_context yield) {
    MessagesBySearchResult result;

    if (results.empty()) {
        return result;
    }

    std::map<Uid, std::vector<Id>> uidToMids;
    for (const auto& res: results) {
        uidToMids[res.uid].emplace_back(res.id);
    }

    for (const auto& [uid, mids]: uidToMids) {
        yamail::expected<std::string> envelopes = hound::filterSearch(uid, mids, *req.client, cfg->hound, yield);
        if (!envelopes) {
            throw mail_errors::system_error(make_error(RemoteServiceError::proxy, envelopes.error().what()));
        }
        result.data[uid] = std::move(envelopes).value();
    }
    if (results.size() == params.page.count()) {
        const unsigned first = params.page.first() ? std::stoll(*params.page.first()) : 0;
        const std::size_t next = first + params.page.count();
        result.next = std::to_string(next);
    }

    return result;
}
}

yamail::expected<MessagesBySearchResult> messagesBySearchAndUid(CommonParams common, MessagesBySearchAndUidParams params,
                                                                RequestContext req, ConfigPtr cfg,
                                                                boost::asio::yield_context yield) {

    const Search search = searchById(common, params, cfg, yield);
    SearchResults results = cachedResults(common, params, search, cfg, yield);

    if (needToAddResults(results, params, search)) {
        const auto yy = io_result::make_yield_context(yield);

        std::time_t dateTo;
        if (results.empty()) {
            if (params.page.first()) {
                dateTo = cfg->repo->minReceivedDateBySearchAndUid(common, params.searchId, params.uid, yy).value_or(*search.minSearchDate);
            } else {
                dateTo = *search.minSearchDate;
            }
        } else {
            dateTo = results.back().received_date;
        }

        const MailSearchCommonParams mailSearchCommonParams {
            .dateFrom = search.dateFrom,
            .dateTo = std::max(dateTo, 1l) - 1l,
            .length = params.page.count() - static_cast<unsigned>(results.size())
        };
        SearchResults freshSearchResults = makeSearch(search, params.uid, mailSearchCommonParams, *req.client, cfg->syncSearch,
                                                      Request::sync_search, yield);

        if (!freshSearchResults.empty()) {
            cfg->repo->cacheSearchResults(common, params.searchId, freshSearchResults, yy);

            auto end = freshSearchResults.end();
            if (results.size() + freshSearchResults.size() > params.page.count()) {
                end = freshSearchResults.begin() + params.page.count() - results.size();
            }
            std::copy(
                std::make_move_iterator(freshSearchResults.begin()),
                std::make_move_iterator(end),
                std::back_inserter(results)
            );
        }
    }

    return saturateWithEnvelopes(results, params, req, cfg, yield);
}

yamail::expected<MessagesBySearchAsyncResult> messagesBySearch(CommonParams common, MessagesBySearchParams params,
                                                               RequestContext req, ConfigPtr cfg,
                                                               boost::asio::yield_context yield) {

    const Search search = searchById(common, params, cfg, yield);
    SearchResults results = cachedResults(common, params, search, cfg, yield);

    if (needToAddResults(results, params, search)) {
        cfg->repo->continueSearch(common, params.searchId, io_result::make_yield_context(yield));

        const std::time_t dateTo = std::max(*search.minSearchDate, 1l) - 1l;

        detail::addSearchTask(common, params.searchId, search.dateFrom, dateTo, cfg, req.logger, yield);

        MessagesBySearchAsyncResult result = SearchCreateResult {
            .search_id=params.searchId
        };
        return result;
    }

    return MessagesBySearchAsyncResult { saturateWithEnvelopes(results, params, req, cfg, yield) };
}

}
