#include <mail/spaniel/service/include/handlers/organization.h>
#include <mail/spaniel/service/include/task_params.h>
#include <mail/spaniel/service/include/request.h>
#include <mail/spaniel/service/include/message_access.h>
#include <mail/spaniel/core/include/types_reflection.h>
#include <mail/spaniel/core/include/types_error.h>
#include <mail/ymod_queuedb_worker/include/task_control.h>


namespace spaniel {

using namespace http_getter::detail::operators;

yamail::expected<void> organizationActivateTaskImpl(OrganizationParams common, WorkerRequestContext req, WorkerConfigPtr cfg,
                                                    yplatform::task_context_ptr ctx, boost::asio::yield_context yield) {
    const auto yy = io_result::make_yield_context(yield);
                                            
    cfg->repo->activateOrganization(common, yy);

    ymod_queuedb::delayOnCancelledTask(ctx);

    const auto endpoint = cfg->billing.format(fmt::arg("org_id", std::to_string(common.orgId)));
    yamail::expected<void> result = make_unexpected(corgi::UnexpectedError::exception, "billing cannot activate organization with org_id: " + std::to_string(common.orgId));

    req.client->req(req.client->toPUT(endpoint))->call(Request::activate_organization, http_getter::withDefaultHttpWrap([&] (yhttp::response) {
        result = yamail::make_expected();
    }), yy);

    return result;
}

yamail::expected<void> organizationDeactivateTaskImpl(OrganizationParams common, WorkerRequestContext req, WorkerConfigPtr cfg,
                                                      yplatform::task_context_ptr ctx, boost::asio::yield_context yield) {
    const auto yy = io_result::make_yield_context(yield);
                                            
    cfg->repo->deactivateOrganization(common, cfg->intervalBeforeRemovingOrganization + std::time(nullptr), yy);

    ymod_queuedb::delayOnCancelledTask(ctx);

    const auto endpoint = cfg->billing.format(fmt::arg("org_id", std::to_string(common.orgId)));
    yamail::expected<void> result = make_unexpected(corgi::UnexpectedError::exception, "billing cannot deactivate organization with org_id: " + std::to_string(common.orgId));

    req.client->req(req.client->toDELETE(endpoint))->call(Request::deactivate_organization, http_getter::withDefaultHttpWrap([&] (yhttp::response) {
        result = yamail::make_expected();
    }), yy);

    return result;
}

yamail::expected<void> organizationActivateDeactivateTask(bool activate, OrganizationParams common, SwitchOrganizationParams params,
                                                          WorkerRequestContext req, WorkerConfigPtr cfg, bool lastTry,
                                                          yplatform::task_context_ptr ctx, boost::asio::yield_context yield) {
    const auto yy = io_result::make_yield_context(yield);

    if (!cfg->repo->registerTaskId(common, params.taskId, yy)) {
        return ymod_queuedb::make_unexpected(ymod_queuedb::TaskControl::delay);
    }

    const auto action = activate ? organizationActivateTaskImpl : organizationDeactivateTaskImpl;
                                            
    yamail::expected<void> result;
    try {
        result = action(common, req, cfg, ctx, yield);
        if (result) {
            cfg->repo->removeTaskId(common, params.taskId, yy);
        }
    } catch (const boost::coroutines::detail::forced_unwind&) {
        throw;
    } catch (const boost::system::system_error& e) {
        result = yamail::make_unexpected(mail_errors::error_code(e.code()));
    } catch (const std::exception& e) {
        result = make_unexpected(corgi::UnexpectedError::exception, e.what());
    }

    if (!result && lastTry) {
        LOGDOG_(req.logger, error,
            log::org_id=std::to_string(common.orgId),
            log::message="task cannot switch organization state");

        return ymod_queuedb::make_unexpected(ymod_queuedb::TaskControl::delay);
    }
    
    return result;
}

yamail::expected<OrganizationEnableResult> organizationEnable(OrganizationParams common, ConfigPtr cfg, boost::asio::yield_context yield) {
    const auto yy = io_result::make_yield_context(yield);
    const auto org = cfg->repo->getOrganization(common, yy);
    
    if (org && org->state == OrganizationState::frozen) {
        cfg->queuedb->addTask(
            ymod_queuedb::Uid(0), organizationActivateType(),
            dumpOrganizationParams(common),
            cfg->ogranizationSwitchTaskTimeout, ymod_queuedb::RequestId(common.requestId), yy
        );
    }

    return {};
}

yamail::expected<OrganizationActivateResult> organizationActivate(CommonParams common, ConfigPtr cfg, boost::asio::yield_context yield) {
    const auto yy = io_result::make_yield_context(yield);

    const auto params = OrganizationParams { .orgId = common.orgId, .requestId = common.requestId };
    const auto org = cfg->repo->getOrganization(params, yy);

    if (!org || org->state == OrganizationState::disabled) {
        cfg->queuedb->addTask(
            ymod_queuedb::Uid(common.adminUid), organizationActivateType(),
            dumpOrganizationParams(std::move(params)),
            cfg->ogranizationSwitchTaskTimeout, ymod_queuedb::RequestId(std::move(common.requestId)), yy
        );
    }
                                                                
    return {};
}

yamail::expected<OrganizationShowResult> organizationShow(OrganizationParams common, ConfigPtr cfg,
                                                          boost::asio::yield_context yield) {

    OrganizationShowResult ret;
    const auto org = cfg->repo->getOrganization(common, io_result::make_yield_context(yield));
    if (org) {
        ret.state = org->state;
    }

    return ret;
}

yamail::expected<OrganizationDisableResult> organizationDisable(OrganizationParams common, ConfigPtr cfg, boost::asio::yield_context yield) {
    cfg->repo->disableOrganization(common, cfg->intervalBeforeRemovingOrganization + std::time(nullptr), io_result::make_yield_context(yield));

    return {};
}

yamail::expected<OrganizationDeactivateResult> organizationDeactivate(CommonParams common, ConfigPtr cfg, boost::asio::yield_context yield) {
    const auto yy = io_result::make_yield_context(yield);

    const auto params = OrganizationParams  { .orgId = common.orgId, .requestId = common.requestId };
    const auto org = cfg->repo->getOrganization(params, yy);

    if (org && org->state != OrganizationState::disabled) {
        cfg->queuedb->addTask(
            ymod_queuedb::Uid(common.adminUid), organizationDeactivateType(),
            dumpOrganizationParams(std::move(params)),
            cfg->ogranizationSwitchTaskTimeout, ymod_queuedb::RequestId(std::move(common.requestId)), yy
        );
    }

    return {};
}

yamail::expected<EmptyResult> organizationUpdateAll(const RequestId& requestId, ConfigPtr cfg, boost::asio::yield_context yield) {
    cfg->queuedb->addTask(
        ymod_queuedb::Uid(0), organizationUpdateAllType(),
        dumpOrganizationUpdateAllParams(requestId), cfg->ogranizationUpdateAllTaskTimeout,
        ymod_queuedb::RequestId(requestId), io_result::make_yield_context(yield)
    );

    return EmptyResult();
}

yamail::expected<void> organizationUpdateAll(const RequestId& requestId, http_getter::TypedClientPtr client, WorkerConfigPtr cfg, 
                                             yplatform::task_context_ptr ctx, boost::asio::yield_context yield) {
    const auto range = cfg->repo->listActiveOrganizationIds(requestId, io_result::make_yield_context(yield));

    std::vector<OrgId> ids;
    std::copy(std::make_move_iterator(range.begin()), std::make_move_iterator(range.end()), std::back_inserter(ids));

    for (OrgId id: ids) {
        OrganizationParams params { .orgId=id, .requestId=requestId };
        OrganizationResolverWithoutAdminUid resolver(cfg->resolverConfig, client, params);

        LOGDOG_(getContextLogger(params), notice, log::org_id=std::to_string(id.t), log::message="updating organization");

        ymod_queuedb::delayOnCancelledTask(ctx);
        organizationUpdate(params, resolver, cfg, ctx, yield).value_or_throw();
    }

    return {};
}

yamail::expected<void> organizationUpdate(OrganizationParams common, OrganizationResolverWithoutAdminUid resolver,
                                          WorkerConfigPtr cfg, yplatform::task_context_ptr ctx, boost::asio::yield_context yield) {
    const auto org = cfg->repo->getOrganization(common, io_result::make_yield_context(yield));

    if (!org || org->state != OrganizationState::active) {
        return {};
    }

    const auto yy = io_result::make_yield_context(yield);

    UidsSet currentUids;
    boost::copy(
        resolver.resolveUsers(ResolveUsers(), yield),
        std::inserter(currentUids, currentUids.end())
    );

    ymod_queuedb::delayOnCancelledTask(ctx);

    const auto cachedUids = cfg->repo->getOrganizationUids(common, yy);
    OrganizationUpdateParams params;

    boost::copy(currentUids, std::back_inserter(params.all));

    std::set_difference (
        currentUids.begin(), currentUids.end(), cachedUids.begin(), cachedUids.end(),
        std::inserter(params.newbie, params.newbie.end())
    );

    if (!params.newbie.empty()) {
        cfg->queuedb->addTask(
            ymod_queuedb::Uid(0), organizationUpdateType(),
            dumpOrganizationUpdateParams(common, params),
            cfg->ogranizationUpdateTaskTimeout, ymod_queuedb::RequestId(common.requestId), yy
        );
    }

    return {};
}


yamail::expected<void> organizationUpdate(OrganizationParams common, OrganizationUpdateParams params, WorkerRequestContext req,
                                          WorkerConfigPtr cfg, yplatform::task_context_ptr ctx, boost::asio::yield_context yield) {
    const auto yy = io_result::make_yield_context(yield);
    const auto org = cfg->repo->getOrganization(common, yy);

    if (org && org->state == OrganizationState::active) {
        for (const Uid& uid: params.newbie) {
            bool ok = false;
            req.client->req(
                req.client
                    ->toPOST(cfg->enableUser)
                     .getArgs("uid"_arg=std::to_string(uid.t))
            )->call(Request::enable_user, http_getter::withDefaultHttpWrap([&] (yhttp::response) { ok = true; }), yy);

            if (!ok) {
                return make_unexpected(
                    corgi::UnexpectedError::exception,
                    "cannot update user with uid " + std::to_string(uid.t)
                );
            }

            ymod_queuedb::delayOnCancelledTask(ctx);
        }

        const int updated = cfg->repo->updateOrganizationUids(common, std::move(params.all), yy);

        if (updated == 0) {
            return make_unexpected(corgi::UnexpectedError::exception, "cannot update organization uids");
        }
    }

    return {};
}

}
