#include <mail/spaniel/service/include/handlers/search.h>
#include <mail/spaniel/service/include/task_params.h>
#include <mail/spaniel/service/include/message_access.h>
#include <mail/spaniel/service/include/search.h>
#include <mail/spaniel/service/include/hound.h>
#include <mail/spaniel/core/include/types_error.h>
#include <mail/spaniel/core/include/types_reflection.h>
#include <mail/ymod_queuedb_worker/include/task_control.h>
#include <yamail/data/serialization/json_writer.h>
#include <mail/mail_errors/error_result/error_result.h>

#include <algorithm>
#include <limits>


namespace spaniel {

std::string makeQueryJson(const Query& sp) {
    return yamail::data::serialization::JsonWriter(sp).result();
}

namespace detail {
void addSearchTask(const CommonParams& common, SearchId searchId, std::time_t dateFrom, std::time_t dateTo,
                   ConfigPtr cfg, ContextLogger logger, boost::asio::yield_context yield) {
    AsyncSearchParams continuation {
        .searchId=searchId,
        .dateFrom=dateFrom,
        .dateTo=dateTo
    };

    const auto yy = io_result::make_yield_context(yield);

    const auto taskId = cfg->queuedb->addTask(
        ymod_queuedb::Uid(common.adminUid), asyncSearchType(), dumpAsyncSearchParams(common, continuation),
        cfg->asyncSearchTaskTimeout, ymod_queuedb::RequestId(common.requestId),
        yy
    );

    LOGDOG_(logger, notice,
            log::message="task created",
            log::task_type=asyncSearchType(),
            log::task_id=taskId);
}
}

yamail::expected<void> asyncSearchImpl(const CommonParams& common, const AsyncSearchParams& params,
                                       WorkerRequestContext req, WorkerConfigPtr cfg, const ymod_queuedb::ExecOrWait& eow,
                                       yplatform::task_context_ptr ctx, boost::asio::yield_context yield) {

    const auto yy = io_result::make_yield_context(yield);

    boost::optional<Search> search;
    eow([&] () {
        search = cfg->repo->searchById(common, params.searchId, yy);
        return static_cast<bool>(search);
    });

    if (!search) {
        return yamail::make_expected();
    }

    ymod_queuedb::delayOnCancelledTask(ctx);

    MailSearchCommonParams mailSearchCommonParams {
        .dateFrom = params.dateFrom,
        .dateTo = params.dateTo,
        .length = cfg->asyncSearchResultLength
    };
    SearchResults results = makeSearch(*search, mailSearchCommonParams, *req.client, cfg->search,
                                       Request::async_search, yield);

    ymod_queuedb::delayOnCancelledTask(ctx);

    const SearchState targetState = results.size() < mailSearchCommonParams.length ? SearchState::complete
                                                                                   : SearchState::partially_complete;

    cfg->repo->fillSearch(common, params.searchId, results, targetState, yy);

    return yamail::make_expected();
}

yamail::expected<void> asyncSearch(CommonParams common, AsyncSearchParams params,
                                   WorkerRequestContext req, WorkerConfigPtr cfg, const ymod_queuedb::ExecOrWait& eow,
                                   bool lastTry, yplatform::task_context_ptr ctx, boost::asio::yield_context yield) {
    yamail::expected<void> result;
    try {
        result = asyncSearchImpl(common, params, req, cfg, eow, ctx, yield);
    } catch (const boost::coroutines::detail::forced_unwind&) {
        throw;
    } catch (const boost::system::system_error& e) {
        result = yamail::make_unexpected(mail_errors::error_code(e.code(), e.what()));
    } catch (const std::exception& e) {
        result = make_unexpected(corgi::UnexpectedError::exception, e.what());
    }

    if (!result && lastTry) {
        mail_errors::error_code ec;
        const auto yy = io_result::make_yield_context(yield, ec);
        cfg->repo->failSearch(common, params.searchId, mail_errors::makeErrorResult(result.error()), yy);

        if (ec) {
            LOGDOG_(req.logger, error,
                    log::message="cannot fail running search",
                    log::error_code=ec);
            return ymod_queuedb::make_unexpected(ymod_queuedb::TaskControl::delay);
        }
    }

    return result;
}

yamail::expected<SearchCreateResult> searchCreate(CommonParams common, SearchCreateParams params,
                                                  RequestContext req, ConfigPtr cfg, boost::asio::yield_context yield) {
    const auto yy = io_result::make_yield_context(yield);

    Uids filtered = req.orgResolver.resolveUsers(params.queryParams.resolveUsers, yield);
    if (filtered.empty()) {
        return make_unexpected(corgi::ResolveOrganizationError::noPddUidsInRequest);
    }

    SearchId searchId = cfg->repo->reserveSearchId(common, yy);

    detail::addSearchTask(common, searchId, params.dateFrom, params.dateTo, cfg, req.logger, yield);

    cfg->repo->createSearch(
        common, searchId, params.dateFrom, params.dateTo, makeQueryJson(params.queryParams),
        std::move(params.name.value_or("")), filtered, yy
    );

    return SearchCreateResult {
        .search_id=searchId
    };
}

yamail::expected<SearchShowResult> searchShow(CommonParams common, SearchShowParams params,
                                              ConfigPtr cfg, boost::asio::yield_context yield) {
    const auto yy = io_result::make_yield_context(yield);

    boost::optional<Search> search = cfg->repo->searchById(common, params.searchId, yy);

    if (search) {
        return *search;
    } else {
        return make_unexpected(ServiceError::noSearchWithSearchId, std::to_string(params.searchId));
    }
}

yamail::expected<SearchRenameResult> searchRename(CommonParams common, SearchRenameParams params,
                                                  ConfigPtr cfg, boost::asio::yield_context yield) {
    const int updated = cfg->repo->renameSearch(common, params.searchId, params.name, io_result::make_yield_context(yield));

    if (updated == 0) {
        return make_unexpected(ServiceError::noSearchWithSearchId, std::to_string(params.searchId));
    } else {
        return SearchRenameResult();
    }
}

yamail::expected<SearchListResult> searchList(CommonParams common, SearchListParams params,
                                              ConfigPtr cfg, boost::asio::yield_context yield) {
    auto range = cfg->repo->searchList(
        common, params.page, io_result::make_yield_context(yield)
    );

    SearchListResult res;
    std::copy(std::make_move_iterator(range.begin()), std::make_move_iterator(range.end()), std::back_inserter(res.data));

    if (!res.data.empty()) {
        res.next = std::to_string(res.data.back().searchId.t);
    }

    return res;
}

yamail::expected<SearchLastIdResult> searchLastId(CommonParams common, ConfigPtr cfg, boost::asio::yield_context yield) {
    const auto yy = io_result::make_yield_context(yield);

    SearchLastIdResult result;
    if (const auto id = cfg->repo->lastSearchId(common, yy); id) {
        result.search_id = *id;
    }

    return result;
}

yamail::expected<SearchArchiveResult> searchArchive(CommonParams common, SearchArchiveParams params, ConfigPtr cfg, boost::asio::yield_context yield) {
    cfg->repo->archiveSearch(common, params.searchId, io_result::make_yield_context(yield));

    return yamail::expected<SearchArchiveResult>();
}

}
