#include <mail/spaniel/service/include/proxy_helpers.h>
#include <mail/spaniel/core/include/types_error.h>

#include <boost/algorithm/string.hpp>


namespace spaniel {

HttpArguments makeProxyArgs(const std::string& url) {
    HttpArguments args;

    args.fromUrl(url);

    return args;
}

http::headers makeProxyHeaders(const http_api::Context& ctx, std::function<bool(const std::string&)> filter) {
    http::headers headers;

    for (const auto& it: ctx.req->headers) {
        if (filter(it.first)) {
            headers.add(it.first, it.second);
        }
    }
    
    return headers;
}

bool proxyHeadersWithoutTickets(const std::string& header) {
    const std::string lowercasedHeader = boost::algorithm::to_lower_copy(header);

    return lowercasedHeader != "x-ya-service-ticket" && lowercasedHeader != "x-ya-user-ticket"
        && lowercasedHeader != "x-request-id" && lowercasedHeader != "host"; 
}

auto proxyResponseHandler(yamail::expected<JsonProxyResult>& result) {
    return [&](yhttp::response resp) {
        result = JsonProxyResult { 
            .code = static_cast<ymod_webserver::codes::code>(resp.status), 
            .body = std::move(resp.body),
            .headers = std::move(resp.headers)
        };

        return http_getter::Result::success;
    };
}

yamail::expected<JsonProxyResult> doProxyGetRequest(HttpArguments args, http::headers headers, http_getter::TypedClientPtr client, 
                                                    const http_getter::TypedEndpoint& ep, Request requestType,
                                                    RemoteServiceError defaultError, boost::asio::yield_context yield) {
    using namespace http_getter::detail::operators;

    yamail::expected<JsonProxyResult> result = make_unexpected(defaultError, "unknown error");

    auto request = client->toGET(ep)
                                .getArgs("args"_arg=std::move(args))
                                .headers("hdrs"_hdr=std::move(headers));

    client->req(std::move(request))->call(requestType, proxyResponseHandler(result), io_result::make_yield_context(yield));

    return result;
}

yamail::expected<JsonProxyResult> doProxyPostRequest(HttpArguments args, http::headers headers, std::string body,
                                                     http_getter::TypedClientPtr client, const http_getter::TypedEndpoint& ep,
                                                     Request requestType, RemoteServiceError defaultError,
                                                     boost::asio::yield_context yield) {
    using namespace http_getter::detail::operators;

    yamail::expected<JsonProxyResult> result = make_unexpected(defaultError, "unknown error");

    auto request = client->toPOST(ep)
                                .getArgs("args"_arg=std::move(args))
                                .body(std::move(body))
                                .headers("hdrs"_hdr=std::move(headers));

    client->req(std::move(request))->call(requestType, proxyResponseHandler(result), io_result::make_yield_context(yield));

    return result;
}

}
