#include <mail/spaniel/service/include/search.h>
#include <mail/webmail/corgi/include/trycatch.h>
#include <mail/spaniel/core/include/types_error.h>
#include <yamail/data/deserialization/json_reader.h>
#include <yamail/data/serialization/json_writer.h>


namespace spaniel {
namespace reflect {
struct Request {
    const Uids& users;
};

struct Document {
    std::string mid;
    std::time_t received_date;
};
using Documents = std::vector<Document>;

struct Result {
    std::string uid;
    Documents documents;
};
struct Response {
    std::vector<Result> result;
};
}
namespace detail {
namespace yds = yamail::data::serialization;
namespace ydd = yamail::data::deserialization;
namespace hg = http_getter;
using namespace http_getter::detail::operators;


template <typename RequestBuilder>
void addQueryArgs(RequestBuilder& requestBuilder, const Query& query) {
    auto apply = [&] (auto argName, auto optValue) {
        if (optValue) {
            requestBuilder.getArgs(argName=*optValue);
        }
    };

    apply("body_text"_arg, query.text);
    apply("hdr_subject"_arg, query.subject);
    apply("hdr_from"_arg, query.from);
    apply("hdr_to"_arg, query.to);
    apply("hdr_cc"_arg, query.cc);
    apply("hdr_bcc"_arg, query.bcc);
    apply("hdr_to_cc_bcc"_arg, query.toCcBcc);
    apply("scope"_arg, query.scope);
    apply("request"_arg, query.request);

    if (query.hasAttachments) {
        requestBuilder.getArgs("has_attachments"_arg="true");
    }
}

reflect::Response perform(const Search& search, const Uids& uids, const MailSearchCommonParams& params, const http_getter::TypedClient& client,
                          const http_getter::TypedEndpoint& endpoint, Request requestType, boost::asio::yield_context yield) {

    yamail::expected<reflect::Response> result = make_unexpected(RemoteServiceError::search, "uninitialized search result");

    const auto h = hg::withDefaultHttpWrap([&] (yhttp::response resp) {
        trycatch(result, [&] () {
            result = ydd::JsonReader<reflect::Response>(resp.body).result();
        });
        return result ? hg::Result::success : hg::Result::retry;
    });

    auto request = client.toPOST(endpoint)
            .body(yds::JsonWriter(reflect::Request{.users=uids}).result())
            .getArgs(
                "length"_arg=std::to_string(params.length + ADDITIONAL_SEARCH_RESULTS),
                "from"_arg=std::to_string(params.dateFrom),
                "to"_arg=std::to_string(params.dateTo),
                "exclude-trash"_arg="false",
                "exclude-hidden-trash"_arg="false",
                "exclude-spam"_arg="false",
                "exclude-draft"_arg="true",
                "org_id"_arg=std::to_string(search.orgId),
                "search_id"_arg=std::to_string(search.searchId)
            );
    addQueryArgs(request, search.query);

    client.req(std::move(request))->call(requestType, h, io_result::make_yield_context(yield));

    return result.value_or_throw();
}

void flat(reflect::Response&& resp, SearchResults& ret) {
    for (const reflect::Result& r: resp.result) {
        Uid uid = makeUid(r.uid);

        boost::copy(
            r.documents
            | boost::adaptors::transformed([&] (const reflect::Document& doc) {
                return SearchResult {
                    .uid=uid,
                    .id=makeId(doc.mid),
                    .received_date=doc.received_date,
                };
            })
            , std::back_inserter(ret)
        );
    }
}

bool comparator(const SearchResult& lhs, const SearchResult& rhs) {
    if (lhs.received_date != rhs.received_date) {
         return lhs.received_date > rhs.received_date;
    }

    if (lhs.id != rhs.id) {
        return lhs.id > rhs.id;
    }

    return lhs.uid > rhs.uid;
}

void sort(SearchResults& ret) {
    std::sort(ret.begin(), ret.end(), comparator);
}

std::optional<std::time_t> extractNextFromSequence(const SearchResults& results, std::size_t requestedSize) {
    const bool tailWithUnknownLengthIsFound = results.size() > requestedSize;
    if (tailWithUnknownLengthIsFound) {
        const bool lastResultHasSameDateEqualsToTheLastRequested = results[requestedSize - 1].received_date == results.back().received_date;

        if (lastResultHasSameDateEqualsToTheLastRequested) {
            const bool weDoNotShureTheEndIsFound = requestedSize + ADDITIONAL_SEARCH_RESULTS == results.size();

            if (weDoNotShureTheEndIsFound) {
                return results.back().received_date;
            }
        }
    }

    return std::nullopt;
}

SearchResults requestFullPageOfResults(std::time_t date, const RequestResults& requestResults) {

    SearchResults results;
    MailSearchCommonParams params { .dateFrom=date, .dateTo=date, .length=0 };

    do {
        results.clear();
        params.length += PAGE_LENGTH_FOR_LOADING_RESULTS;
        requestResults(params, results);
    } while (results.size() >= params.length);

    return results;
}

SearchResults search(const MailSearchCommonParams& params, const RequestResults& requestResults) {
    SearchResults results;

    requestResults(params, results);
    sort(results);

    if (const auto possibleNext = extractNextFromSequence(results, params.length); possibleNext) {
        SearchResults fresh = requestFullPageOfResults(*possibleNext, requestResults);

        std::copy(
            std::make_move_iterator(fresh.begin()),
            std::make_move_iterator(fresh.end()),
            std::back_inserter(results)
        );
        sort(results);
    }

    return results;
}
}

using namespace detail;

SearchResults makeSearch(const Search& search, Uid uid, const MailSearchCommonParams& params, const http_getter::TypedClient& client,
                         const http_getter::TypedEndpoint& endpoint, Request requestType, boost::asio::yield_context yield) {

    const auto requestResults =
        [&search, &client, &endpoint, uid, requestType, yield] (const MailSearchCommonParams& params, SearchResults& results) {
            flat(perform(search, { uid }, params, client, endpoint, requestType, yield), results);
    };

    return detail::search(params, requestResults);
}

SearchResults makeSearch(const Search& search, const MailSearchCommonParams& params, const http_getter::TypedClient& client,
                         const http_getter::TypedEndpoint& endpoint, Request requestType, boost::asio::yield_context yield) {

    const auto requestResults =
        [&search, &client, &endpoint, requestType, yield] (const MailSearchCommonParams& params, SearchResults& results) {
            flat(perform(search, search.requestedUids, params, client, endpoint, requestType, yield), results);
    };

    return detail::search(params, requestResults);
}

}

BOOST_FUSION_ADAPT_STRUCT(spaniel::reflect::Request, users)

BOOST_FUSION_ADAPT_STRUCT(spaniel::reflect::Document, mid, received_date)

BOOST_FUSION_ADAPT_STRUCT(spaniel::reflect::Result, uid, documents)

BOOST_FUSION_ADAPT_STRUCT(spaniel::reflect::Response, result)
