from argparse import ArgumentParser
from mail.unistat.cpp.cython.meters import (
    HttpClientHttpRequestTotalTimeHist,
    HttpClientHttpRequestCountByStatus,
    SupervisorLogRestartMeters,
)
from collections import namedtuple
import os
import yaml
import logging
import mail.unistat.cpp.cython.logs as logs
import mail.spaniel.unistat.cpp.run as unistat
from mail.ymod_queuedb_worker.unistat.cpp.metrics import WorkerMetricsLog, WorkerMetric


logging.basicConfig(level=logging.WARNING, format='[%(asctime)s] [%(levelname)s]: %(message)s')
log = logging.getLogger(__name__)


def parse_args():
    parser = ArgumentParser()
    parser.add_argument('-H', '--host', default='::')
    parser.add_argument('-p', '--port', default=8082, type=int)
    parser.add_argument('-d', '--dir', default='.')
    parser.add_argument('-s', action='store_true', help='read file logs from start')
    parser.add_argument('-l', '--log', default='/var/log/spaniel/unistat.log', help='path for yplatform.log')
    parser.add_argument('--supervisorlog', help='path for supervisord.log')
    parser.add_argument('spaniel_config')
    return parser.parse_args()


def remove_scheme(s):
    return s.replace('https://', '').replace('http://', '')


def remove_scheme_and_port(s):
    s = remove_scheme(s)
    i = s.rfind(':')
    if i != -1:
        return s[:i]
    return s


def make_spaniel_config(data):
    http_api = list(filter(lambda m: m['_name'] == 'http_api', data['config']['modules']['module']))[0]
    worker = list(filter(lambda m: m['_name'] == 'worker', data['config']['modules']['module']))[0]

    return SpanielConfig(
        httpclient_log=os.path.join(os.curdir, data['config']['log']['http_client']['sinks'][0]['path']),
        spaniel_log=os.path.join(os.curdir, data['config']['log']['spaniel']['sinks'][0]['path']),
        worker_log=os.path.join(os.curdir, data['config']['log']['worker_access']['sinks'][0]['path']),
        blackbox_host=remove_scheme_and_port(http_api['configuration']['blackbox']['url']),
        directory_host=remove_scheme_and_port(http_api['configuration']['directory']['users']['url']),
        search_host=remove_scheme_and_port(worker['configuration']['tasks']['async_search']['endpoint']['url']),
        hound_host=remove_scheme_and_port(http_api['configuration']['hound']['url']),
        mbody_host=remove_scheme_and_port(http_api['configuration']['mbody']['url']),
        sendbernar_host=remove_scheme_and_port(http_api['configuration']['sendbernar']['url']),
    )


def make_http_client_log_meters(cfg):
    meters = []

    for host in [cfg.blackbox_host, cfg.directory_host, cfg.search_host, cfg.hound_host, cfg.mbody_host, cfg.sendbernar_host]:
        meters.append(HttpClientHttpRequestCountByStatus(host, "http_client"))
        meters.append(HttpClientHttpRequestTotalTimeHist((0, 20, 50, 100, 300, 500), host, "http_client_"+host))

    return meters


def make_supervisor_meters():
    return [
        SupervisorLogRestartMeters('supervisor')
    ]


def make_worker_log_meters():
    return [WorkerMetric()]


SpanielConfig = namedtuple('SpanielConfig', (
    'httpclient_log',
    'spaniel_log',
    'worker_log',
    'blackbox_host',
    'directory_host',
    'search_host',
    'hound_host',
    'mbody_host',
    'sendbernar_host',
))


def main():
    args = parse_args()
    log.info('chdir %s' % os.path.abspath(args.dir))
    os.chdir(args.dir)

    with open(args.spaniel_config) as f:
        spaniel_config = make_spaniel_config(yaml.load(f, Loader=yaml.FullLoader))

    fast_forward = args.s

    http_client_log = logs.HttpClientLog(
        [],
        make_http_client_log_meters(spaniel_config),
        fast_forward,
        spaniel_config.httpclient_log
    )

    worker_log = WorkerMetricsLog(
        [],
        make_worker_log_meters(),
        fast_forward,
        spaniel_config.worker_log
    )

    logs_list = [http_client_log, worker_log]

    if args.supervisorlog is not None:
        supervisor_log = logs.SupervisorLog(
            [],
            make_supervisor_meters(),
            fast_forward,
            args.supervisorlog
        )
        logs_list.append(supervisor_log)

    unistat.run(args.host, args.port, logs_list, args.log, logLevel='info')

if __name__ == '__main__':
    main()
