#include <yplatform/find.h>
#include <yplatform/module_registration.h>
#include <yplatform/loader.h>

#include <mail/spaniel/service/include/request_context.h>
#include <mail/spaniel/service/include/task_params.h>
#include <mail/spaniel/core/include/log.h>

#include <mail/spaniel/ymod_db/include/repository.h>

#include <mail/spaniel/service/include/handlers/search.h>
#include <mail/spaniel/service/include/handlers/organization.h>

#include <yamail/data/deserialization/ptree_reader.h>
#include <mail/http_getter/client/include/endpoint_reflection.h>

#include <mail/ymod_queuedb_worker/include/module.h>
#include <mail/ymod_queuedb_worker/include/exec_or_wait.h>

#include <mail/webmail/http_api_helpers/include/find_dependency.h>

#include <mail/ymod_queuedb/include/queue.h>


namespace spaniel {

struct Worker: public yplatform::module {
    void init(const yplatform::ptree& cfg) {
        namespace ydd = yamail::data::deserialization;
        using http_api::findDependency;

        const auto worker = findDependency<ymod_queuedb::WorkerModule>(cfg, "dependencies.worker");
        const auto getter = findDependency<http_getter::TypedClientModule>(cfg, "dependencies.http_getter");
        const auto repo = findDependency<spaniel::Repository>(cfg, "dependencies.spanieldb");
        const auto reactor = findDependency<yplatform::reactor>(cfg, "dependencies.reactor");

        auto ogranizationUpdateTimeout = yplatform::time_traits::duration_cast<std::chrono::seconds>(
            cfg.get<yplatform::time_traits::duration>("tasks.organization_update_all.timeout_dur")
        );

        auto intervalBeforeRemovingOrganization = yplatform::time_traits::duration_cast<std::chrono::seconds>(
            cfg.get<yplatform::time_traits::duration>("interval_before_removing_organization_dur")
        );

        const ymod_queuedb::ExecOrWaitFactory factory(cfg.get_child("exec_or_wait"), reactor);
        const auto config = std::make_shared<WorkerConfig>(WorkerConfig {
            .search=ydd::fromPtree<http_getter::TypedEndpoint>(cfg.get_child("tasks.async_search.endpoint")),
            .enableUser=ydd::fromPtree<http_getter::TypedEndpoint>(cfg.get_child("tasks.organization_update.endpoint")),
            .billing=ydd::fromPtree<http_getter::TypedEndpoint>(cfg.get_child("tasks.organization_enable.billing")),
            .asyncSearchResultLength=cfg.get<unsigned>("tasks.async_search.result_length"),
            .resolverConfig=corgi::fromPtree(cfg.get_child("tasks.organization_update_all")),
            .repo=repo,
            .queuedb=findDependency<ymod_queuedb::Queue>(cfg, "dependencies.queuedb"),
            .ogranizationUpdateTaskTimeout=ymod_queuedb::Timeout(ogranizationUpdateTimeout),
            .intervalBeforeRemovingOrganization=intervalBeforeRemovingOrganization.count(),
        });

        ymod_queuedb::addHandler(
            cfg.get_child("tasks.organization_activate"), organizationActivateType(), *worker,
            [=] (const ymod_queuedb::Task& task, bool lastTry, yplatform::task_context_ptr ctx, boost::asio::yield_context yield) {
                OrganizationParams common;
                SwitchOrganizationParams params;
                std::tie(common, params) = parseOrganizationSwitchParams(task);

                auto log = http_getter::withLog(getter->httpLogger("", common.requestId));
                WorkerRequestContext req(getter->create(common.requestId, log), common);

                return organizationActivateDeactivateTask(true, std::move(common), std::move(params),std::move(req), config, lastTry, ctx, yield);
            }
        );

        ymod_queuedb::addHandler(
            cfg.get_child("tasks.organization_deactivate"), organizationDeactivateType(), *worker,
            [=] (const ymod_queuedb::Task& task, bool lastTry, yplatform::task_context_ptr ctx, boost::asio::yield_context yield) {
                OrganizationParams common;
                SwitchOrganizationParams params;
                std::tie(common, params) = parseOrganizationSwitchParams(task);

                auto log = http_getter::withLog(getter->httpLogger("", common.requestId));
                WorkerRequestContext req(getter->create(common.requestId, log), common);

                return organizationActivateDeactivateTask(false, std::move(common), std::move(params), std::move(req), config, lastTry, ctx, yield);
            }
        );

        ymod_queuedb::addHandler(
            cfg.get_child("tasks.async_search"), asyncSearchType(), *worker,
            [=] (const ymod_queuedb::Task& task, bool lastTry, yplatform::task_context_ptr ctx, boost::asio::yield_context yield) {
                ymod_queuedb::ExecOrWait eow = factory.product(yield, ctx);
                CommonParams common;
                AsyncSearchParams params;
                std::tie(common, params) = parseAsyncSearchParams(task);

                auto log = http_getter::withLog(getter->httpLogger(std::to_string(common.adminUid.t), common.requestId));
                WorkerRequestContext req(getter->create(common.requestId, log), common);

                return asyncSearch(common, params, req, config, eow, lastTry, ctx, yield);
            }
        );

        ymod_queuedb::addHandler(
            cfg.get_child("tasks.organization_update"), organizationUpdateType(), *worker,
            [=] (const ymod_queuedb::Task& task, bool /*lastTry*/, yplatform::task_context_ptr ctx, boost::asio::yield_context yield) {
                OrganizationParams common;
                OrganizationUpdateParams params;
                std::tie(common, params) = parseOrganizationUpdateParams(task);

                auto log = http_getter::withLog(getter->httpLogger("", common.requestId));
                WorkerRequestContext req(getter->create(common.requestId, log), common);

                return organizationUpdate(common, params, req, config, ctx, yield);
            }
        );

        ymod_queuedb::addHandler(
            cfg.get_child("tasks.organization_update_all"), organizationUpdateAllType(), *worker,
            [=] (const ymod_queuedb::Task& task, bool /*lastTry*/, yplatform::task_context_ptr ctx, boost::asio::yield_context yield) {
                RequestId reqId = parseOrganizationUpdateAllParams(task);

                auto log = http_getter::withLog(getter->httpLogger("", reqId));

                return organizationUpdateAll(reqId, getter->create(reqId, log), config, ctx, yield);
            }
        );
    }
};

}

DEFINE_SERVICE_OBJECT(spaniel::Worker)
