#pragma once

#include <io_result/io_result.h>
#include <io_result/hooks.h>
#include <mail/spaniel/core/include/types.h>


namespace spaniel {

using OnOptionalSearchId = io_result::Hook<io_result::Optional<SearchId>>;
using OnSearchId = io_result::Hook<SearchId>;
using OnSearch = io_result::Hook<io_result::Optional<Search>>;
using OnOrganization = io_result::Hook<io_result::Optional<Organization>>;
using OnOrganizationUids = io_result::Hook<io_result::Sequence<Uid>>;
using OnOrganizationIds = io_result::Hook<io_result::Sequence<OrgId>>;
using OnSearches = io_result::Hook<io_result::Sequence<Search>>;
using OnSearchResults = io_result::Hook<io_result::Sequence<SearchResult>>;
using OnReceivedDate = io_result::Hook<std::time_t>;
using OnOptionalReceivedDate = io_result::Hook<io_result::Optional<std::time_t>>;
using OnMessagesInSearch = io_result::Hook<bool>;
using OnExecute = io_result::Hook<>;
using OnUpdate = io_result::Hook<int>;
using OnActionHistoryItems = io_result::Hook<io_result::Sequence<ActionHistoryItem>>;
using OnTaskCreated = io_result::Hook<bool>;

struct Repository {
    virtual ~Repository() = default;

    template <typename Handler>
    auto activateOrganization(const OrganizationParams& c, Handler handler) const {
        io_result::detail::init_async_result<Handler, OnExecute> init(handler);
        asyncActivateOrganization(c, init.handler);
        return init.result.get();
    }

    template <typename Handler>
    auto getOrganization(const OrganizationParams& c, Handler handler) const {
        io_result::detail::init_async_result<Handler, OnOrganization> init(handler);
        asyncGetOrganization(c, init.handler);
        return init.result.get();
    }

    template <typename Handler>
    auto listActiveOrganizationIds(const RequestId& reqId, Handler handler) const {
        io_result::detail::init_async_result<Handler, OnOrganizationIds> init(handler);
        asyncListActiveOrganizationIds(reqId, init.handler);
        return init.result.get();
    }

    template <typename Handler>
    auto disableOrganization(const OrganizationParams& c, std::time_t doomDate,
                             Handler handler) const {
        io_result::detail::init_async_result<Handler, OnExecute> init(handler);
        asyncDisableOrganization(c, doomDate, init.handler);
        return init.result.get();
    }

    template <typename Handler>
    auto deactivateOrganization(const OrganizationParams& c, std::time_t doomDate,
                             Handler handler) const {
        io_result::detail::init_async_result<Handler, OnExecute> init(handler);
        asyncDeactivateOrganization(c, doomDate, init.handler);
        return init.result.get();
    }

    template <typename Handler>
    auto getOrganizationUids(const OrganizationParams& c, Handler handler) const {
        io_result::detail::init_async_result<Handler, OnOrganizationUids> init(handler);
        asyncOrganizationUids(c, init.handler);
        return init.result.get();
    }

    template <typename Handler>
    auto updateOrganizationUids(const OrganizationParams& c, Uids uids, Handler handler) const {
        io_result::detail::init_async_result<Handler, OnUpdate> init(handler);
        asyncUpdateOrganizationUids(c, std::move(uids), init.handler);
        return init.result.get();
    }

    template <typename Handler>
    auto messagesInSearch(const CommonParams& c, const MessagesAccessParams& ma,
                          Handler handler) const {
        io_result::detail::init_async_result<Handler, OnMessagesInSearch> init(handler);
        asyncMessagesInSearch(c, ma, init.handler);
        return init.result.get();
    }

    template <typename Handler>
    auto messagesInSearch(const CommonParams& c, const SingleMessageAccessParams& ma,
                          Handler handler) const {
        io_result::detail::init_async_result<Handler, OnMessagesInSearch> init(handler);
        asyncMessagesInSearch(c, MessagesAccessParams {
            .uid=ma.uid,
            .mids={ma.mid},
            .searchId=ma.searchId
        }, init.handler);
        return init.result.get();
    }

    template <typename Handler>
    auto reserveSearchId(const CommonParams& common, Handler handler) const {
        io_result::detail::init_async_result<Handler, OnSearchId> init(handler);
        asyncReserveSearchId(common, init.handler);
        return init.result.get();
    }

    template <typename Handler>
    auto createSearch(const CommonParams& common, SearchId searchId, std::time_t dateFrom,
                      std::time_t dateTo, QueryJson query, SearchName name, Uids uids,
                      Handler handler) const {
        io_result::detail::init_async_result<Handler, OnExecute> init(handler);
        asyncCreateSearch(common, searchId, dateFrom, dateTo, std::move(query), std::move(name), std::move(uids), init.handler);
        return init.result.get();
    }

    template <typename Handler>
    auto fillSearch(const CommonParams& common, SearchId searchId,
                    SearchResults results, SearchState targetState, Handler handler) const {
        io_result::detail::init_async_result<Handler, OnExecute> init(handler);
        asyncFillSearch(common, searchId, std::move(results), targetState, init.handler);
        return init.result.get();
    }

    template <typename Handler>
    auto cacheSearchResults(const CommonParams& common, SearchId searchId,
                            SearchResults results, Handler handler) const {
        io_result::detail::init_async_result<Handler, OnExecute> init(handler);
        asyncCacheSearchResults(common, searchId, std::move(results), init.handler);
        return init.result.get();
    }

    template <typename Handler>
    auto searchById(const CommonParams& common, SearchId searchId,
                    Handler handler) const {
        io_result::detail::init_async_result<Handler, OnSearch> init(handler);
        asyncSearchById(common, searchId, init.handler);
        return init.result.get();
    }

    template <typename Handler>
    auto lastSearchId(const CommonParams& common, Handler handler) const {
        io_result::detail::init_async_result<Handler, OnOptionalSearchId> init(handler);
        asyncLastSearchId(common, init.handler);
        return init.result.get();
    }

    template <typename Handler>
    auto archiveSearch(const CommonParams& common, SearchId searchId, Handler handler) const {
        io_result::detail::init_async_result<Handler, OnExecute> init(handler);
        asyncArchiveSearch(common, searchId, init.handler);
        return init.result.get();
    }

    template <typename Handler>
    auto renameSearch(const CommonParams& common, SearchId searchId, const std::string& name,
                      Handler handler) const {
        io_result::detail::init_async_result<Handler, OnUpdate> init(handler);
        asyncRenameSearch(common, searchId, name, init.handler);
        return init.result.get();
    }

    template <typename Handler>
    auto searchList(const CommonParams& common, const PageParams& page,
                    Handler handler) const {
        io_result::detail::init_async_result<Handler, OnSearches> init(handler);
        asyncSearchList(common, StrongPageParams(page, std::numeric_limits<decltype(SearchId().t)>::max()), init.handler);
        return init.result.get();
    }

    template <typename Handler>
    auto failSearch(const CommonParams& common, SearchId searchId,
                    const Notice& notice, Handler handler) const {
        io_result::detail::init_async_result<Handler, OnExecute> init(handler);
        asyncFailSearch(common, searchId, notice, init.handler);
        return init.result.get();
    }

    template <typename Handler>
    auto continueSearch(const CommonParams& common, SearchId searchId,
                        Handler handler) const {
        io_result::detail::init_async_result<Handler, OnExecute> init(handler);
        asyncContinueSearch(common, searchId, init.handler);
        return init.result.get();
    }

    template <typename Handler>
    auto searchByOneUser(const CommonParams& common, const Search& search, Uid uid,
                         const PageParams& page, Handler handler) const {
        io_result::detail::init_async_result<Handler, OnSearchResults> init(handler);
        asyncSearchByOneUser(common, search.searchId, uid, StrongPageParams(page, 0), init.handler);
        return init.result.get();
    }

    template <typename Handler>
    auto searchByAllUsers(const CommonParams& common, const Search& search, PageParams page,
                          Handler handler) const {
        io_result::detail::init_async_result<Handler, OnSearchResults> init(handler);
        asyncSearchByAllUsers(common, search.searchId, StrongPageParams(page, 0), init.handler);
        return init.result.get();
    }

    template <typename Handler>
    auto minReceivedDateBySearchAndUid(const CommonParams& common, SearchId searchId, Uid uid, Handler handler) const {
        io_result::detail::init_async_result<Handler, OnOptionalReceivedDate> init(handler);
        asyncMinReceivedDateBySearchAndUid(common, searchId, uid, init.handler);
        return init.result.get();
    }

    template <typename Handler>
    auto logAction(const CommonParams& common, ActionHistoryType type, std::string infoJson, Handler handler) const {
        io_result::detail::init_async_result<Handler, OnExecute> init(handler);
        asyncLogAction(common, type, std::move(infoJson), init.handler);
        return init.result.get();
    }

    template <typename Handler>
    auto getActionHistory(const CommonParams& common, const PageParams& page, Handler handler) const {
        io_result::detail::init_async_result<Handler, OnActionHistoryItems> init(handler);
        asyncGetActionHistory(common, StrongPageParams(page, std::numeric_limits<decltype(Id().t)>::max()), init.handler);
        return init.result.get();
    }

    template <typename Handler>
    auto registerTaskId(const OrganizationParams& c, ymod_queuedb::TaskId taskId, Handler handler) const {
        io_result::detail::init_async_result<Handler, OnTaskCreated> init(handler);
        asyncRegisterTaskId(c, taskId, init.handler);
        return init.result.get();
    }

    template <typename Handler>
    auto removeTaskId(const OrganizationParams& c, ymod_queuedb::TaskId taskId, Handler handler) const {
        io_result::detail::init_async_result<Handler, OnExecute> init(handler);
        asyncRemoveTaskId(c, taskId, init.handler);
        return init.result.get();
    }

protected:
    virtual void asyncOrganizationUids(const OrganizationParams&, OnOrganizationUids) const = 0;
    virtual void asyncUpdateOrganizationUids(const OrganizationParams&, Uids, OnUpdate) const = 0;
    virtual void asyncListActiveOrganizationIds(const RequestId&, OnOrganizationIds) const = 0;

    virtual void asyncActivateOrganization(const OrganizationParams&, OnExecute) const = 0;
    virtual void asyncGetOrganization(const OrganizationParams&, OnOrganization) const = 0;
    virtual void asyncDisableOrganization(const OrganizationParams&, std::time_t, OnExecute) const = 0;
    virtual void asyncDeactivateOrganization(const OrganizationParams&, std::time_t, OnExecute) const = 0;
    virtual void asyncReserveSearchId(const CommonParams&, OnSearchId) const = 0;
    virtual void asyncLastSearchId(const CommonParams&, OnOptionalSearchId) const = 0;
    virtual void asyncCreateSearch(const CommonParams&, SearchId, std::time_t, std::time_t, QueryJson, SearchName,
                                   Uids, OnExecute) const = 0;
    virtual void asyncFillSearch(const CommonParams&, SearchId, SearchResults, SearchState, OnExecute) const = 0;
    virtual void asyncCacheSearchResults(const CommonParams&, SearchId, SearchResults, OnExecute) const = 0;
    virtual void asyncSearchById(const CommonParams&, SearchId, OnSearch) const = 0;
    virtual void asyncSearchList(const CommonParams&, const StrongPageParams&, OnSearches) const = 0;
    virtual void asyncFailSearch(const CommonParams&, SearchId, const Notice&, OnExecute) const = 0;
    virtual void asyncContinueSearch(const CommonParams&, SearchId, OnExecute) const = 0;
    virtual void asyncSearchByOneUser(const CommonParams&, SearchId, Uid, const StrongPageParams&, OnSearchResults) const = 0;
    virtual void asyncSearchByAllUsers(const CommonParams&, SearchId, const StrongPageParams&, OnSearchResults) const = 0;
    virtual void asyncMinReceivedDateBySearchAndUid(const CommonParams& c, SearchId, Uid, OnOptionalReceivedDate cb) const = 0;
    virtual void asyncLogAction(const CommonParams&, ActionHistoryType, std::string, OnExecute) const = 0;
    virtual void asyncGetActionHistory(const CommonParams&, const StrongPageParams&, OnActionHistoryItems) const = 0;
    virtual void asyncRenameSearch(const CommonParams&, SearchId, const std::string&, OnUpdate) const = 0;
    virtual void asyncArchiveSearch(const CommonParams&, SearchId, OnExecute) const = 0;
    virtual void asyncMessagesInSearch(const CommonParams&, const MessagesAccessParams&, OnMessagesInSearch) const = 0;

    virtual void asyncRegisterTaskId(const OrganizationParams&, ymod_queuedb::TaskId, OnTaskCreated) const = 0;
    virtual void asyncRemoveTaskId(const OrganizationParams&, ymod_queuedb::TaskId, OnExecute) const = 0;
};

using RepositoryPtr = std::shared_ptr<Repository>;

}
