#include <mail/spaniel/ymod_db/include/repository.h>
#include <mail/spaniel/ymod_db/include/internal/types.h>
#include <mail/spaniel/core/include/log.h>
#include <mail/spaniel/core/include/types_error.h>
#include <mail/spaniel/core/include/types_reflection.h>
#include <mail/webmail/commondb/include/with_executers.h>
#include <mail/webmail/commondb/include/settings_reflection.h>
#include <mail/webmail/commondb/include/logger.h>
#include <yamail/data/deserialization/ptree.h>
#include <yamail/data/deserialization/json_reader.h>
#include <yamail/data/serialization/json_writer.h>
#include <yplatform/module_registration.h>
#include <pgg/database/pool_factory.h>
#include <pgg/factory.h>
#include <fstream>


BOOST_FUSION_DEFINE_STRUCT((spaniel), Settings,
    (std::string, reactor)
    (std::string, logger)
)


namespace spaniel {
namespace log {
using namespace ::commondb::attr;
}

namespace ydr = yamail::data::reflection;
namespace ydd = yamail::data::deserialization;
using namespace commondb;
using namespace std::string_literals;

namespace {
auto fromReflection(reflection::WithResult& data) {
    return data.result;
}

auto fromReflection(reflection::SearchId& data) {
    return spaniel::SearchId(data.search_id);
}

auto fromReflection(reflection::ReceivedDate& data) {
    return data.received_date;
}

auto fromReflection(reflection::Search& search) {
    spaniel::Search s;
    s.orgId = search.org_id;
    s.searchId = search.search_id;
    s.adminUid = search.admin_uid;
    s.requestedUids = Uids(search.requested_uids.begin(), search.requested_uids.end());
    if (search.found_uids) {
        s.foundUids = Uids(search.found_uids->begin(), search.found_uids->end());
    }
    s.query = ydd::fromJson<Query>(search.query);
    s.state = ydr::from_string<SearchState>(search.state);
    s.dateFrom = search.date_from;
    s.dateTo = search.date_to;
    s.created = search.created;
    s.updated = search.updated;
    s.name = search.name;
    if (search.min_search_date) {
        s.minSearchDate = *search.min_search_date;
    }

    return s;
}

auto fromReflection(reflection::ActionHistoryRow& action) {
    spaniel::ActionHistoryItem res{
        .actionId = Id(action.action_id),
        .adminUid = Uid(action.admin_uid),
        .type = ydd::from_string<spaniel::ActionHistoryType>(action.type),
        .info = ydd::fromJson<spaniel::ActionHistoryInfo>(action.info),
        .date = action.date
    };
    const auto notEmpty = [](auto& opt) {
        if (!opt.has_value()) {
            throw std::invalid_argument("action_history.info contains incorrect value");
        }
    };
    switch(res.type) {
        case spaniel::ActionHistoryType::create_search:
            notEmpty(res.info.createSearch);
            break;
        case spaniel::ActionHistoryType::send_share:
            notEmpty(res.info.sendShare);
            break;
    }
    return res;
}

auto fromReflection(reflection::Organization& data) {
    return spaniel::Organization {
        .state = ydr::from_string<OrganizationState>(data.state),
        .doomDate = data.doom_date,
        .created = data.created
    };
}

auto fromReflection(reflection::OptionalOrganization& data) {
    io_result::Optional<spaniel::Organization> s;

    if (data.organization) {
        s = fromReflection(*data.organization);
    }
    return s;
}

auto fromReflection(reflection::OptionalSearch& data) {
    io_result::Optional<spaniel::Search> s;

    if (data.search) {
        s = fromReflection(*data.search);
    }
    return s;
}

auto fromReflection(reflection::WithSearch& data) {
    return fromReflection(data.search);
}

auto fromReflection(reflection::WithUid& data) {
    return Uid(data.uid);
}

auto fromReflection(reflection::WithOrgId& data) {
    return OrgId(data.org_id);
}

auto fromReflection(reflection::WithBool& data) {
    return data.result;
}

auto fromReflection(reflection::SearchResult& data) {
    return spaniel::SearchResult{
        .uid = Uid(data.uid),
        .id = Id(data.id),
        .received_date = data.received_date
    };
}

auto fromReflection(reflection::OptionalSearchId& data) {
    io_result::Optional<spaniel::SearchId> s;

    if (data.search_id) {
        s = spaniel::SearchId(*data.search_id);
    }

    return s;
}

decltype(auto) wrapDatabaseStringErrorAsPermanentFail(OnExecute handler) {
    using OptStr = boost::optional<std::string>;
    io_result::Hook<OptStr> hook = [h = std::move(handler)] (mail_errors::error_code ec, const auto& result) {
        if (ec) {
            h(ec);
        } else if (result) {
            h(make_error(DbError::permanentFail, *result));
        } else {
            h(mail_errors::error_code());
        }
    };
    return hook;
}

}

struct Database: public WithExecuters, public Repository, public yplatform::module {

    virtual ~Database() = default;

    void init(const yplatform::ptree& cfg) {
        common = ydd::fromPtree<Settings>(cfg);
        pg = readPgSettings(cfg.get_child("pg"));

        initExecuters(common.reactor, pg, query::QueriesRegister(), query::ParametersRegister());

        LOGDOG_(getModuleLogger(common.logger), notice,
            log::message="spaniel::RepositoryImpl module initialized",
            log::settings=yamail::data::serialization::JsonWriter(common).result(),
            log::pg_settings=yamail::data::serialization::JsonWriter(pg).result(),
            log::password_length=passwordSize()
        );
    }

    using WithExecuters::anyExecutor;
    using WithExecuters::readWriteExecutor;

    template<class Params>
    auto master(const Params& c) const {
        auto ctxLogger = getContextLogger(c);
        auto logger = makePggLogger(ctxLogger);
        auto profiler = commondb::pggProfiler(std::to_string(c.orgId.t), c.requestId, "spanieldb");
        return readWriteExecutor(c.requestId, logger, profiler);
    }

    template<class Params>
    auto anything(const Params& c) const {
        auto ctxLogger = getContextLogger(c);
        auto logger = makePggLogger(ctxLogger);
        auto profiler = commondb::pggProfiler(std::to_string(c.orgId.t), c.requestId, "spanieldb");
        return anyExecutor(c.requestId, logger, profiler);
    }

    template<>
    auto anything(const RequestId& requestId) const {
        auto ctxLogger = getContextLogger(std::nullopt, std::nullopt, requestId);
        auto logger = makePggLogger(ctxLogger);
        auto profiler = commondb::pggProfiler("-", requestId, "spanieldb");
        return anyExecutor(requestId, logger, profiler);
    }

    void asyncOrganizationUids(const OrganizationParams& c, OnOrganizationUids cb) const override {
        anything(c)->request(
            pgg::withQuery<query::GetOrganizationUids>(
                query::OrgId(c.orgId)
            ),
            std::move(cb),
            [] (reflection::WithUid data) { return fromReflection(data); }
        );
    }

    void asyncUpdateOrganizationUids(const OrganizationParams& c, Uids uids, OnUpdate cb) const override {
        master(c)->update(
            pgg::withQuery<query::UpdateOrganizationUids>(
                query::OrgId(c.orgId),
                query::UidVec(reflection::UidVec(uids.begin(), uids.end()))
            ),
            std::move(cb)
        );
    }

    void asyncListActiveOrganizationIds(const RequestId& reqId, OnOrganizationIds cb) const override {
        anything(reqId)->request(
            pgg::withQuery<query::ListActiveOrganizationIds>(),
            std::move(cb),
            [] (reflection::WithOrgId data) { return fromReflection(data); }
        );
    }

    void asyncDisableOrganization(const OrganizationParams& c, std::time_t doomDate, OnExecute cb) const override {
        master(c)->execute(
            pgg::withQuery<query::DisableOrganization>(
                query::OrgId(c.orgId),
                query::DoomDate(doomDate)
            ),
            std::move(cb)
        );
    }

    void asyncActivateOrganization(const OrganizationParams& c, OnExecute cb) const override {
        master(c)->execute(
            pgg::withQuery<query::ActivateOrganization>(
                query::OrgId(c.orgId)
            ),
            std::move(cb)
        );
    }

    void asyncDeactivateOrganization(const OrganizationParams& c, std::time_t doomDate, OnExecute cb) const override {
        master(c)->execute(
            pgg::withQuery<query::DeactivateOrganization>(
                query::OrgId(c.orgId),
                query::DoomDate(doomDate)
            ),
            std::move(cb)
        );
    }

    void asyncGetOrganization(const OrganizationParams& c, OnOrganization cb) const override {
        anything(c)->request(
            pgg::withQuery<query::GetOrganization>(
                query::OrgId(c.orgId)
            ),
            std::move(cb),
            [] (reflection::OptionalOrganization data) { return fromReflection(data); }
        );
    }

    void asyncReserveSearchId(const CommonParams& c, OnSearchId cb) const override {
        master(c)->request(
            pgg::withQuery<query::ReserveSearchId>(
                query::OrgId(c.orgId)
            ),
            std::move(cb),
            [] (reflection::SearchId data) { return fromReflection(data); }
        );
    }

    void asyncLastSearchId(const CommonParams& c, OnOptionalSearchId cb) const override {
        anything(c)->request(
            pgg::withQuery<query::LastSearchId>(
                query::OrgId(c.orgId)
            ),
            std::move(cb),
            [] (reflection::OptionalSearchId data) { return fromReflection(data); }
        );
    }

    void asyncCreateSearch(const CommonParams& c, SearchId searchId, std::time_t dateFrom,
                           std::time_t dateTo, QueryJson query, SearchName name, Uids uids, OnExecute cb) const override {
        master(c)->request(
            pgg::withQuery<query::CreateSearch>(
                query::SearchId(searchId),
                query::QueryJson(std::move(query)),
                query::Name(std::move(name)),
                query::UidVec(reflection::UidVec(uids.begin(), uids.end())),
                query::DateFrom(dateFrom),
                query::DateTo(dateTo),
                query::RequestInfo(reflection::RequestInfo(c))
            ),
            wrapDatabaseStringErrorAsPermanentFail(std::move(cb)),
            [] (reflection::WithResult data) { return fromReflection(data); }
        );
    }

    void asyncCacheSearchResults(const CommonParams& c, SearchId searchId, SearchResults results, OnExecute cb) const override {
        reflection::SearchResultVec vec;
        vec.reserve(results.size());
        std::transform(
            results.begin(), results.end(), std::back_inserter(vec), [] (auto&& res) {
                return reflection::SearchResult {.uid=res.uid, .id=res.id, .received_date=res.received_date};
            }
        );

        master(c)->request(
            pgg::withQuery<query::CacheSearchResults>(
                query::SearchId(searchId),
                query::SearchResultVec(std::move(vec)),
                query::RequestInfo(reflection::RequestInfo(c))
            ),
            wrapDatabaseStringErrorAsPermanentFail(std::move(cb)),
            [] (reflection::WithResult data) { return fromReflection(data); }
        );
    }

    void asyncFillSearch(const CommonParams& c, SearchId searchId,
                         SearchResults results, SearchState targetState, OnExecute cb) const override {
        reflection::SearchResultVec vec;
        vec.reserve(results.size());
        std::transform(
            results.begin(), results.end(), std::back_inserter(vec), [] (auto&& res) {
                return reflection::SearchResult {.uid=res.uid, .id=res.id, .received_date=res.received_date};
            }
        );

        master(c)->request(
            pgg::withQuery<query::FillSearch>(
                query::SearchId(searchId),
                query::SearchResultVec(std::move(vec)),
                query::SearchState(std::string(ydr::to_string(targetState))),
                query::RequestInfo(reflection::RequestInfo(c))
            ),
            wrapDatabaseStringErrorAsPermanentFail(std::move(cb)),
            [] (reflection::WithResult data) { return fromReflection(data); }
        );
    }

    void asyncContinueSearch(const CommonParams& c, SearchId searchId,
                             OnExecute cb) const override {
        master(c)->request(
            pgg::withQuery<query::ContinueSearch>(
                query::SearchId(searchId),
                query::RequestInfo(reflection::RequestInfo(c))
            ),
            wrapDatabaseStringErrorAsPermanentFail(std::move(cb)),
            [] (reflection::WithResult data) { return fromReflection(data); }
        );
    }

    void asyncSearchById(const CommonParams& c, SearchId searchId, OnSearch cb) const override {
        anything(c)->request(
            pgg::withQuery<query::SearchById>(
                query::OrgId(c.orgId),
                query::SearchId(searchId)
            ),
            std::move(cb),
            [] (reflection::OptionalSearch data) { return fromReflection(data); }
        );
    }

    void asyncSearchList(const CommonParams& c, const StrongPageParams& page, OnSearches cb) const override {
        anything(c)->request(
            pgg::withQuery<query::SearchList>(
                query::OrgId(c.orgId),
                query::RowCount(page.count),
                query::RowFrom(page.first)
            ),
            std::move(cb),
            [] (reflection::WithSearch data) { return fromReflection(data); }
        );
    }

    void asyncFailSearch(const CommonParams& c, SearchId searchId, const Notice& notice, OnExecute cb) const override {
        master(c)->execute(
            pgg::withQuery<query::FailSearch>(
                query::SearchId(searchId),
                query::Notice(notice),
                query::RequestInfo(reflection::RequestInfo(c))
            ),
            std::move(cb)
        );
    }

    void asyncSearchByOneUser(const CommonParams& c, SearchId searchId, Uid uid, const StrongPageParams& page, OnSearchResults cb) const override {
        anything(c)->request(
            pgg::withQuery<query::SearchByOneUser>(
                query::OrgId(c.orgId),
                query::SearchId(searchId),
                query::Uid(uid),
                query::RowFrom(page.first),
                query::RowCount(page.count)
            ),
            std::move(cb),
            [] (reflection::SearchResult data) { return fromReflection(data); }
        );
    }

    void asyncSearchByAllUsers(const CommonParams& c, SearchId searchId, const StrongPageParams& page, OnSearchResults cb) const override {
        anything(c)->request(
            pgg::withQuery<query::SearchByAllUsers>(
                query::OrgId(c.orgId),
                query::SearchId(searchId),
                query::RowFrom(page.first),
                query::RowCount(page.count)
            ),
            std::move(cb),
            [] (reflection::SearchResult data) { return fromReflection(data); }
        );
    }

    void asyncMinReceivedDateBySearchAndUid(const CommonParams& c, SearchId searchId, Uid uid, OnOptionalReceivedDate cb) const override {
        anything(c)->request(
            pgg::withQuery<query::MinReceivedDateBySearchAndUid>(
                query::OrgId(c.orgId),
                query::Uid(uid),
                query::SearchId(searchId)
            ),
            std::move(cb),
            [] (reflection::ReceivedDate data) { return fromReflection(data); }
        );
    }

    void asyncLogAction(const CommonParams& c, ActionHistoryType type, std::string infoJson, OnExecute cb) const override {
        master(c)->execute(
            pgg::withQuery<query::LogAction>(
                query::ActionType(std::string(ydr::to_string(type))),
                query::RequestInfo(reflection::RequestInfo(c)),
                query::InfoJson(std::move(infoJson))
            ),
            std::move(cb)
        );
    }

    void asyncGetActionHistory(const CommonParams& c, const StrongPageParams& page, OnActionHistoryItems cb) const override {
        anything(c)->request(
            pgg::withQuery<query::GetActionHistory>(
                query::OrgId(c.orgId),
                query::RowCount(page.count),
                query::RowFrom(page.first)
            ),
            std::move(cb),
            [] (reflection::ActionHistoryRow data) { return fromReflection(data); }
        );
    }

    void asyncRenameSearch(const CommonParams& c, SearchId searchId, const std::string& name, OnUpdate cb) const override {
        master(c)->update(
            pgg::withQuery<query::SearchRename>(
                query::OrgId(c.orgId),
                query::SearchId(searchId),
                query::Name(name)
            ),
            std::move(cb)
        );
    }

    void asyncArchiveSearch(const CommonParams& c, SearchId searchId, OnExecute cb) const override {
        master(c)->request(
            pgg::withQuery<query::ArchiveSearch>(
                query::SearchId(searchId),
                query::RequestInfo(reflection::RequestInfo(c))
            ),
            wrapDatabaseStringErrorAsPermanentFail(std::move(cb)),
            [] (reflection::WithResult data) { return fromReflection(data); }
        );
    }

    void asyncMessagesInSearch(const CommonParams& c, const MessagesAccessParams& ma, OnMessagesInSearch cb) const override {
        reflection::IdVec vec;
        vec.resize(ma.mids.size());
        std::transform(
            ma.mids.begin(), ma.mids.end(), vec.begin(),
            [] (const std::string& mid) { return makeId(mid); }
        );

        anything(c)->request(
            pgg::withQuery<query::MessagesInSearch>(
                query::OrgId(c.orgId),
                query::SearchId(ma.searchId),
                query::Uid(ma.uid),
                query::IdVec(std::move(vec))
            ),
            std::move(cb),
            [] (reflection::WithBool data) { return fromReflection(data); }
        );
    }

    void asyncRegisterTaskId(const OrganizationParams& c, ymod_queuedb::TaskId taskId, OnTaskCreated cb) const override {
        master(c)->request(
            pgg::withQuery<query::RegisterTaskId>(
                query::OrgId(c.orgId),
                query::TaskId(taskId)
            ),
            std::move(cb),
            [] (reflection::WithBool data) { return fromReflection(data); }
        );
    }

    void asyncRemoveTaskId(const OrganizationParams& c, ymod_queuedb::TaskId taskId, OnExecute cb) const override {
        master(c)->execute(
            pgg::withQuery<query::RemoveTaskId>(
                query::OrgId(c.orgId),
                query::TaskId(taskId)
            ),
            std::move(cb)
        );
    }

    Settings common;
    PgSettings pg;
};

}

DEFINE_SERVICE_OBJECT(spaniel::Database)
