#include <mail/alabay/ymod_logbroker/include/consumer/event_buffer.h>
#include <mail/alabay/ymod_logbroker/include/consumer/event_buffer_proxy.h>
#include <mail/alabay/ymod_logbroker/include/producer/buffer.h>
#include <mail/alabay/ymod_logbroker/include/parse_helpers.h>
#include <mail/sqbar/service/include/log.h>
#include <mail/webmail/http_api_helpers/include/find_dependency.h>
#include <mail/webmail/tskv/parser.h>
#include <yplatform/module_registration.h>

namespace sqbar {

using namespace ymod_logbroker::helpers;

bool requiredTarget(const ParsedEvent& event) {
    return isEventContainsValueFromRange(event, "target", "message", "folders", "mailbox");
}

bool requiredOperation(const ParsedEvent& event) {
    return isEventContainsValueFromRange(event, "operation",
        "receive", "send", "delete", "reset_fresh", "create", "rename", "copy",
        "move", "deletee", "abuse", "undo_sending"
    );
}

std::optional<std::string> joinEvent(std::string_view uid, const ParsedEvent& event) {
    std::map<std::string, std::string> sqbarEvent;

    sqbarEvent["uid"] = std::string(uid);
    sqbarEvent["ip"] = get(event, "ip");
    sqbarEvent["port"] = get(event, "port", "0000");

    sqbarEvent["target"] = get(event, "target");
    sqbarEvent["operation"] = get(event, "operation");

    sqbarEvent["emailFrom"] = get(event, "emailFrom");
    sqbarEvent["emailTo"] = get(event, "emailTo");
    sqbarEvent["emailCc"] = get(event, "emailCc");
    sqbarEvent["emailBcc"] = get(event, "emailBcc");

    sqbarEvent["mid"] = get(event, "mid");
    sqbarEvent["mids"] = get(event, "mids");
    sqbarEvent["subject"] = get(event, "subject");
    sqbarEvent["userAgent"] = get(event, "userAgent");

    return webmail::tskv::join(sqbarEvent);
}

struct FilteredUserJournalBuffer: public ymod_logbroker::EventBuffer, public yplatform::module {
    FilteredUserJournalBuffer(ymod_logbroker::OutputTaskBuffer& buffer)
        : ymod_logbroker::EventBuffer()
        , messages(std::make_shared<ymod_logbroker::Task::Messages>())
        , producer(buffer)
    { }

    virtual ~FilteredUserJournalBuffer() = default;

    void asyncAddEvents(std::string_view, ParsedEvent event, yplatform::task_context_ptr, ymod_logbroker::OnProbablyFlush cb) override {
        if (auto u = getOpt(event, "uid"); u) {
            std::string_view strUid = makeUid(*u);
            ParsedUid uid = std::stoull(strUid.data());

            const bool need = productionUser(uid) && requiredTarget(event) && requiredOperation(event);
            if (need) {
                if (auto joined = joinEvent(strUid, event); joined) {
                    messages->emplace_back(std::move(*joined));
                }
            }
        }
        cb(mail_errors::error_code(), false);
    }

    void asyncFlush(yplatform::task_context_ptr ctx, ymod_logbroker::OnFlush cb) override {
        producer.pushBack(ymod_logbroker::Task(ctx, messages, std::move(cb)));
        messages = std::make_shared<ymod_logbroker::Task::Messages>();
    }

    std::size_t size() const override {
        return messages->size();
    }

    ymod_logbroker::Task::MessagesPtr messages;
    ymod_logbroker::OutputTaskBuffer& producer;
};

struct FilteredUserJournalFactory: public ymod_logbroker::EventBufferFactory, public yplatform::module {
    virtual ~FilteredUserJournalFactory() = default;

    void init(const yplatform::ptree& cfg) {
        using namespace http_api;

        consumerName_ = cfg.get<TString>("logbroker.consumer_name");
        producer_ = findDependency<ymod_logbroker::OutputTaskBuffer>(cfg, "dependencies.producer");
        yplatform::read_ptree(topics_, cfg.get_child("logbroker"), "topics");
        chunkSize_ = cfg.get<std::size_t>("logbroker.events_chunk_size");

        LOGDOG_(getModuleLogger(cfg.get<std::string>("dependencies.logger")), notice, log::message="sqbar::FilteredUserJournalFactory loaded");
    }

    ymod_logbroker::EventBufferPtr create() const override {
        return std::make_shared<ymod_logbroker::WithSize>(
            std::make_shared<FilteredUserJournalBuffer>(*producer_), chunkSize_
        );
    }

    const std::vector<TString>& topics() const override {
        return topics_;
    }

    const TString& consumerName() const override {
        return consumerName_;
    }

    std::size_t chunkSize_ = 0;
    std::vector<TString> topics_;
    TString consumerName_;
    ymod_logbroker::OutputTaskBufferPtr producer_;
};

}

DEFINE_SERVICE_OBJECT(sqbar::FilteredUserJournalFactory)
