#!/usr/bin/python3
# coding: utf-8

from http.server import HTTPServer, BaseHTTPRequestHandler
from collections import namedtuple
import os
import sys
import socket
import requests
import logging
import yaml
import psycopg2

Config = namedtuple('Config', ['port', 'peers', 'databases'])
DbConfig = namedtuple('DbConfig', ['conninfo', 'max_connections'])
component = os.environ["QLOUD_COMPONENT"]

def load_databases(db_configs):
    return [DbConfig(c["conninfo"], c["max_connections"]) for c in db_configs]

def load_config():
    config_tree = yaml.load(open(sys.argv[1], "r"))
    config_tree = config_tree['config'][component]
    return Config(
        int(config_tree["port"]),
        config_tree["peers"],
        load_databases(config_tree["databases"]))

config = load_config()

def setup_logging():
    LOGFORMAT = '%(asctime)s %(message)s'
    loghandler = logging.FileHandler(
        '/var/log/dirty-orange/%s.log' % (component,))
    logging.basicConfig(format=LOGFORMAT,
                        level=logging.INFO,
                        handlers=(loghandler,))
    logging.getLogger("requests").setLevel(logging.WARNING)
    logging.getLogger("urllib3").setLevel(logging.WARNING)

def main():
    setup_logging()
    server = HTTPServerV6(('::', config.port), Handler)
    logging.info('config loaded, listening on %s', config.port)
    server.serve_forever()
    logging.info('stopped')

class HTTPServerV6(HTTPServer):
    address_family = socket.AF_INET6

class Handler(BaseHTTPRequestHandler):
    def log_message(self, format, *args):
        logging.info('%s | %s', self.client_address[0], format%args)

    def do_GET(self):
        method_to_call = self.path[1:]
        if hasattr(self, method_to_call):
            getattr(self, method_to_call)()
        else:
            self.respond(404, 'NotFound')

    def hey(self):
        try:
            self.hey_peers()
            self.hey_dbs()
            self.respond(200, 'OK')
        except Exception as e:
            logging.exception('hey failed: %s' % (str(e)))
            self.respond(500, str(e))

    def hey_peers(self):
        for peer in config.peers:
            resp = requests.get('http://%s/hey' % (peer,))
            resp.raise_for_status()

    def hey_dbs(self):
        for db in config.databases:
            with psycopg2.connect(db.conninfo) as connection:
                connection.autocommit = True
                with connection.cursor() as cursor:
                    cursor.execute("SELECT 'hey'")

    def respond(self, code, text):
        self.send_response(code, text)
        self.end_headers()

if __name__ == '__main__':
    main()
