#!/bin/bash

[[ "$TRACE" ]] && set -x
set -eo pipefail

function usage() {
    >&2 echo "Usage: $0 [--check] [--fail-fast] [--filter=regexp]"
    >&2 echo "    --check         - do not apply alerts, only check them. Enables --fail-fast"
    >&2 echo "    --fail-fast     - fail on first error, do not process other files"
    >&2 echo "    --filter=regex  - apply only files mathing regexp"
    >&2 echo "    --help, -h, -?  - show this message and exit"
}


CHECK="false"
FAIL_FAST="false"
FILTER=".*"

for arg in "$@"; do
    case ${arg} in
        --check)
            CHECK="true"
            FAIL_FAST="true"
            shift
            ;;
        --fail-fast)
            FAIL_FAST="true"
            shift
            ;;
        --filter=*)
            FILTER="${arg#*=}"
            shift
            ;;
        --help|-h|-\?)
            usage
            exit 0
            ;;
        *)
            >&2 echo "Unknown option: '${arg}'"
            usage
            exit 1
        ;;
    esac
done


[[ -z "${ANSIBLE_JUGGLER_ENV_DIR}" ]] && ANSIBLE_JUGGLER_ENV_DIR=/tmp/ansible-juggler-venv

if [[ ! -d "${ANSIBLE_JUGGLER_ENV_DIR}" || -z $(ls -1A "${ANSIBLE_JUGGLER_ENV_DIR}") ]]; then
    >&2 echo "Initialize virtual environment in '${ANSIBLE_JUGGLER_ENV_DIR}'"
    python2 -m virtualenv ${ANSIBLE_JUGGLER_ENV_DIR}
fi

source "${ANSIBLE_JUGGLER_ENV_DIR}/bin/activate"

if [[ ! -x "${ANSIBLE_JUGGLER_ENV_DIR}/bin/ansible-playbook" ]]; then
    >&2 echo "ansible-playbook not found. Trying to install it."
    pip install six==1.12.0 setuptools==44.0.0
    pip install --index-url https://pypi.yandex-team.ru/simple/ ansible-juggler2
fi

declare -a FAILED_LIST
ALERT_FILES=$(ls -1 *.yaml | grep -e "${FILTER}")
if [[ $? -eq 0 ]]; then
    CHECK_OPT=$([[ "${CHECK}" == "true" ]] && echo "--check --diff" || echo "")

    for file in ${ALERT_FILES[@]}; do
        >&2 echo "Processing file '${file}'..."
        ANSIBLE_DEPRECATION_WARNINGS=False \
        ansible-playbook -vvv ${CHECK_OPT} "${file}"
        if [[ $? -ne 0 ]]; then
            FAILED_LIST+=("${file}")
            >&2 echo "Found error(s) in '${file}'"
            if [[ "${FAIL_FAST}" == "true" ]]; then
                exit 1
            fi
        fi
    done
fi

if [[ ${#FAILED_LIST[@]} -ne 0 ]]; then
    >&2 echo "There are error(s) in file(s):"
    for f in "${FAILED_LIST[@]}"; do >&2 echo "${f}"; done
    exit 1
fi

>&2 echo "Finished succesfully"
